#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from functools import partial
import copy
import os

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                     QStackedLayout, QLabel, QGroupBox, QToolButton, QPushButton, QComboBox,
                     QRadioButton, QDialog, QDialogButtonBox, QCheckBox,
                     QLineEdit, QSize)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog
from calibre.gui2.widgets2 import Dialog
from calibre.ebooks.metadata.book.formatter import SafeFormat
from calibre.utils.date import parse_date, UNDEFINED_DATE

from calibre_plugins.action_chains.actions.base import ChainAction
from calibre_plugins.action_chains.common_utils import get_icon
from calibre_plugins.action_chains.templates import check_template, TEMPLATE_ERROR
from calibre_plugins.action_chains.templates.dialogs import TemplateBox
from calibre_plugins.action_chains.gui.single_field_widgets import (get_metadata_widget,
                                                                    get_predefined_widget_default)
from calibre_plugins.action_chains.database import (get_marked_id_map, bulk_modify_marked,
                                                    set_marked_ids, set_marked_for_book, add_cover_from_file)

try:
    load_translations()
except NameError:
    prints("ActionChains::actions/single_field.py - exception when loading translations")

def get_possible_cols(db):
    standard = [
        'title',
        'authors',
        'tags',
        'series',
        'publisher',
        'pubdate',
        'rating',
        'languages',
        'last_modified',
        'timestamp',
        'comments',
        'author_sort',
        'sort',
        'marked',
        'identifiers',
        'cover',
        'formats'
    ]                
    custom = sorted([ k for k,v in db.field_metadata.custom_field_metadata().items() if v['datatype'] not in [None,'composite'] ])
    return standard + custom

def is_enum(db, col_name, val):
    col_metadata = db.field_metadata.all_metadata()[col_name]
    col_type = col_metadata['datatype']
    if not col_type == 'enumeration':
        raise ValueError
    vals = col_metadata['display']['enum_values'] + ['']
    if not val in vals:
        raise ValueError
    else:
        return val

def is_bool(val):
    if str(val).lower() in ['yes','y','true','1']:
        return True
    elif str(val).lower() in ['no','n','false','0']:
        return False
    elif str(val).strip() in ['','undefined','n/a']:
        return ''
    else:
        raise ValueError

def to_int(val, strict_int=True):
    '''
    calibre templates return numbers in text float form (e.g. 5.0)
    which fails when python try to convert using int('5.0')
    this function converts to float first
    '''
    try:
        val = float(val)
    except:
        raise ValueError
    if strict_int:
        if not val.is_integer():
            raise ValueError
    return int(val)

class PromptWidgetDialog(Dialog):
    def __init__(self, parent, widget_cls, plugin_action, col_name, book_ids):
        self.plugin_action = plugin_action
        self.db = self.plugin_action.gui.current_db
        self.book_ids = book_ids
        self.widget = widget_cls(self.plugin_action, col_name, self)
        self.widget.initialize(book_ids)
        Dialog.__init__(self, 'Edit Field', f'action-chains-prompt-edit-field-{col_name}', parent)

    def setup_ui(self):
        l = QVBoxLayout()
        self.setLayout(l)
        l.addWidget(self.widget)
        l.addWidget(self.bb)

    def accept(self):
        self.widget.commit(self.book_ids)
        Dialog.accept(self)

class PredefinedWidget(QWidget):
    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.db = plugin_action.gui.current_db
        self.stack = {}
        self.possible_cols = get_possible_cols(self.db)
        self.count = len(self.possible_cols)
        self.l = l = QVBoxLayout()
        self.setLayout(l)
        self.sl = QStackedLayout()
        l.addLayout(self.sl, 0)
        l.addStretch(1)
        for col_name in self.possible_cols:
            widget_cls = get_metadata_widget(self.db, col_name, mode='predefined')
            widget = self.stack[col_name] = widget_cls(self.plugin_action, col_name, self)
            #if hasattr(widget, 'all_values'):
                #widget.main_widget.update_items_cache(widget.all_values)
            ## hide apply changes checkbox. Later it is checked automatically at execution time.
            #if hasattr(widget, 'a_c_checkbox'):
                #widget.a_c_checkbox.hide()
            self.sl.addWidget(widget)
        # Add and empty widget at the end and switch to it.
        self.empty_widget = QWidget()
        self.sl.addWidget(self.empty_widget)
        self.sl.setCurrentIndex(self.count-1)

    @property
    def current_col(self):
        idx = self.sl.currentIndex()
        if idx < self.count:
            return self.possible_cols[idx]
        else:
            return ''

    def set_current_col(self, col_name):
        if col_name in ['', None]:
            self.sl.setCurrentIndex(self.count-1)
        else:
            if self.current_col != col_name:
                self.sl.setCurrentIndex(self.possible_cols.index(col_name))

            widget = self.stack[col_name]
            if hasattr(widget, 'all_values'):
                widget.main_widget.update_items_cache(widget.all_values)
            # hide apply changes checkbox. Later it is checked automatically at execution time.
            if hasattr(widget, 'a_c_checkbox'):
                widget.a_c_checkbox.hide()

    def read(self):
        idx = self.sl.currentIndex()
        if idx < self.count:
            col_name = self.possible_cols[idx]
            val = self.stack[self.current_col].current_val
            return val

    def write(self, val):
        idx = self.sl.currentIndex()
        if idx < self.count:
            # Wrap in try block in case we are trying to restore value to absent column
            # In which case datatype of value might match current_col datatype
            try:
                self.stack[self.current_col].setter(val)
            except:
                import traceback
                prints(traceback.format_exc())

    def adjust_size(self):
        '''
        adjust size to fit the pre-defined widget if present
        '''
        hints = {'comments': (500, 600), 'series': (600, 400), 'datetime': (400, 400)}
        idx = self.sl.currentIndex()
        if idx < self.count:
            col_name = self.possible_cols[idx]
            col_type = self.db.field_metadata.all_metadata()[col_name]['datatype']
            size = hints.get(col_type, (400, 400))
            w, h = size
            self.setMinimumSize(w,h)
            self.updateGeometry()

def template_to_field(plugin_action, chain, book_ids, template, col_name):

    db = plugin_action.gui.current_db
    cmeta = db.field_metadata.all_metadata()[col_name]
    col_type = cmeta['datatype']
    ui_to_list = cmeta['is_multiple'].get('ui_to_list')

    convert_map = {
        'int': to_int,
        'float': float,
        'rating': float,
        'datetime': parse_date,
        'enumeration': partial(is_enum, db, col_name),
        'bool': is_bool
    }

    id_col_map = {}

    for book_id in book_ids:
        template_output = chain.evaluate_template(template, book_id)
        if col_name == 'marked':
            orig_value = get_marked_id_map(plugin_action).get(book_id, '')
        else:
            orig_value = db.new_api.field_for(col_name, book_id)
        
        if template_output == '':
            # unset column value, only if it has a value, skip for cover and formats
            val = None
            if col_name in ['cover', 'formats']:
                continue
            if col_name == 'timestamp':
                val = UNDEFINED_DATE
            if orig_value not in  ['', None, (), UNDEFINED_DATE]:
                id_col_map[book_id] = val
        else:

            # validate datatype based on target column
            if col_type in convert_map.keys() and (col_name not in ['cover', 'formats']):
                convert = convert_map[col_type]
                try:
                    template_output = convert(template_output)
                except Exception as e:
                    if DEBUG:
                        prints(f'Action Chains: failed converting {template_output} to {col_type}\n{e}')
                    continue
            if col_name == 'formats':
                try:
                    fmt = os.path.splitext(template_output)[-1].lower().replace('.', '').upper()
                except:
                    fmt = ''

                if os.access(template_output, os.R_OK):   
                    db.new_api.add_format(book_id, fmt, template_output)
                else:
                    if DEBUG:
                        prints(f'Cannot add format from file: {template_output}')
                continue
            elif col_name == 'cover':
                add_cover_from_file(db, book_id, template_output)
                continue
            # make sure no to update the colum value if the template returns a value that is identical to the original
            if col_name == 'identifiers':
                if not template_output:
                    new_value = {}
                else:
                    try:
                        new_value = [x.strip().split(':') for x in template_output.split(ui_to_list) if x.strip() != '']
                        new_value = {k: v for k, v in new_value}
                    except:
                        new_value = {}
                if orig_value == new_value:
                    continue
            if ui_to_list:
                if not template_output:
                    new_value = ()
                else:
                    new_value = [x.strip() for x in template_output.split(ui_to_list)]
                if set(orig_value) == set(new_value):
                    continue
            else:
                if orig_value is None:
                    orig_value = ''
                if orig_value == template_output:
                    continue
            #
            if col_name == 'marked':
                set_marked_for_book(plugin_action, book_id, template_output)
            else:
                id_col_map[book_id] = template_output
    if col_name == 'marked':
        # we already set the marks per book in the loop
        pass
    # with last_modified db.new_api.set_field will reset the value to current time, use db.update_last_modified() instead
    elif col_name == 'last_modified':
        for book_id, timestamp in id_col_map.items():
            db.update_last_modified([book_id], now=timestamp)
    else:
        db.new_api.set_field(col_name, id_col_map)

class SingleEditConfigWidget(QWidget):
    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.gui = plugin_action.gui
        self.db = self.gui.current_db
        self.possible_cols = self.get_possible_cols()
        self.template = ''
        self.template_settings = {}
        self._init_controls()

    def _init_controls(self):
        self.blockSignals(True)
        l = self.l = QVBoxLayout()
        self.setLayout(l)

        col_box = QGroupBox(_('Choose column:'))
        col_box_layout = QVBoxLayout()
        col_box.setLayout(col_box_layout)
        self.col_combobox = QComboBox()
        self.col_combobox.addItems(self.possible_cols)
        self.col_combobox.setCurrentIndex(-1)
        self.col_combobox.currentIndexChanged.connect(self._on_col_change)
        col_box_layout.addWidget(self.col_combobox)
        l.addWidget(col_box)
                
        self.opts_box = QGroupBox(_('Options:'))
        self.opts_box_l = opts_box_l = QVBoxLayout()
        self.opts_box.setLayout(self.opts_box_l)
        l.addWidget(self.opts_box)
        
        self.prompt_opt = QRadioButton(_('Ask at runtime'))
        self.prompt_opt.setChecked(True)
        opts_box_l.addWidget(self.prompt_opt)

        self.clear_opt = QRadioButton(_('Clear value'))
        opts_box_l.addWidget(self.clear_opt)

        template_layout = QHBoxLayout()
        self.template_opt = QRadioButton(_('Template'))
        self.template_opt.setEnabled(False)
        self.template_button = QPushButton(_('Add template'))
        self.template_button.clicked.connect(self._on_template_opt_chosen)
        template_layout.addWidget(self.template_opt, 1)
        template_layout.addWidget(self.template_button)
        opts_box_l.addLayout(template_layout)
        
        self.predefined_opt = QRadioButton(_('Predefined value'))
        self.predefined_opt.toggled.connect(self._on_predefined_opt_chosen)
        self.predefined_opt.setEnabled(False)
        self.predefined_widget = PredefinedWidget(self.plugin_action)
        opts_box_l.addWidget(self.predefined_opt)
        opts_box_l.addWidget(self.predefined_widget)

        self._on_col_change()

        l.addStretch(1)
        self.setMinimumSize(300,200)
        self.blockSignals(False)

    def get_possible_cols(self):
        return get_possible_cols(self.db)

    def _on_template_opt_chosen(self):
        d = TemplateBox(self, self.plugin_action, template_text=self.template)
        if d.exec_() == d.Accepted:
            self.template = d.template
            self.template_opt.setEnabled(True)
            self.template_opt.setChecked(True)
            self.template_button.setText(_('Edit template'))

    def _on_col_change(self):
        col_name = self.col_combobox.currentText()
        self.predefined_widget.set_current_col(col_name)
        self.updateGeometry()
        if col_name:
            self.predefined_widget.setEnabled(self.predefined_opt.isChecked())
            self.predefined_opt.setEnabled(True)
            self.template_button.setEnabled(True)
            
            if col_name in ['cover','formats']:
                #self.predefined_opt.setChecked(True)
                self.prompt_opt.setEnabled(False)
                self.prompt_opt.setChecked(False)
            else:
                self.prompt_opt.setEnabled(True)

        else:
            self.template_button.setEnabled(False)
            self.predefined_widget.setEnabled(False)

    def _on_predefined_opt_chosen(self):
        self.predefined_widget.setEnabled(self.predefined_opt.isChecked())

    def load_settings(self, settings):
        if settings:
            self.col_combobox.setCurrentText(settings['col_name'])
            if settings['value_type'] == 'prompt':
                self.prompt_opt.setChecked(True)
            elif settings['value_type'] == 'clear':
                self.clear_opt.setChecked(True)
            elif settings['value_type'] == 'template':
                self.template_opt.setEnabled(True)
                self.template_opt.setChecked(True)
                self.template_button.setText(_('Edit template'))
                self.template = settings['template']
                #self.template_settings = settings['template_settings']
            elif settings['value_type'] == 'predefined':
                self.predefined_opt.setChecked(True)
                self.predefined_widget.write(settings['value'])

    def save_settings(self):
        settings = {}
        settings['col_name'] = self.col_combobox.currentText()
        if self.prompt_opt.isChecked():
            settings['value_type'] = 'prompt'
        if self.clear_opt.isChecked():
            settings['value_type'] = 'clear'
        elif self.template_opt.isChecked():
            settings['value_type'] = 'template'
            settings['template'] = self.template
            #settings['template_settings'] = self.template_settings
        elif self.predefined_opt.isChecked():
            settings['value_type'] = 'predefined'
            settings['value'] = self.predefined_widget.read()
        return settings

class SingleFieldAction(ChainAction):

    name = 'Single Field Edit'
    _is_builtin = True
    support_scopes = True

    def run(self, gui, settings, chain):
        book_ids = chain.scope().get_book_ids()
        
        db = gui.current_db

        col_name = settings['col_name']
        value_type = settings['value_type']
        cmeta = db.field_metadata.all_metadata()[col_name]
        col_type = cmeta['datatype']
        
        if len(book_ids) == 0:
            return

        # if value is specified by template
        if value_type == 'template':
            template = settings['template']
            template_to_field(self.plugin_action, chain, book_ids, template, col_name)

        # if value(s) to be determined at runtime
        elif value_type == 'prompt':
            QApplication.setOverrideCursor(Qt.ArrowCursor)
            try:
                if len(book_ids) == 1:
                    book_ids = book_ids[0]
                    widget_cls = get_metadata_widget(db, col_name, mode='single')
                elif len(book_ids) > 1:
                    widget_cls = get_metadata_widget(db, col_name, mode='bulk')
                
                d = PromptWidgetDialog(gui, widget_cls, self.plugin_action, col_name, book_ids)
                if d.exec_() == d.Accepted:
                    #dialog.commit(book_ids)
                    pass
            finally:
                QApplication.restoreOverrideCursor()

        # if values are predefined
        elif value_type == 'predefined':
            widget_cls = get_metadata_widget(db, col_name, mode='predefined')
            value = settings['value']
            w = widget_cls(self.plugin_action, col_name, gui)
            w.resize(QSize(0, 0))
            w.setter(value)
            w.a_c_checkbox.setChecked(True)
            w.commit(book_ids)
            del w

        # the user set the value to clear the field
        elif value_type == 'clear':
            if col_name == 'marked':
                marked_text = get_marked_id_map(self.plugin_action)
                for book_id in book_ids:
                    try:
                        del marked_text[book_id]
                    except:
                        pass
                set_marked_ids(self.plugin_action, marked_text)
            elif col_name == 'cover':
                for book_id in book_ids:
                    db.remove_cover(book_id, notify=False, commit=False)
            elif col_name == 'formats':
                for book_id in book_ids:
                    fmts_string = db.formats(book_id, index_is_id=True)
                    if fmts_string:
                        available_fmts = [ fmt.strip().upper() for fmt in fmts_string.split(',') ]
                    else:
                        available_fmts = []
                    for fmt in available_fmts:
                        db.remove_format(book_id, fmt, index_is_id=True, notify=False)
            else:
                val = None
                if col_type == 'bool':
                    if not db.new_api.pref('bools_are_tristate'):
                        val = False
                if col_name == 'timestamp':
                    val = UNDEFINED_DATE
                id_map = {book_id: val for book_id in book_ids}
                db.new_api.set_field(col_name, id_map)

    def validate(self, settings):
        gui = self.plugin_action.gui
        db = gui.current_db
        if not settings:
            return (_('Settings Error'), _('You must configure this action before running it'))
        col_name = settings['col_name']
        if not col_name:
            return (_('Column Error'), _('Column name cannot be empty'))
        if not col_name in get_possible_cols(db):
            return (_('Column Error'), _(f'Cannot find a column with name "{col_name}" in current library'))
        if not settings.get('value_type'):
            return _('Option Error'), _('No option selected among (prompt - clear - template - predefined)')
        if settings['value_type'] == 'template':
            if not settings['template']:
                return (_('Settings Error'), _('No template value specified'))
            is_template_valid = check_template(settings['template'], self.plugin_action, print_error=False)
            if is_template_valid is not True:
                return is_template_valid
        cmeta = db.field_metadata.all_metadata()[col_name]
        if settings['value_type'] == 'predefined':
            value = settings['value']
            value_if_widget_not_altered = get_predefined_widget_default(self.plugin_action, col_name)
            # dont allow empty default values except for rating, bool, enum types
            if ( cmeta['datatype'] not in ['rating', 'enumeration', 'bool', 'int', 'float'] ) and ( col_name not in ['cover', 'formats'] ):
                if value == value_if_widget_not_altered:
                    return (_('Undefined value'), _('Predefined options chosen without specifying a value'))
        return True

    def config_widget(self):
        return SingleEditConfigWidget


