#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                      QGroupBox, QModelIndex, QRadioButton, QCheckBox, QComboBox,
                      QLabel, QLineEdit, QTextBrowser)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog, config

from calibre_plugins.action_chains.actions.base import ChainAction
from calibre_plugins.action_chains.database import (is_search_valid, is_saved_search_valid,
                                           get_valid_saved_searches, get_books_modified_since)

try:
    load_translations()
except NameError:
    prints("ActionChains::actions/selection_modifier.py - exception when loading translations")

class SelectionModifierConfigWidget(QWidget):
    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.gui = plugin_action.gui
        self.db = self.gui.current_db
        self._init_controls()

    def _init_controls(self):

        l = self.l = QVBoxLayout()
        self.setLayout(l)

        vl_gb = QGroupBox(_('Virtual library'))
        vl_gb_l = QVBoxLayout()
        vl_gb.setLayout(vl_gb_l)
        l.addWidget(vl_gb)
        vl_clear_chk = self.vl_clear_chk = QCheckBox(_('Clear all virtual libraries and restrictions'))
        vl_gb_l.addWidget(vl_clear_chk)
        
        search_gb = self.search_gb = QGroupBox(_('Search'))
        search_gb_l = QVBoxLayout()
        search_gb.setLayout(search_gb_l)
        l.addWidget(search_gb)
        search_keep_opt = self.search_keep_opt = QRadioButton(_('Dont modify search'))
        search_gb_l.addWidget(search_keep_opt)
        search_keep_opt.setChecked(True)
        search_clear_opt = self.search_clear_opt = QRadioButton(_('Clear current search'))
        search_gb_l.addWidget(search_clear_opt)
        
        hl3 = QHBoxLayout()
        new_search_opt = self.new_search_opt = QRadioButton(_('New search'))
        searchbox = self.searchbox = QLineEdit(self)
        searchbox.textChanged.connect(self._on_searchbox_text_change)
        hl3.addWidget(new_search_opt)
        hl3.addWidget(searchbox)
        search_gb_l.addLayout(hl3)

        hl4 = QHBoxLayout()
        saved_search_opt = self.saved_search_opt = QRadioButton(_('Choose saved search'))
        search_combo = self.search_combo = QComboBox()
        search_combo.addItems(get_valid_saved_searches(self.db))
        hl4.addWidget(saved_search_opt)
        hl4.addWidget(search_combo)
        search_gb_l.addLayout(hl4)

        help_text = _('''<ul><li>Depending on the actions of this chain, you might need to choose 
                       the options to clear current search or restrictions, otherwise,
                       calibre may not be able to select all the books you specify
                       in the selection options below. Only those within the scope
                       of the current search and restriction can be selected.</li>
                       <li>If you choose a search above, it only changes the current
                       library view to the results of that search, you need to tick
                       the option "select all books in the current library view".</li>
                       <li>Search highlighting is temporarily disabled to allow selection
                       of books in current search only.</li></ul>''')
        tb = QTextBrowser()
        tb.setHtml(help_text)
        l.addWidget(tb, 0)

        selection_gb = QGroupBox(_('Selection options'))
        selection_gb_l = QVBoxLayout()
        selection_gb.setLayout(selection_gb_l)
        l.addWidget(selection_gb)

        chain_modified_opt = self.chain_modified_opt = QRadioButton(_('Select books modified since the start of this chain'))
        last_modified_opt = self.last_modified_opt = QRadioButton(_('Select books modified by last action'))
        current_view_opt = self.current_view_opt = QRadioButton(_('Select all books in current library view'))
        clear_selection_opt = self.clear_selection_opt = QRadioButton(_('Clear selections'))
        select_first_opt = self.select_first_opt = QRadioButton(_('Select first book in current view'))
        
        for opt in [chain_modified_opt, last_modified_opt,
                    current_view_opt, clear_selection_opt, select_first_opt]:
            selection_gb_l.addWidget(opt)
        
        chain_modified_opt.setChecked(True)
        
        l.addStretch(1)

        self.setMinimumSize(350,500)

    def _on_searchbox_text_change(self):
        text = self.searchbox.text()
        if text:
            ok = is_search_valid(self.db, text)
            ss = 'QLineEdit { border: 2px solid %s; border-radius: 3px }' % (
            '#50c878' if ok else '#FF2400')
            self.searchbox.setStyleSheet(ss)
        else:
            self.searchbox.setStyleSheet('')

    def load_settings(self, settings):
        if settings:
            if settings['selection'] == 'last_action_modified':
                self.last_modified_opt.setChecked(True)
            elif settings['selection'] == 'chain_modified':
                self.chain_modified_opt.setChecked(True)
            elif settings['selection'] == 'current_view':
                self.current_view_opt.setChecked(True)
            elif settings['selection'] == 'clear':
                self.clear_selection_opt.setChecked(True)
            elif settings['selection'] == 'select_first':
                self.select_first_opt.setChecked(True)
            if settings['restriction']:
                if settings['restriction']['opt'] == 'clear':
                    self.vl_clear_chk.setChecked(True)
            if settings['search']:
                if settings['search']['opt'] == 'clear':
                    self.search_clear_opt.setChecked(True)
                elif settings['search']['opt'] == 'new':
                    self.new_search_opt.setChecked(True)
                    self.searchbox.setText(settings['search']['search_text'])
                    #self._on_searchbox_text_change()
                elif settings['search']['opt'] == 'saved':
                    self.saved_search_opt.setChecked(True)
                    self.search_combo.setCurrentText(settings['search']['saved_search'])
            else:
                self.search_keep_opt.setChecked(True)

    def save_settings(self):
        settings = {}
        if self.last_modified_opt.isChecked():
            settings['selection'] = 'last_action_modified'
        elif self.chain_modified_opt.isChecked():
            settings['selection'] = 'chain_modified'
        elif self.current_view_opt.isChecked():
            settings['selection'] = 'current_view'
        elif self.clear_selection_opt.isChecked():
            settings['selection'] = 'clear'
        elif self.select_first_opt.isChecked():
            settings['selection'] = 'select_first'
        settings['restriction'] = {}
        if self.vl_clear_chk.isChecked():
            settings['restriction']['opt'] = 'clear'
        settings['search'] = {}
        if not self.search_keep_opt.isChecked():
            if self.search_clear_opt.isChecked():
                settings['search']['opt'] = 'clear'
            elif self.new_search_opt.isChecked():
                settings['search']['opt'] = 'new'
                search_text = self.searchbox.text()
                settings['search']['search_text'] = search_text
            elif self.saved_search_opt.isChecked():
                settings['search']['opt'] = 'saved'
                saved_search_name = self.search_combo.currentText()
                settings['search']['saved_search'] = saved_search_name
        return settings

class SelectionModifierAction(ChainAction):

    name = 'Selection Modifier'
    _is_builtin = True

    def _apply_highlight_if_different(self, gui, new_state):
        if config['highlight_search_matches'] != new_state:
            config['highlight_search_matches'] = new_state
            gui.set_highlight_only_button_icon()

    def run(self, gui, settings, chain):

        # Before proceeding, we need search highlighting to be turned off to allow selection
        # for all books in current search view, first we save the current hightlight state,
        # and then we turn off the highlighting if turned on. At the end we restore the original
        # user setting for search highlight
        orig_highlighting_state = config['highlight_search_matches']
        self._apply_highlight_if_different(gui, False)
        
        try:        
            db = gui.current_db

            if settings.get('restriction', {}).get('opt') == 'clear':
                gui.search.clear()
                gui.apply_virtual_library()
                gui.clear_additional_restriction()

            if settings['search']:
                if settings['search']['opt'] == 'clear':
                    gui.search.clear()
                elif settings['search']['opt'] == 'new':
                    search_text = settings['search']['search_text']
                    gui.search.set_search_string(search_text)
                    gui.search.do_search()
                elif settings['search']['opt'] == 'saved':
                    saved_search = settings['search']['saved_search']
                    gui.search.set_search_string(f'search:"={saved_search}"')
                    gui.search.do_search()
            
            row_count = gui.library_view.model().rowCount(QModelIndex())
            current_view_rows = list(range(row_count))
            current_view_book_ids = list(map(gui.library_view.model().id, current_view_rows))

            if settings['selection'] == 'current_view':
                book_ids = current_view_book_ids
            elif settings['selection'] == 'clear':
                book_ids = []
            elif settings['selection'] == 'select_first':
                book_ids = []
                if row_count > 0:
                    book_ids = [gui.library_view.model().id(0)]
            else:
                if settings['selection'] == 'last_action_modified':
                    last_step = chain.last_step
                    last_step_start = chain.chain_links[chain.last_step]['start_time']
                    book_ids = get_books_modified_since(db, last_step_start)
                elif settings['selection'] == 'chain_modified':
                    chain_start = chain.start_time
                    book_ids = get_books_modified_since(db, chain_start)

                # filter out books no in current view as they cannot be selected
                #book_ids = set(book_ids).intersection(current_view_book_ids)
                book_ids = set(book_ids).intersection(db.all_ids())

            gui.library_view.select_rows(book_ids)
        finally:
            self._apply_highlight_if_different(gui, orig_highlighting_state)

    def run_as_job(self, settings):
        return False

    def validate(self, settings):
        gui = self.plugin_action.gui
        db = gui.current_db
        if not settings:
            return (_('Settings Error'), _('You must configure this action before running it'))
        if settings['search']:
            if settings['search']['opt'] == 'new':
                search_text = settings['search']['search_text']
                if not is_search_valid(db, search_text):
                    return (_('Invalid search'), _(f'Search "{search_text}" is not valid'))
            elif settings['search']['opt'] == 'saved':
                saved_search = settings['search']['saved_search']
                if not is_saved_search_valid(db, saved_search):
                    return (_('Invalid search'), _(f'Saved search "{saved_search}" is not valid'))
        return True

    def config_widget(self):
        return SelectionModifierConfigWidget


