#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import subprocess
import os
import sys
import copy
import shlex
import re

from qt.core import (
    QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
    QLabel, QGroupBox, QToolButton, QPushButton, QComboBox,
    QRadioButton, QDialog, QDialogButtonBox, QCheckBox, QSizePolicy,
    QLineEdit)

from calibre import prints
from calibre.constants import iswindows, isosx, islinux, DEBUG
from calibre.gui2 import error_dialog, open_local_file, choose_files
from calibre.ebooks.metadata.book.formatter import SafeFormat

from calibre_plugins.action_chains.actions.base import ChainAction
from calibre_plugins.action_chains.common_utils import (
    DragDropComboBox, get_icon, get_file_path)
from calibre_plugins.action_chains.templates import (
    check_template, TEMPLATE_ERROR)
from calibre_plugins.action_chains.templates.dialogs import (
    TemplateBox, TemplateLineEditor)


try:
    load_translations()
except NameError:
    prints("ActionChains::actions/open_with.py - exception when loading translations")

DEFAULT_SEP = ' '

class OpenError(Exception):
    def __init__(self, message, errors):
        self.message = message
        self.errors = errors

class OpenWithWidget(QWidget):
    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.gui = plugin_action.gui
        self.db = self.gui.current_db
        self.template = ''
        self._init_controls()

    def _init_controls(self):

        l = self.l = QVBoxLayout()
        self.setLayout(l)

        self.default_chk = QCheckBox(_('Open with default app'))
        self.default_chk.stateChanged.connect(self._on_use_default_checked)
        l.addWidget(self.default_chk)

        self.binary_box = QGroupBox(_('&Choose binary:'))
        l.addWidget(self.binary_box)
        binary_layout = QVBoxLayout()
        self.binary_box.setLayout(binary_layout)
        self.binary_combo = DragDropComboBox(self, drop_mode='file')
        binary_layout.addWidget(self.binary_combo)
        hl1 = QHBoxLayout()
        binary_layout.addLayout(hl1)
        hl1.addWidget(self.binary_combo, 1)
        self.choose_binary_button = QToolButton(self)
        self.choose_binary_button.setToolTip(_('Choose binary'))
        self.choose_binary_button.setIcon(get_icon('document_open.png'))
        self.choose_binary_button.clicked.connect(self._choose_file)
        hl1.addWidget(self.choose_binary_button)

        args_box = QGroupBox(_('Args (optional): '), self)
        binary_layout.addWidget(args_box)
        args_l = QVBoxLayout()
        args_box.setLayout(args_l)
        self.args_edit = TemplateLineEditor(self, self.plugin_action)
        self.args_edit.setToolTip(
            _('Inpufile (format, folder or cover) will be appneded after the args.\n'
              'If you want to place it elsewhere, use the place holder {inputfile}\n'
              'in the place where you want it to be.\n'
              'Args should separated with spaces.\n'
              'Templates are supported (Only GPM and Python mode)'))
        args_l.addWidget(self.args_edit)
        # args not supported on osx
        if isosx:
            args_box.hide()

        self.shell_chk = QCheckBox(_('Execute command through shell'))
        self.shell_chk.setChecked(False)
        l.addWidget(self.shell_chk)

        path_gb = QGroupBox(_('Path: '), self)
        l.addWidget(path_gb)
        path_l = QVBoxLayout()
        path_gb.setLayout(path_l)
        self.fmts_layout = QHBoxLayout()
        self.fmts_opt = QRadioButton(_('Book format'))
        self.fmts_combo = QComboBox()
        self.fmts_layout.addWidget(self.fmts_opt, 1)
        self.fmts_layout.addWidget(self.fmts_combo)
        path_l.addLayout(self.fmts_layout)
        self.cover_opt = QRadioButton(_('Book cover'))
        path_l.addWidget(self.cover_opt)
        self.folder_opt = QRadioButton(_('Containing folder'))
        path_l.addWidget(self.folder_opt)
        template_layout = QHBoxLayout()
        self.template_opt = QRadioButton(_('Template'))
        self.template_opt.setEnabled(False)
        self.template_button = QPushButton(_('Add template'))
        self.template_button.clicked.connect(self._on_template_button_clicked)
        template_layout.addWidget(self.template_opt, 1)
        template_layout.addWidget(self.template_button)
        path_l.addLayout(template_layout)
        
        multiple_box = self.multiple_box = QGroupBox()
        l.addWidget(multiple_box)
        multiple_l = QVBoxLayout()
        multiple_box.setLayout(multiple_l)
        multiple_chk = self.multiple_chk = QCheckBox(
            _('Allow multiple selections (Experimental)'))
        multiple_chk.setChecked(False)
        multiple_l.addWidget(multiple_chk)
        sep_box = QGroupBox(_('Separator'))
        sep_l = QHBoxLayout()
        sep_box.setLayout(sep_l)
        multiple_l.addWidget(sep_box)
        multiple_l.addWidget(sep_box)
        default_opt = self.default_opt = QRadioButton(_('Default'))
        default_opt.setChecked(True)
        sep_l.addWidget(default_opt)
        sep_l.addStretch(1)
        other_opt = self.other_opt = QRadioButton(_('Other'))
        sep_l.addWidget(other_opt)
        other_edit = self.other_edit = QLineEdit()
        other_edit.setFixedWidth(30)
        sep_l.addWidget(other_edit)
        
        l.addStretch(1)        

        self.all_formats = self.gui.library_view.model().db.all_formats()
        if not self.all_formats:
            self.fmts_layout.setEnabled(False)
        else:
            self.fmts_combo.addItems(self.all_formats)
            self.fmts_combo.setCurrentIndex(-1)

        self.setMinimumSize(400,550)
        
    def _on_use_default_checked(self):
        is_default_checked = self.default_chk.isChecked()
        self.binary_box.setEnabled(not is_default_checked)
        self.multiple_box.setEnabled(not is_default_checked)
        if is_default_checked:
            self.multiple_chk.setChecked(False)

    def _on_template_button_clicked(self):
        d = TemplateBox(self, self.plugin_action, template_text=self.template)
        if d.exec_() == d.Accepted:
            self.template = d.template
            self.template_opt.setEnabled(True)
            self.template_opt.setChecked(True)
            self.template_button.setText(_('Edit template'))   

    def _choose_file(self):
        files = choose_files(
            None,
            _('Select binary dialog'),
            _('Select a binary'),
            all_files=True,
            select_only_single_file=True)
        if not files:
            return
        binary_path = files[0]
        if iswindows:
            binary_path = os.path.normpath(binary_path)

        self.block_events = True
        existing_index = self.binary_combo.findText(binary_path, Qt.MatchExactly)
        if existing_index >= 0:
            self.binary_combo.setCurrentIndex(existing_index)
        else:
            self.binary_combo.insertItem(0, binary_path)
            self.binary_combo.setCurrentIndex(0)
        self.block_events = False

    def load_settings(self, settings):
        if settings:
            self.default_chk.setChecked(settings['use_default_app'])
            if not settings['use_default_app']:
                self.binary_combo.setCurrentText(settings['path_to_binary'])
                self.args_edit.setText(settings['args'])
            if settings['path_opt'] == 'cover':
                self.cover_opt.setChecked(True)
            elif settings['path_opt'] == 'folder':
                self.folder_opt.setChecked(True)
            elif settings['path_opt'] == 'template':
                self.template_opt.setEnabled(True)
                self.template_opt.setChecked(True)
                self.template = settings['template']
                self.template_button.setText(_('Edit template'))
            elif settings['path_opt'] == 'format':
                fmt = settings['format']
                if fmt in self.all_formats:
                    self.fmts_layout.setEnabled(True)
                    self.fmts_opt.setChecked(True)
                    idx = self.fmts_combo.findText(fmt)
                    if idx == -1:
                        self.fmts_combo.addItem(fmt)
                    else:
                        self.fmts_combo.setCurrentIndex(idx)
                else:
                    self.fmts_layout.setEnabled(False)
            self.multiple_chk.setChecked(settings['allow_multiple'])
            if settings['allow_multiple']:
                sep = settings['path_sep']
                if sep == DEFAULT_SEP:
                    self.default_opt.setChecked(True)
                else:
                    self.other_opt.setChecked(True)
                    self.other_edit.setText(sep)
            self.shell_chk.setChecked(settings.get('shell', False))
            self._on_use_default_checked()

    def save_settings(self):
        settings = {}
        use_default = self.default_chk.isChecked()
        settings['use_default_app'] = use_default
        if not use_default:
            settings['path_to_binary'] = self.binary_combo.currentText().strip()
            settings['args'] = self.args_edit.text()
        if self.fmts_opt.isChecked():
            settings['path_opt'] = 'format'
            settings['format'] = self.fmts_combo.currentText()
        elif self.cover_opt.isChecked():
            settings['path_opt'] = 'cover'
        elif self.folder_opt.isChecked():
            settings['path_opt'] = 'folder'
        elif self.template_opt.isChecked():
            settings['path_opt'] = 'template'
            settings['template'] = self.template
        settings['allow_multiple'] = self.multiple_chk.isChecked()
        settings['path_sep'] = ' '
        if self.multiple_chk.isChecked():
            if self.other_opt.isChecked():
                settings['path_sep'] = self.other_edit.text()
            else:
                settings['path_sep'] = DEFAULT_SEP
        settings['shell'] = self.shell_chk.isChecked()
        return settings

class OpenWithAction(ChainAction):

    name = 'Open With'
    _is_builtin = True
    support_scopes = True

    def get_file_path(self, db, book_id, settings, chain, show_error=False):
        if settings['path_opt'] == 'cover':
            if not db.has_cover(book_id):
                raise OpenError(
                    _('Cannot open with'),
                    _('Book has no cover'))
            path_to_cover = os.path.join(
                db.library_path,
                db.path(book_id, index_is_id=True),
                'cover.jpg')
            return path_to_cover     

        elif settings['path_opt'] == 'template':
            template = settings['template']
            template_output = chain.evaluate_template(template, book_id)
            if template_output.startswith(TEMPLATE_ERROR):
                raise OpenError(
                    _('Cannot open with'),
                    _('Template returned error'))
            return template_output
        elif settings['path_opt'] == 'folder':
            path_to_folder = db.abspath(book_id, index_is_id=True)
            return path_to_folder
        else:
            book_format = settings['format']
            try:
                path_to_book = db.format_abspath(book_id, book_format, index_is_id=True)
                return path_to_book
            except:
                raise OpenError(
                    _('Cannot open with'),
                    _(f'Book has no format: {book_format}'))

    def run(self, gui, settings, chain):
        if settings.get('allow_multiple'):
            book_ids = chain.scope().get_book_ids()
        else:
            current_id = chain.scope().get_current_book_id()
            if current_id:
                book_ids = [current_id]
            else:
                book_ids = []
        
        db = gui.current_db

        path = []
        error_msg = ''
        for book_id in book_ids:
            try:
                path_to_book = self.get_file_path(db, book_id, settings, chain)
                if path_to_book:
                    path.append(path_to_book)
                else:
                    if DEBUG:
                        prints(f'Action Chains: value not found for book: {book_id}')
            except OpenError as e:
                #return error_dialog(gui, e.message, e.errors, show=True)
                book_title = db.title(book_id, index_is_id=True)
                error_msg += f'{book_title} ({book_id}): {e.errors}\n'
                
        if error_msg:
            if DEBUG:
                prints(f'Actions Chain: Open With: Errors with the following books:\n{error_msg}')

        if not path:
            return

        sep = settings.get('path_sep', DEFAULT_SEP)

        multiple = False

        # Multiple Selections Allowed
        if settings.get('allow_multiple'):
            multiple = True
        #

        if settings['use_default_app']:
            if DEBUG:
                prints(f'Action Chains: Open with default app: (path): {path[0]}')
            # multiple selections not allowed with default program option
            try:
                # file paths stored as a template can be relative, so process them first
                file_path = get_file_path(path[0])
                open_local_file(file_path)
            except TypeError as e:
                if DEBUG:
                    prints('Action Chains: Open With: failed with TypeError')
            
        else:
            external_app_path_template = settings['path_to_binary']
            external_app_path = chain.evaluate_template(external_app_path_template, book_ids[0])
            app_args = settings['args']
            if re.search(r'^(program|python):', app_args.strip()):
                app_args = chain.evaluate_template(app_args, book_ids[0])
            # Confirm we have defined an application for that format in tweaks
            if external_app_path is None:
                return error_dialog(
                    gui,
                    _('Cannot open with'),
                    _('Path not specified for this format in your configuration.'),
                    show=True)
            try:
                self.launch_app(
                    external_app_path,
                    app_args,
                    path,
                    multiple=multiple,
                    sep=sep,
                    shell=settings.get('shell', False))
            except TypeError as e:
                if DEBUG:
                    prints('Action Chains: Open With: failed with TypeError')

    def launch_app(
        self,
        external_app_path,
        app_args,
        path_to_files,
        wrap_args=True,
        multiple=False,
        sep='',
        shell=False
    ):
        external_app_path = os.path.expandvars(external_app_path)
    #         path_to_files = path_to_files.encode('utf-8')
        if DEBUG:
            prints(f'(command): {external_app_path}\n'
                   f'(file): {path_to_files}\n'
                   f'(args): {app_args}\n'
                   f'(multiple): {multiple}\n'
                   f'(sep): ({sep})')

        clean_env = dict(os.environ)

        kw = {'env': clean_env}

        if isosx:
            # For OSX we will not support optional command line
            # arguments currently
            shell = True
            if external_app_path.lower().endswith(".app"):
                external_app_path += 'open -a '

        elif iswindows:
            # Add to the recently opened files list to support
            # windows jump lists etc.
            if not multiple:
                from calibre.gui2 import add_to_recent_docs
                try:
                    add_to_recent_docs(path_to_files)
                except:
                    import traceback
                    traceback.print_exc()

            #DETACHED_PROCESS = 0x00000008
            #kw['creationflags'] = DETACHED_PROCESS
            del clean_env['PATH']

        else: #Linux
            clean_env['LD_LIBRARY_PATH'] = ''

        if shell is True:
            if (' ' in external_app_path):
                external_app_path = f'"{external_app_path}"'
            cmd = f'{external_app_path} {app_args}'
            inputfile = sep.join([f'"{x}"' for x in path_to_files])
            if '{inputfile}' in cmd:
                cmd = cmd.format(inputfile=inputfile)
            else:
                cmd += f' {inputfile}'
        else:
            cmd = [external_app_path]
            if app_args:
                cmd += shlex.split(app_args)
            l = []
            for f in path_to_files:
                if sep.strip():
                    l.append(sep)
                l.append(f)
                if l[0] == sep:
                    del l[0]
            try:
                index = cmd.index('{inputfile}')
                cmd = cmd[0:index] + l + cmd[index+1:]
            except ValueError:
                cmd += l

        print('About to run a command:', cmd)
        kw['shell'] = shell
        subprocess.Popen(cmd, **kw)

    def validate(self, settings):
        if not settings:
            return (
                _('Settings Error'),
                _('You must configure this action before running it'))
        if not settings['use_default_app']:
            if not settings['path_to_binary']:
                return (
                    _('No Binary'),
                    _('You must specify a path to valid binary'))
        if not settings.get('path_opt'):
            return (
                _('No Path'),
                _('You must choose a path option'))
        if settings['path_opt'] == 'format':
            if not settings['format']:
                return (
                    _('No format'),
                    _('You must choose a valid format value'))
        elif settings['path_opt'] == 'template':
            if not settings['template']:
                return (
                    _('Settings Error'),
                    _('No template value specified'))
            is_template_valid = check_template(
                settings['template'],
                self.plugin_action,
                print_error=False)
            if is_template_valid is not True:
                return is_template_valid

        # binary path & args can be templates, validate them
        external_app_path_template = settings.get('path_to_binary')
        if external_app_path_template:
            is_template_valid = check_template(
                external_app_path_template,
                self.plugin_action,
                print_error=False)
            if is_template_valid is not True:
                return is_template_valid

        app_args = settings.get('args', '')
        if re.search(r'^(program|python):', app_args.strip()):
            is_template_valid = check_template(
                app_args,
                self.plugin_action,
                print_error=False)
            if is_template_valid is not True:
                return is_template_valid
        #
        return True

    def config_widget(self):
        return OpenWithWidget
