#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from functools import partial
from collections import OrderedDict
import re, traceback
import statistics
import inspect

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                     QDialog, QDialogButtonBox, QSizePolicy, QSize, QLineEdit, QLabel,
                     QPlainTextEdit, QGroupBox, QRadioButton, QPushButton, QInputDialog)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog

from calibre_plugins.action_chains.actions.base import ChainAction
from calibre_plugins.action_chains.templates import (get_metadata_object, TEMPLATE_ERROR,
                                                    check_template, get_template_functions)
from calibre_plugins.action_chains.templates.dialogs import TemplateBox
from calibre_plugins.action_chains.templates import TemplateFunction
from calibre_plugins.action_chains.common_utils import ViewLog

try:
    load_translations()
except NameError:
    prints("ActionChains::actions/formulas.py - exception when loading translations")

#=================================
# Formulas
#=================================

class FormulasFunction(TemplateFunction):

    doc = _('No documentation provided')
    name = 'no name provided'
    category = 'Formulas'
    arg_count = 0
    aliases = []
    is_python = True

    def evaluate(self, formatter, kwargs, mi, locals, *args):
        return self.run(*args)

    @property
    def program_text(self):
        try:
            source = inspect.getsource(self.run)
        except:
            source = ''
        return source

    def run(self, *args):
        raise NotImplementedError

    def to_numerical_iterable(self, iterable, val='', sep=','):
        if val:
            try:
                val = int(val)
            except Exception as e:
                try:
                    val = float(val)
                except:
                    raise ValueError('val must be a numerical value')
        new = []
        for x in iterable.split(sep):
            x = x.strip()
            try:
                x = int(x)
            except:
                try:
                    x = float(x)
                except:
                    if val in ['']:
                        continue
                    x = val
            new.append(x)
        return new

    def check_column(self, col_name):
        gui = self.plugin_action.gui
        db = gui.current_db
        meta = db.field_metadata.all_metadata()
        cols = [col for col, cmeta in meta.items() if cmeta['kind'] == 'field']
        if col_name not in cols:
            raise ValueError(f'Unknown field: {col_name}')

class SelectionFormula(FormulasFunction):

    doc = _('Return a list of values from column col_name for all books selected in library view')    
    name = 'from_selection'
    arg_count = -1
    _is_builtin = True

    def run(self, col_name, sep=','):
        self.check_column(col_name)
        if self.context == 'template_dialog_mode':
            return sep.join(['1','2','3','4','5'])
        gui = self.plugin_action.gui
        db = gui.current_db
        rows = gui.current_view().selectionModel().selectedRows()
        book_ids = [ gui.library_view.model().db.id(row.row()) for row in rows ]
        if col_name == 'id':
            res = [ str(book_id) for book_id in book_ids ]
        else:
            res = [str(db.new_api.field_for(col_name, book_id)) for book_id in book_ids]
        return sep.join(res)

class SearchFormula(FormulasFunction):

    doc = _('Return a list of values from column col_name for all books in the search specified by the `query` arg')
    name = 'from_search'
    arg_count = -1
    _is_builtin = True

    def run(self, col_name, query, sep=','):
        self.check_column(col_name)
        if self.context == 'template_dialog_mode':
            return sep.join(['1','2','3','4','5'])
        gui = self.plugin_action.gui
        db = gui.current_db
        book_ids = db.search_getting_ids(query, None)
        if col_name == 'id':
            res = [ str(book_id) for book_id in book_ids ]
        else:
            res = [db.new_api.field_for(col_name, book_id) for book_id in book_ids]
            res = [str(x) for x in res]
        return sep.join(res)

class ReplaceNonNumericalFormula(FormulasFunction):

    doc = _('Replace non-numerial values in `iterable` with a numerical value specified by `value`.'
            'If `value` is not specified or is an empty string, non-numerical values will be deleted.')    
    name = 're_non_numerical'
    arg_count = -1
    _is_builtin = True

    def run(self, iterable, value='', sep=','):
        iterable = self.to_numerical_iterable(iterable, value, sep)
        return sep.join([str(x) for x in iterable])

class SumFormula(FormulasFunction):

    doc = _('Return the sum of the items in `iterable`.')
    name = 'sum'
    arg_count = 1
    _is_builtin = True

    def run(self, iterable, start=0):
        iterable = self.to_numerical_iterable(iterable)
        return str(sum(iterable, start))

class MaxFormula(FormulasFunction):

    doc = _('Return the maximum value of the items in `iterable`.')
    name = 'max'
    arg_count = 1
    _is_builtin = True

    def run(self, iterable):
        iterable = self.to_numerical_iterable(iterable)
        return str(max(iterable))

class MinFormula(FormulasFunction):

    doc = _('Return the minimum value of the items in `iterable`.')
    name = 'min'
    arg_count = 1
    _is_builtin = True

    def run(self, iterable):
        iterable = self.to_numerical_iterable(iterable)
        return str(min(iterable))

class MedianFormula(FormulasFunction):

    doc = _('Return the median value of the items in `iterable`.')
    name = 'median'
    arg_count = 1
    _is_builtin = True

    def run(self, iterable):
        iterable = self.to_numerical_iterable(iterable)
        return str(statistics.median(iterable))

class MeanFormula(FormulasFunction):

    doc = _('Return the mean value of the items in `iterable`.')
    name = 'mean'
    arg_count = 1
    _is_builtin = True

    def run(self, iterable):
        iterable = self.to_numerical_iterable(iterable)
        try:
            return str(statistics.mean(iterable))
        except statistics.StatisticsError as e:
            return '0'

class ModeFormula(FormulasFunction):

    doc = _('Return the mode value of the items in `iterable`.')
    name = 'mode'
    arg_count = 1
    _is_builtin = True

    def run(self, iterable):
        iterable = self.to_numerical_iterable(iterable)
        return str(statistics.mode(iterable))

class MultiplyByFormula(FormulasFunction):

    doc = _('Return a new list comprised of all items in `iterable` multiplied by `factor`.')
    name = 'multiply_by'
    arg_count = -1
    _is_builtin = True

    def run(self, iterable, factor, sep=','):
        factor = float(factor)
        iterable = self.to_numerical_iterable(iterable)
        res = [ x * factor for x in iterable ]
        return sep.join([str(x) for x in res])

BUILTIN_FORMULAS = [
    SelectionFormula,
    SearchFormula,
    ReplaceNonNumericalFormula,
    SumFormula,
    MinFormula,
    MaxFormula,
    MedianFormula,
    MeanFormula,
    ModeFormula,
    MultiplyByFormula
]

def get_user_formulas(plugin_action):
    user_formulas = {}
    for cls in plugin_action.user_modules.get_classes(class_filters=[FormulasFunction]):
        name = cls.name
        # must define a name attribute
        if name in ['', 'no name provided']:
            continue
        user_formulas[name] = cls            
    return user_formulas

def get_formulas(plugin_action):
    all_funcs = OrderedDict()

    builtin_formulas = OrderedDict()
    for cls in BUILTIN_FORMULAS:
        builtin_formulas[cls.name] = cls

    user_formulas = get_user_formulas(plugin_action)
    
    #template_funcs = get_template_functions(plugin_action)

    for formula_name, formula_cls in user_formulas.items():
        try:
            formula = formula_cls(plugin_action)
            all_funcs[formula_name] = formula
        except Exception as e:
            import traceback
            if DEBUG:
                prints(f'Action Chains: Error intializing user formula: {formula_name}\n{traceback.format_exc()}')

    for formula_name, formula_cls in builtin_formulas.items():
        formula = formula_cls(plugin_action)
        all_funcs[formula_name] = formula
    
#    for name, func in template_funcs.items():
#        all_funcs[name] = func

    return all_funcs

#===========================================================
# Formula Dialog
#===========================================================

class FormulaDialog(TemplateBox):
                                
    def __init__(self, parent, plugin_action, template, all_functions, global_vars={}):
        TemplateBox.__init__(
            self,
            parent,
            plugin_action,
            template_text=template,
            placeholder_text=_('Enter your formula here'),
            all_functions=all_functions,
            global_vars={}
        )
        self.setWindowTitle(_('Formulas'))
        self.help_label = QLabel(self)
        self.help_label.setWordWrap(True)
        self.help_label.setOpenExternalLinks(True)
        self.help_label.setObjectName("heading")
        self.help_label.setText('<a href="https://www.mobileread.com/forums/showpost.php?p=4077879&postcount=245">Formulas help</a>')
        self.help_label.setTextInteractionFlags(Qt.TextBrowserInteraction)
        self.help_label.setAlignment(Qt.AlignLeft)
        
        self.user_layout_1.addWidget(self.help_label)


#===========================================================
# config widget
#===========================================================

class FormulasConfigWidget(QWidget):
    def __init__(self, plugin_action, template=''):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.gui = plugin_action.gui
        self.db = self.gui.current_db
        self.template = template
        self._init_controls()

    def _init_controls(self):

        l = self.l = QVBoxLayout()
        self.setLayout(l)

        opts_gb = QGroupBox(_('Options: '), self)
        l.addWidget(opts_gb)
        opts_l = QVBoxLayout()
        opts_gb.setLayout(opts_l)
        self.fmts_layout = QHBoxLayout()
        self.runtime_opt = QRadioButton(_('Ask at runtime'))
        self.runtime_opt.setChecked(True)
        opts_l.addWidget(self.runtime_opt)
        template_layout = QHBoxLayout()
        self.predefined_opt = QRadioButton(_('Predefined'))
        self.predefined_opt.setEnabled(False)
        self.template_button = QPushButton(_('Add template'))
        self.template_button.clicked.connect(self._on_template_button_clicked)
        template_layout.addWidget(self.predefined_opt, 1)
        template_layout.addWidget(self.template_button)
        opts_l.addLayout(template_layout)
        
        l.addStretch(1)

        self.setMinimumSize(300,300)

    def _on_template_button_clicked(self):
        action = self.plugin_action.actions['Formulas']
        #action.formula_context_info('template_dialog_mode')
        d = FormulaDialog(self, self.plugin_action, self.template, all_functions=self.plugin_action.template_functions)
        if d.exec_() == d.Accepted:
            self.template = d.template
            self.predefined_opt.setEnabled(True)
            self.predefined_opt.setChecked(True)
            self.template_button.setText(_('Edit template'))

    def load_settings(self, settings):
        if settings:
            if settings['opt'] == 'runtime':
                self.runtime_opt.setChecked(True)
            elif settings['opt'] == 'predefined':
                self.predefined_opt.setEnabled(True)
                self.predefined_opt.setChecked(True)
                self.template = settings['template']
                self.template_button.setText(_('Edit template'))

    def save_settings(self):
        settings = {}
        if self.runtime_opt.isChecked():
            settings['opt'] = 'runtime'
        elif self.predefined_opt.isChecked():
            settings['opt'] = 'predefined'
            settings['template'] = self.template
        return settings

#===========================================================
# Chain Action
#===========================================================

class Formulas(ChainAction):

    name = 'Formulas'
    _is_builtin = True
    
    def __init__(self, plugin_action):
        ChainAction.__init__(self, plugin_action)
        self.gui = plugin_action.gui
        self.db = self.gui.current_db

    def run(self, gui, settings, chain):

        if settings['opt'] == 'predefined':
            template = settings['template']
        elif settings['opt'] == 'runtime':
            QApplication.setOverrideCursor(Qt.ArrowCursor)
            try:
                d = FormulaDialog(gui, self.plugin_action, '', all_functions=self.plugin_action.template_functions, global_vars=chain.chain_vars)
                if d.exec_() == d.Accepted:
                    template = d.template
                else:
                    return

            finally:
                QApplication.restoreOverrideCursor()

        mi  = get_metadata_object(gui)

        template_output = chain.evaluate_template(template, template_functions=self.plugin_action.template_functions)

        QApplication.setOverrideCursor(Qt.ArrowCursor)
        try:
            ViewLog(_('Formulas'), template_output, gui)
        finally:
            QApplication.restoreOverrideCursor()
        return template_output

    def validate(self, settings):
        if not settings:
            return (_('Settings Error'), _('You must configure this action before running it'))
            pass
        elif settings['opt'] == 'predefined':
            if not settings['template']:
                return (_('Settings Error'), _('No template value specified'))
            is_template_valid = check_template(settings['template'], self.plugin_action,
                                               print_error=False,
                                               template_functions=self.plugin_action.template_functions)
            if is_template_valid is not True:
                return is_template_valid
        return True

    def config_widget(self):
        return FormulasConfigWidget

    def on_templates_update(self):
        if DEBUG:
            prints('Action chains: formulas: running on_templates_update()')
        self.funcs = get_formulas(self.plugin_action)
        return self.funcs
