#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import copy
import os

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                     QLabel, QGroupBox, QPushButton, QLineEdit, QToolButton,
                     QRadioButton, QDialog, QDialogButtonBox, QCheckBox)

from calibre import prints
from calibre.constants import DEBUG, iswindows
from calibre.gui2 import error_dialog, choose_save_file
from calibre.ebooks.metadata.book.formatter import SafeFormat

from calibre_plugins.action_chains.actions.base import ChainAction
from calibre_plugins.action_chains.templates import check_template, TEMPLATE_ERROR
from calibre_plugins.action_chains.templates.dialogs import TemplateBox
from calibre_plugins.action_chains.common_utils import validate_writable_file_path, get_file_path, DragDropComboBox, get_icon

try:
    load_translations()
except NameError:
    prints("ActionChains::actions/copy_to_clipboard.py - exception when loading translations")

DEFAULT_SEP = os.linesep

def get_file_size_if_exists(file_path):
    try:
        return os.stat(file_path).st_size
    except:
        return 0

class TemplateError(Exception):
    def __init__(self, message, errors):
        self.message = message
        self.errors = errors

class CopyToClipboardConfigWidget(QWidget):
    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.gui = plugin_action.gui
        self.db = self.gui.current_db
        self.template = ''
        self._init_controls()

    def _init_controls(self):

        l = self.l = QVBoxLayout()
        self.setLayout(l)

        template_layout = QHBoxLayout()
        self.template_button = QPushButton(_('Add template'))
        self.template_button.clicked.connect(self._on_template_button_clicked)
        l.addWidget(self.template_button)
        
        multiple_box = QGroupBox()
        l.addWidget(multiple_box)
        multiple_l = QVBoxLayout()
        multiple_box.setLayout(multiple_l)
        multiple_chk = self.multiple_chk = QCheckBox(_('Allow multiple selections'))
        multiple_chk.setChecked(False)
        multiple_l.addWidget(multiple_chk)
        sep_box = QGroupBox(_('Separator'))
        sep_l = QHBoxLayout()
        sep_box.setLayout(sep_l)
        multiple_l.addWidget(sep_box)
        multiple_l.addWidget(sep_box)
        default_opt = self.default_opt = QRadioButton(_('New Line'))
        default_opt.setChecked(True)
        sep_l.addWidget(default_opt)
        sep_l.addStretch(1)
        other_opt = self.other_opt = QRadioButton(_('Other'))
        sep_l.addWidget(other_opt)
        other_edit = self.other_edit = QLineEdit()
        other_edit.setFixedWidth(30)
        sep_l.addWidget(other_edit)

        self.file_box = QGroupBox(_('(Optional) Copy to file:'))
        l.addWidget(self.file_box)
        self.file_box_layout = QVBoxLayout()
        self.file_box.setLayout(self.file_box_layout)
        file_layout = self.file_layout = QHBoxLayout()
        self.file_box_layout.addLayout(file_layout)
        self.file_combo = DragDropComboBox(self, drop_mode='file')
        file_layout.addWidget(self.file_combo)
        self.choose_file_button = QToolButton(self)
        self.choose_file_button.setToolTip(_('Choose file'))
        self.choose_file_button.setIcon(get_icon('document_open.png'))
        self.choose_file_button.clicked.connect(self._choose_file)
        file_layout.addWidget(self.choose_file_button)
        append_chk = self.append_chk = QCheckBox(_('Append to file if it already exists'))
        append_chk.setChecked(False)
        self.file_box_layout.addWidget(append_chk)
        
        l.addStretch(1)

        self.setMinimumSize(300,300)

    def _on_template_button_clicked(self):
        d = TemplateBox(self, self.plugin_action, template_text=self.template)
        if d.exec_() == d.Accepted:
            self.template = d.template
            self.template_button.setText(_('Edit template'))

    def _choose_file(self):
        file_path = choose_save_file(self, 'copy_to_clipboard_file', _('Choose file'), all_files=True)
        if not file_path:
            return
            
        if iswindows:
            file_path = os.path.normpath(file_path)

        self.block_events = True
        existing_index = self.file_combo.findText(file_path, Qt.MatchExactly)
        if existing_index >= 0:
            self.file_combo.setCurrentIndex(existing_index)
        else:
            self.file_combo.insertItem(0, file_path)
            self.file_combo.setCurrentIndex(0)
        self.block_events = False

    def load_settings(self, settings):
        if settings:
            self.template = settings['template']
            if self.template:
                self.template_button.setText(_('Edit Template'))
            if settings.get('allow_multiple'):
                self.multiple_chk.setChecked(True)
                sep = settings.get('sep', DEFAULT_SEP)
                if sep != DEFAULT_SEP:
                    self.other_opt.setChecked(True)
                    self.other_edit.setText(sep)
                else:
                    self.default_opt.setChecked(True)
            filename = settings.get('filename')
            if filename:
                self.file_combo.setCurrentText(filename)
                self.append_chk.setChecked(settings.get('append_to_file', False))

    def save_settings(self):
        settings = {}
        settings['template'] = self.template
        if self.multiple_chk.isChecked():
            settings['allow_multiple'] = True
            if self.default_opt.isChecked():
                pass
            elif self.other_opt.isChecked():
                settings['sep'] = self.other_edit.text()
        settings['filename'] = self.file_combo.currentText().strip()
        if settings['filename']:
            settings['append_to_file'] = self.append_chk.isChecked()
        return settings

class CopyToClipboardAction(ChainAction):

    name = 'Copy To Clipboard'
    _is_builtin_ = True
    support_scopes = True

    def get_template_output(self, db, book_id, settings, chain, show_error=False):   

        template = settings['template']
        template_output = chain.evaluate_template(template, book_id)
        if template_output.startswith(TEMPLATE_ERROR):
            raise TemplateError(_('Error'), _('Template returned error'))
        return template_output        

    def run(self, gui, settings, chain):
        if settings.get('allow_multiple'):
            book_ids = chain.scope().get_book_ids()
        else:
            current_id = chain.scope().get_current_book_id()
            if current_id:
                book_ids = [current_id]
            else:
                book_ids = []
        
        db = gui.current_db

        all_templates = []
        error_msg = ''
        for book_id in book_ids:
            try:
                template_for_book = self.get_template_output(db, book_id, settings, chain)
                if template_for_book:
                    all_templates.append(template_for_book)
                else:
                    if DEBUG:
                        prints(f'Action Chains: value not found for book: {book_id}')
            except TemplateError as e:
                #return error_dialog(gui, e.message, e.errors, show=True)
                book_title = db.title(book_id, index_is_id=True)
                error_msg += f'{book_title} ({book_id}): {e.errors}\n'
                
        if error_msg:
            if DEBUG:
                prints(f'Actions Chain: Copy to clipboard: Errors with the following books:\n{error_msg}')

        if not all_templates:
            return
            
        sep = settings.get('sep', DEFAULT_SEP)
        clipboard_payload = sep.join(all_templates)

        QApplication.clipboard().setText(clipboard_payload)
        
        filename = settings.get('filename')
        if filename:
            file_path = get_file_path(filename)
        else:
            file_path = ''
        mode = 'a' if settings.get('append_to_file') else 'w'
        if clipboard_payload and mode == 'a':
            if get_file_size_if_exists(file_path) > 0:
                clipboard_payload = sep + clipboard_payload
        if filename:
            with open(file_path, mode) as f:
                f.write(clipboard_payload)

    def validate(self, settings):
        if not settings:
            return (_('Settings Error'), _('You must configure this action before running it'))
        template = settings['template']
        if not template:
            return (_('Template Error'), _('You must enter a template for this action'))
        is_template_valid = check_template(template, self.plugin_action, print_error=False)
        if is_template_valid is not True:
            return is_template_valid
        filename = settings.get('filename')
        if filename:
            file_path = get_file_path(filename)
            if not validate_writable_file_path(file_path):
                return (_('Invalid filename'), _(f'Filename: {file_path} is not a valid filename'))
        return True

    def config_widget(self):
        return CopyToClipboardConfigWidget


