#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2022, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import os
import copy
from collections import defaultdict

from qt.core import (Qt, QApplication, QWidget, QVBoxLayout, QHBoxLayout,
                     QGridLayout, QCheckBox, QIcon, QGroupBox, QLabel,
                     QFrame, QPushButton)

from calibre import prints
from calibre.constants import DEBUG
from calibre.customize.conversion import OptionRecommendation
from calibre.customize.ui import (
    input_format_plugins, output_format_plugins, plugin_for_input_format,
    plugin_for_output_format)
from calibre.gui2.convert.gui_conversion import gui_convert
from calibre.gui2.preferences.conversion import (
    OutputOptions, InputOptions, CommonOptions)
from calibre.gui2.preferences import AbortCommit
from calibre.gui2.widgets2 import Dialog
from calibre.ebooks.conversion.config import (
    get_output_formats, get_input_format_for_book,
    sort_formats_by_preference, NoSupportedInputFormats,
    GuiRecommendations)
from calibre.ebooks.conversion.plumber import supported_input_formats
from calibre.ptempfile import (
    PersistentTemporaryFile, PersistentTemporaryDirectory)
from calibre.utils.config import prefs
from calibre.utils.date import now

from calibre_plugins.action_chains.actions.base import ChainAction
from calibre_plugins.action_chains.templates.dialogs import TemplateLineEditor
from calibre_plugins.action_chains.actions.calibre_actions import (
    unfinished_job_ids, responsive_wait, responsive_wait_until)

def sync_dict_with_defaults(default_dict, new_dict):
    '''
    make sure all keys for new_dict and its subdictionaries
    are in sync with default_dict.
    '''
    for k, default_value in default_dict.items():
        try:
            new_value = new_dict[k]
            if isinstance(default_value, dict):
                if isinstance(new_value, dict):
                    sync_dict_with_defaults(default_value, new_value)
        except KeyError:
            new_dict[k] = copy.deepcopy(default_value)

class WidgetDialog(Dialog):
    def __init__(self, parent, title, name, widget_cls):
        self.plugin_action = parent.plugin_action
        self.gui = self.plugin_action.gui
        self.db = self.gui.current_db
        self.widget = widget_cls()
        Dialog.__init__(self, title, name, parent)

    def setup_ui(self):
        l = QVBoxLayout()
        self.setLayout(l)

        l.addWidget(self.widget)
        self.widget.genesis(self.gui)

        l.addWidget(self.bb)

    def set_recommendations(self, recs):
        for widget in self.widget.model.widgets:
            widget_recs = recs[widget.COMMIT_NAME]
            widget.apply_recommendations(widget_recs)

    def get_recommendations(self):
        recs = {}
        for widget in self.widget.model.widgets:
            if not widget.pre_commit_check():
                raise AbortCommit('abort')
            widget_recs = widget.commit(save_defaults=False)
            recs[widget.COMMIT_NAME] = widget_recs
        return recs

    def accept(self):
        self.recs = self.get_recommendations()
        Dialog.accept(self)

class ConfigWidget(QWidget):

    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.gui = plugin_action.gui
        self.db = self.gui.current_db
        #self.input_options = self.common_options = self.output_options = GuiRecommendations()
        self.input_options_defaults =self.input_options = WidgetDialog(self, _('Input Options'), 'convert-action-input-options', InputOptions).get_recommendations()
        self.common_options_defaults = self.common_options =  WidgetDialog(self, _('Common Options'), 'convert-action-common-options', CommonOptions).get_recommendations()
        self.output_options_defaults = self.output_options = WidgetDialog(self, _('Output Options'), 'convert-action-output-options', OutputOptions).get_recommendations()
        self._init_controls()

    def _init_controls(self):

        l = QVBoxLayout()
        self.setLayout(l)

        input_label = QLabel(_('Input formats (comma separated list oredered by most preferred)'))
        l.addWidget(input_label)        
        self.input_formats_edit = TemplateLineEditor(self, self.plugin_action)
        l.addWidget(self.input_formats_edit)

        label_1 = QLabel('<b>Note:</b> You can use "any" as input format, if you want calibre to decide')
        label_1.setWordWrap(True)
        l.addWidget(label_1)

        line = QFrame(self)
        line.setFrameShape(QFrame.HLine)
        line.setFrameShadow(QFrame.Sunken)
        l.addWidget(line)

        output_label = QLabel(_('Output format'))
        l.addWidget(output_label)
        self.output_formats_edit = TemplateLineEditor(self, self.plugin_action)
        l.addWidget(self.output_formats_edit)
        self.convert_if_exists_chk = QCheckBox('Convert even if output format exists (will overwrite)')
        l.addWidget(self.convert_if_exists_chk)

        #job_wait_chk = self.job_wait_chk = QCheckBox(_('Wait until any jobs stared by the selected action finishes.'))
        #l.addWidget(job_wait_chk)
        #job_wait_chk.setChecked(True)

        self.input_button = QPushButton(_('Input Options'))
        l.addWidget(self.input_button)
        self.input_button.clicked.connect(self._on_input_button_clicked)

        self.common_button = QPushButton(_('Common Options'))
        l.addWidget(self.common_button)
        self.common_button.clicked.connect(self._on_common_button_clicked)

        self.output_button = QPushButton(_('Output Options'))
        l.addWidget(self.output_button)
        self.output_button.clicked.connect(self._on_output_button_clicked)

        l.addStretch(1)

        self.setMinimumSize(400,500)

    def _on_input_button_clicked(self):
        d = WidgetDialog(self, _('Input Options'), 'convert-action-input-options', InputOptions)
        d.set_recommendations(self.input_options)
        if d.exec_() == d.Accepted:
            recs = d.get_recommendations()
            self.input_options = recs

    def _on_output_button_clicked(self):
        d = WidgetDialog(self, _('Output Options'), 'convert-action-output-options', OutputOptions)
        d.set_recommendations(self.output_options)
        if d.exec_() == d.Accepted:
            recs = d.get_recommendations()
            self.output_options = recs

    def _on_common_button_clicked(self):
        d = WidgetDialog(self, _('Common Options'), 'convert-action-common-options', CommonOptions)
        d.set_recommendations(self.common_options)
        if d.exec_() == d.Accepted:
            recs = d.get_recommendations()
            self.common_options = recs

    def load_settings(self, settings):
        if settings:
            self.input_options = settings['input_options']
            self.common_options = settings['common_options']
            self.output_options = settings['output_options']
            sync_dict_with_defaults(self.input_options_defaults, self.input_options)
            sync_dict_with_defaults(self.common_options_defaults, self.common_options)
            sync_dict_with_defaults(self.output_options_defaults, self.output_options)
            self.input_formats_edit.setText(settings['input_formats'])
            self.output_formats_edit.setText(settings['output_formats'])
            self.convert_if_exists_chk.setChecked(settings['convert_if_format_exists'])
            #self.job_wait_chk.setChecked(settings['wait_jobs'])

    def save_settings(self):
        settings = {}
        settings['input_options'] = self.input_options
        settings['common_options'] = self.common_options
        settings['output_options'] = self.output_options
        settings['input_formats'] = self.input_formats_edit.text()
        settings['output_formats'] = self.output_formats_edit.text()
        settings['convert_if_format_exists'] = self.convert_if_exists_chk.isChecked()
        #settings['wait_jobs'] = self.job_wait_chk.isChecked()
        return settings

class ConvertBooksAction(ChainAction):

    name = 'Convert Books'
    _is_builtin = True
    support_scopes = True

    def get_recommendations(self, settings, input_format, output_format):
        input_commit_name = f'{input_format.lower()}_input'
        output_commit_name = f'{output_format.lower()}_output'
        input_plugin = self.input_plugins.get(input_format)
        output_plugin = self.output_plugins.get(output_format)
        if input_plugin:
            if hasattr(input_plugin, 'COMMIT_NAME'):
                input_commit_name = getattr(input_plugin, 'COMMIT_NAME')
            else:
                if DEBUG:
                    prints(f'Action Chains: Convert Books: Input plugin ({input_format}) has no commit name')
        if output_plugin:
            if hasattr(output_plugin, 'COMMIT_NAME'):
                output_commit_name = getattr(output_plugin, 'COMMIT_NAME')
            else:
                if DEBUG:
                    prints(f'Action Chains: Convert Books: Output plugin ({output_format}) has no commit name')

        all_input_options = settings['input_options']
        all_common_options = settings['common_options']
        all_output_options = settings['output_options']

        input_options = all_input_options.get(input_commit_name, {})
        output_options = all_output_options.get(output_commit_name, {})
        common_options = {}
        for k, v in all_common_options.items():
            common_options.update(v)

        options = {}
        options.update(common_options)
        options.update(input_options)
        options.update(output_options)
        recommendations = [(k, v, OptionRecommendation.HIGH) for k, v in options.items()]
        return recommendations

    def run(self, gui, settings, chain):
        db = gui.current_db
        tdir = PersistentTemporaryDirectory('_ac_convert')
        self.input_plugins = {}
        self.output_plugins = {}
        supported_input = supported_input_formats()
        supported_output = get_output_formats(None)
        book_ids_by_input_output_formats = defaultdict(set)
        book_ids = chain.scope().get_book_ids()
        if DEBUG:
            prints(f'Action Chains: Convert Books: allowed input formats: {supported_input} | supported_output_formats: {supported_output}')
        for book_id in book_ids:
            input_template_output = chain.evaluate_template(settings.get('input_formats', ''), book_id=book_id)
            preferred_input_formats = [ x.strip().upper() for x in input_template_output.split(',') ]
            output_template_output = chain.evaluate_template(settings.get('output_formats', ''), book_id=book_id)
            preferred_output_formats = [ x.strip().upper() for x in output_template_output.split(',') ]
            book_formats = db.formats(book_id, index_is_id=True)
            if book_formats:
                book_formats = book_formats.split(',')
            else:
                book_formats = []
                if DEBUG:
                    prints(f'Action Chains: Convert Books: Book has no formats, book_id: {book_id}. '
                           f'Skipping conversion ... ')
                continue
            input_format = ''
            output_format = ''
            for fmt in preferred_input_formats:
                if fmt.lower() == 'any':
                    input_format = fmt
                    break
                elif fmt in book_formats and (fmt.lower() in supported_input):
                    input_format = fmt
                    break
            else:
                if DEBUG:
                    prints(f'Action Chains: Convert Books: No supported input format found for book_id: {book_id}')
            for fmt in preferred_output_formats:
                if fmt in supported_output:
                    if fmt in book_formats:
                        if settings.get('convert_if_format_exists', False):
                            output_format = fmt
                            break
                        else:
                            prints(f'Action Chains: Covert Books: Output format already exitst for book_id: {book_id}. Skipping ...')
                            continue
                    else:
                        output_format = fmt
                        break

            if input_format.lower() == 'any':
                try:
                    input_format, book_input_formats = get_input_format_for_book(db, book_id)
                except NoSupportedInputFormats:
                    if DEBUG:
                        prints(f'Action Chains: Convert Books: No input format found for book_id: {book_id}. Skipping conversion ... ')
                    continue            


#            output_format = output_format if \
#                output_format in get_output_formats(output_format) else \
#                sort_formats_by_preference(supported_output,
#                        [prefs['output_format']])[0]

            if DEBUG:
                prints(f'Action Chains: Convert Books: book_id: {book_id}\n'
                       f'preferred_output_formats: {preferred_output_formats}\n'
                       f'preferred_input_formats: {preferred_input_formats}\n'
                       f'output_format: {output_format}\n'
                       f'input_format: {input_format}\n'
                       f'book_formats: {book_formats}\n'
                       f'supported_input_formats: {supported_input}')

            if not input_format:
                if DEBUG:
                    prints(f'Action Chains: Convert Books: No input format found for book_id: {book_id}. '
                           'Skipping conversion ... ')
                continue

            if not output_format:
                if DEBUG:
                    prints(f'Action Chains: Convert Books: No output format found for book_id: {book_id}. '
                           'Skipping conversion ... ')
                continue

            if not self.input_plugins.get(input_format.lower()):
                self.input_plugins[input_format.lower()] = plugin_for_input_format(input_format)
                if self.input_plugins.get(input_format.lower()) is None:
                    prints(f'No plugin to handle input format: {input_format}')
                    # some input_fmts (e.g. zip) return None plugin_for_input_format()
                    # they can still be used by calibre conversion process
                    #continue
            if not self.output_plugins.get(output_format.lower()):
                self.output_plugins[output_format.lower()] = plugin_for_output_format(input_format)
                if self.output_plugins.get(output_format.lower()) is None:
                    prints(f'No plugin to handle output format: {output_format}')
                    continue

            book_ids_by_input_output_formats[input_format+'|'+output_format].add(book_id)

        jobs_before_ids = unfinished_job_ids(gui)

        start_time = now()

        for input_output, book_ids in book_ids_by_input_output_formats.items():
            input_format, output_format = input_output.split('|')
            for book_id in book_ids:
                title = db.new_api.field_for('title', book_id)
                if DEBUG:
                    prints(f'Action Chains: Convert Books: Start converting book: {title} ({book_id})')
                input_path = db.format_abspath(book_id, input_format, index_is_id=True)
                output_path = os.path.join(tdir, str(book_id) + '.' + output_format.lower())
                recommendations = self.get_recommendations(settings, input_format, output_format)
                if DEBUG:
                    prints(f'Action Chains: Convert Books: recommendations: {recommendations}')
                #TODO: launch parallel jobs for different book_ids instead of performing conversions sequentially
                gui_convert(input_path, output_path, recommendations)
                db.new_api.add_format(book_id, output_format, output_path)
                os.remove(output_path)

        wait_jobs = settings.get('wait_jobs', True)

        if wait_jobs:            
            # wait for jobs spawned by action to kick in
            responsive_wait(1)
            
            # save ids of jobs started after running the action                    
            ids_jobs_by_action = unfinished_job_ids(gui).difference(jobs_before_ids)

            # wait for jobs to finish
            responsive_wait_until(lambda: ids_jobs_by_action.intersection(unfinished_job_ids(gui)) == set())         

    def validate(self, settings):
        if not settings:
            return _('Settings error'), _('This action must be configured')
        if not settings.get('input_formats'):
            return _('Settings error'), _('You must specify at least one input format. If you want calibre to decide, use "any" for format.')
        if not settings.get('output_formats'):
            return _('Settings error'), _('You must specify output format.')
        return True

    def config_widget(self):
        return ConfigWidget

