#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'


from functools import partial

from qt.core import (QApplication, Qt, QWidget, QVBoxLayout, QHBoxLayout, QToolButton,
                      QGroupBox, QAbstractTableModel, QModelIndex, QAbstractItemView,
                      QIcon, QPushButton, QSpacerItem, QSizePolicy, pyqtSignal, QLabel,
                      QFrame, QDialog, QDialogButtonBox)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog

import calibre_plugins.action_chains.config as cfg
from calibre_plugins.action_chains.common_utils import get_icon
from calibre_plugins.action_chains.templates import check_template
from calibre_plugins.action_chains.templates.dialogs import TemplateBox
from calibre_plugins.action_chains.actions.base import ChainAction
from calibre_plugins.action_chains.gui.views import TableView, safe_name

from calibre.constants import numeric_version
if numeric_version >= (5,99,0):
    QT_CHECKED = Qt.CheckState.Checked.value
    QT_UNCHECKED = Qt.CheckState.Unchecked.value
else:
    QT_CHECKED = Qt.Checked
    QT_UNCHECKED = Qt.Unchecked


try:
    load_translations()
except NameError:
    prints("ActionChains::actions/chain_variables.py - exception when loading translations")

DOWN    = 1
UP      = -1

class VariablesModel(QAbstractTableModel):

    error = pyqtSignal(str, str)

    def __init__(self, chain_vars, plugin_action):
        QAbstractTableModel.__init__(self)
        self.chain_vars = chain_vars
        self.plugin_action = plugin_action
        self.col_map = ['iterate','runtime','name','value', '']
        self.editable_columns = ['name']
        self.hidden_cols = []
        self.col_min_width = {
            'name': 150,
            'value': 150,
            'iterate': 25
        }
        all_headers = [_('Iterate'), _('Runtime'), _('Name'), _('Value'), '']
        self.headers = all_headers

    def rowCount(self, parent):
        if parent and parent.isValid():
            return 0
        return len(self.chain_vars)

    def columnCount(self, parent):
        if parent and parent.isValid():
            return 0
        return len(self.headers)

    def headerData(self, section, orientation, role):
        if role == Qt.DisplayRole and orientation == Qt.Horizontal:
            return self.headers[section]
        elif role == Qt.DisplayRole and orientation == Qt.Vertical:
            return section + 1

        if role == Qt.ToolTipRole and orientation == Qt.Horizontal:
            col_name = self.col_map[section]
            if col_name == 'iterate':
                return _('Checking this box will run the template for all books in current scope')
            elif col_name == 'runtime':
                return _('Checking this box will ask for the value of this variable at runtime')

        return None

    def data(self, index, role):
        if not index.isValid():
            return None;
        row, col = index.row(), index.column()
        if row < 0 or row >= len(self.chain_vars):
            return None
        chain_var = self.chain_vars[row]
        col_name = self.col_map[col]
        value = chain_var.get(col_name, '')

        if role in [Qt.DisplayRole, Qt.UserRole, Qt.EditRole]:
            if col_name in ['iterate', 'runtime']:
                pass
            else:
                return value

        elif role == Qt.CheckStateRole:
            if col_name in ['iterate', 'runtime']:
                is_checked = chain_var.get(col_name, False)
                state = QT_CHECKED if is_checked else QT_UNCHECKED
                return state

        return None

    def setData(self, index, value, role):
        done = False

        row, col = index.row(), index.column()
        chain_var = self.chain_vars[row]
        val = str(value).strip()
        col_name = self.col_map[col]
        
        if role == Qt.EditRole:
            if col_name in ['iterate', 'runtime']:
                pass
            elif col_name == 'name':
                old_name = self.data(index, Qt.DisplayRole)
                names = self.get_names()
                if old_name in names:
                    names.remove(old_name)
                if not val:
                    msg = _('Invalid value')
                    details = _(f'Name cannot be empty: {val}')
                    self.error.emit(msg, details)
                    return False
                if val in names:
                    msg = _('Name must be unique')
                    details = _(f'Name ({val}) is already used by another menu')
                    self.error.emit(msg, details)
                    return False
                else:
                    chain_var[col_name] = val
            elif col_name == 'value':
                is_valid = check_template(val, self.plugin_action, print_error=False)
                if is_valid is not True:
                    msg, details = is_valid
                    self.error.emit(msg, details)
                    return False
                chain_var[col_name] = val
            done = True

        elif role == Qt.CheckStateRole:
            if col_name in ['iterate', 'runtime']:
                state = True if (value == QT_CHECKED) else False
                chain_var[col_name] = state
                # iterate and runtime cannot be True at the same time
                if state is True:
                    if col_name == 'iterate':
                        chain_var['runtime'] = False
                    elif col_name == 'runtime':
                        chain_var['iterate'] = False
                    self.layoutChanged.emit()
            return True
            
        return done

    def flags(self, index):
        flags = QAbstractTableModel.flags(self, index)
        if index.isValid():
            chain_var = self.chain_vars[index.row()]
            col_name = self.col_map[index.column()]
            if col_name in self.editable_columns:
                flags |= Qt.ItemIsEditable
            if col_name in ['iterate', 'runtime']:
                flags |=  Qt.ItemIsUserCheckable
        return flags

    def insertRows(self, row, count, idx):
        self.beginInsertRows(QModelIndex(), row, row + count - 1)
        all_names = self.get_names()
        for i in range(0, count):
            chain_var = {}
            new_name = 'new_var'
            new_name = safe_name(new_name, all_names)
            chain_var['name'] = new_name
            chain_var['value'] = ''
            chain_var['iterate'] = False
            chain_var['runtime'] = False
            self.chain_vars.insert(row + i, chain_var)
        self.endInsertRows()
        return True

    def removeRows(self, row, count, idx):
        self.beginRemoveRows(QModelIndex(), row, row + count - 1)
        for i in range(0, count):
            self.chain_vars.pop(row + i)
        self.endRemoveRows()
        return True

    def get_names(self):
        names = []
        col = self.col_map.index('name')
        for row in range(self.rowCount(QModelIndex())):
            index = self.index(row, col, QModelIndex())
            name = self.data(index, Qt.DisplayRole)
            # empty name belong to separators, dont include
            if name:
                names.append(name)
        return names

    def move_rows(self, rows, direction=DOWN):
        srows = sorted(rows, reverse=direction == DOWN)
        for row in srows:
            pop = self.chain_vars.pop(row)
            self.chain_vars.insert(row+direction, pop)
        self.layoutChanged.emit() 

class VariablesTable(TableView):

    def __init__(self, parent):
        TableView.__init__(self, parent)
        self.plugin_action = parent.plugin_action
        self.doubleClicked.connect(self._on_double_clicked)

    def set_model(self, _model):
        self.setModel(_model)
        _model.error.connect(lambda *args: error_dialog(self, *args, show=True))
        self.col_map = _model.col_map

        # Hide columns
        for col_name in _model.hidden_cols:
            col = self.col_map.index(col_name)
            self.setColumnHidden(col, True)

        self.resizeColumnsToContents()
        # Make sure every other column has a minimum width
        for col_name, width in _model.col_min_width.items():
            col = self.col_map.index(col_name)
            self._set_minimum_column_width(col, width)

    def _on_double_clicked(self, index):
        srcm = self.source_model()
        col_name = srcm.col_map[index.column()]
        if col_name == 'value':
            self.open_template_dialog()
        else:
            self.edit(index)

    def open_template_dialog(self):
        index = self.currentIndex()
        srcm = self.source_model()
        col_name = srcm.col_map[index.column()]
        if col_name == 'value':
            template = self.model().data(index, Qt.DisplayRole)

            d = TemplateBox(self,
                           self.plugin_action,
                           template_text=template,
                           global_vars={})

            if d.exec_() == d.Accepted:
                self.model().setData(index, d.template, Qt.EditRole)

class ChainVariblesConfigWidget(QWidget):

    HELP = _(
        '<p>Using this action you can set chain variables that can '
        'be utilized in multiple ways like marking books or using '
        'them in conditional execution of actions.</p>'
        '<p>The plugin offers runtime-defined chain variables, '
        'action variables.</p>'
        '<p>Runtime chain variables are: _chain_name, _chain_iteration. '
        '<br>'
        'Runtime action variables are: _action_name, _action_index,'
        '_action_comment'
        '<br>'
        'Other variables include: location</p>'
        '<p>You can set your variables to calculated at runtime by '
        'using calibre\'s template language.</p>'
        '<br>'
        '<p><b>Note:</b> checking the iterate checkbox makes the '
        'template iterate over all selected books.</p>'
    )

    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.gui = plugin_action.gui
        self.db = self.gui.current_db
        self._init_controls()

    def _init_controls(self):

        l = self.l = QVBoxLayout()
        self.setLayout(l)

        self.heading = QLabel(self)
        self.heading.setWordWrap(True)
        self.heading.setOpenExternalLinks(True)
        self.heading.setObjectName("heading")
        self.heading.setText('<a href="https://www.mobileread.com/forums/showpost.php?p=4073181&postcount=171">help</a>')
        self.heading.setTextInteractionFlags(Qt.TextBrowserInteraction)
        self.heading.setOpenExternalLinks(True)
        self.heading.setToolTip(self.HELP)
        self.heading.setAlignment(Qt.AlignRight)
        l.addWidget(self.heading)

        gb1 = QGroupBox(_('Chain variables'))
        l.addWidget(gb1)
        gb1_l = QHBoxLayout()
        gb1.setLayout(gb1_l)
        self.table = VariablesTable(self)
        gb1_l.addWidget(self.table)

        self._model = VariablesModel([], self.plugin_action)
        self.table.set_model(self._model)
        #FIXME: The slot is not triggred when selection changes
        self.table.selectionModel().selectionChanged.connect(self._on_table_selection_change)

        # restore table state
        state = cfg.plugin_prefs[cfg.KEY_GPREFS][cfg.KEY_VARS_TABLE_STATE]
        if state:
            self.table.apply_state(state)

        # Add a vertical layout containing the the buttons to move up/down etc.
        button_layout = QVBoxLayout()
        gb1_l.addLayout(button_layout)

        move_up_button = self.move_up_button = QToolButton(self)
        move_up_button.setToolTip(_('Move row up'))
        move_up_button.setIcon(get_icon('arrow-up.png'))
        button_layout.addWidget(move_up_button)
        spacerItem1 = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem1)

        add_button = self.add_button = QToolButton(self)
        add_button.setToolTip(_('Add row'))
        add_button.setIcon(get_icon('plus.png'))
        button_layout.addWidget(add_button)
        spacerItem2 = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem2)

        delete_button = self.delete_button = QToolButton(self)
        delete_button.setToolTip(_('Delete row'))
        delete_button.setIcon(get_icon('minus.png'))
        button_layout.addWidget(delete_button)
        spacerItem3 = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem3)

        move_down_button = self.move_down_button = QToolButton(self)
        move_down_button.setToolTip(_('Move row down'))
        move_down_button.setIcon(get_icon('arrow-down.png'))
        button_layout.addWidget(move_down_button)

        move_up_button.clicked.connect(partial(self.table.move_rows,UP))
        move_down_button.clicked.connect(partial(self.table.move_rows,DOWN))
        add_button.clicked.connect(self.table.add_row)
        delete_button.clicked.connect(self.table.delete_rows)

        # Add a horizontal layout at the bottom containing extra buttons
        button_layout2 = QHBoxLayout()
        l.addLayout(button_layout2)

        template_functions_button = QPushButton(_('Template Functions'))
        template_functions_button.setToolTip(_('Edit template functions'))
        template_functions_button.setIcon(get_icon('template_funcs.png'))
        button_layout2.addWidget(template_functions_button)

        button_layout2.addStretch(1)

        template_functions_button.clicked.connect(self.gui.iactions['Template Functions'].show_template_editor)
        
        l.addStretch(1)

        self.setMinimumSize(600,300)

        #FIXME: This is comment out until problem with selectionChanged siganl is fixed
        #self._on_table_selection_change()

    def _on_table_selection_change(self):
        sm = self.table.selectionModel()
        selection_count = len(sm.selectedRows())
        self.delete_button.setEnabled(selection_count > 0)
        self.move_up_button.setEnabled(selection_count > 0)
        self.move_down_button.setEnabled(selection_count > 0)

    def save_table_state(self):
        # save table state
        cfg.plugin_prefs[cfg.KEY_GPREFS][cfg.KEY_VARS_TABLE_STATE] = self.table.get_state()

    def load_settings(self, settings):
        chain_vars = settings.get('chain_vars', [])
        self._model = VariablesModel(chain_vars, self.plugin_action)
        self.table.set_model(self._model)

    def save_settings(self):
        self.save_table_state()
        settings = {}
        settings['chain_vars'] = self._model.chain_vars
        return settings

#######################
# Runtime pop-up dialog
#######################

class RuntimeModel(QAbstractTableModel):

    def __init__(self, runtime_vars, plugin_action):
        QAbstractTableModel.__init__(self)
        self.runtime_vars = runtime_vars
        self.plugin_action = plugin_action
        self.col_map = ['name','value']
        self.editable_columns = ['value']
        self.hidden_cols = []
        self.col_min_width = {
            'name': 150,
            'value': 150
        }
        all_headers = [_('Name'),_('Value')]
        self.headers = all_headers

    def rowCount(self, parent):
        if parent and parent.isValid():
            return 0
        return len(self.runtime_vars)

    def columnCount(self, parent):
        if parent and parent.isValid():
            return 0
        return len(self.headers)

    def headerData(self, section, orientation, role):
        if role == Qt.DisplayRole and orientation == Qt.Horizontal:
            return self.headers[section]
        elif role == Qt.DisplayRole and orientation == Qt.Vertical:
            return section + 1
        return None

    def data(self, index, role):
        if not index.isValid():
            return None;
        row, col = index.row(), index.column()
        if row < 0 or row >= len(self.runtime_vars):
            return None
        runtime_var = self.runtime_vars[row]
        col_name = self.col_map[col]
        value = runtime_var.get(col_name, '')

        if role in [Qt.DisplayRole, Qt.UserRole, Qt.EditRole]:
            return value

        return None

    def setData(self, index, value, role):
        done = False

        row, col = index.row(), index.column()
        runtime_var = self.runtime_vars[row]
        val = str(value).strip()
        col_name = self.col_map[col]
        
        if role == Qt.EditRole:
            runtime_var[col_name] = val
            done = True            
        return done

    def flags(self, index):
        flags = QAbstractTableModel.flags(self, index)
        if index.isValid():
            runtime_var = self.runtime_vars[index.row()]
            col_name = self.col_map[index.column()]
            if col_name in self.editable_columns:
                flags |= Qt.ItemIsEditable
        return flags 

class RuntimeTable(TableView):

    def __init__(self, parent):
        TableView.__init__(self, parent)
        self.plugin_action = parent.plugin_action

    def set_model(self, _model):
        self.setModel(_model)
        self.col_map = _model.col_map

class RuntimeDialog(QDialog):
    def __init__(self, parent, plugin_action, runtime_vars):
        QDialog.__init__(self, parent)
        self.plugin_action = plugin_action
        self.runtime_vars = runtime_vars
        self._init_controls()

    def _init_controls(self):
        l = self.l = QVBoxLayout()
        self.setLayout(l)
        
        self.table = RuntimeTable(self)
        model = RuntimeModel(self.runtime_vars, self.plugin_action)
        self.table.set_model(model)
        l.addWidget(self.table)

        button_box = QDialogButtonBox(QDialogButtonBox.Close)
        button_box.accepted.connect(self.accept)
        button_box.rejected.connect(self.reject)
        l.addWidget(button_box)
        
        self.setWindowTitle(_('Enter Variables Values'))

#######################

class ChainVariablesAction(ChainAction):

    name = 'Chain Variables'
    _is_builtin = True
    support_scopes = True     

    def run(self, gui, settings, chain):
        
        db = gui.current_db
        selected_book_ids = chain.scope().get_book_ids()

        # Get values for runtime variables first
        runtime_vars = []
        for chain_var in settings['chain_vars'][:]:
            name = chain_var['name']
            if chain_var.get('runtime'):
                runtime_vars.append(chain_var)

        if len(runtime_vars) > 0:
            runtime_dialog = RuntimeDialog(gui, self.plugin_action, runtime_vars)
            
            QApplication.setOverrideCursor(Qt.ArrowCursor)
            try:
                runtime_dialog.exec_()
            finally:
                QApplication.restoreOverrideCursor()

        # The order of processing of chain_var is gauranteed to allow them to reference earlier vars
        for chain_var in settings['chain_vars']:
            name = chain_var['name']
            template = chain_var['value']
            if chain_var.get('iterate'):
                book_ids = selected_book_ids
            else:
                book_ids = [None]
            for book_id in book_ids:
                val = chain.evaluate_template(template, book_id)
                chain.set_chain_vars({name: val})       

    def validate(self, settings):
        chain_vars = settings.get('chain_vars', [])
        if not chain_vars:
            return (_('Settings Error'), _('You must add chain variables.'))
        return True

    def config_widget(self):
        return ChainVariblesConfigWidget


