#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                      QGroupBox, QComboBox, QCheckBox)

from calibre import prints
from calibre.constants import DEBUG

from calibre_plugins.action_chains.actions.base import ChainAction
from calibre_plugins.action_chains.chains import Chain
import calibre_plugins.action_chains.config as cfg

try:
    load_translations()
except NameError:
    prints("ActionChains::actions/chain_caller.py - exception when loading translations")

def validate_stack(chain_name, stack=[]):
    '''
    Retrun true only after making sure all called chains are valid, and no
    infinite recursion is happening
    '''
    if chain_name in stack:
        return _('Recursive chain calls'), _(f'When calling chains circualr recursion found in this stack: {stack + [chain_name]}')
    stack.append(chain_name)
    all_chain_names = [chain_config['menuText'] for chain_config in cfg.get_chains_config() if chain_config['menuText']]
    chain_config = cfg.get_chain_config(chain_name)
    chain_settings = chain_config.get('chain_settings', {})
    chain_links = chain_settings.get('chain_links',[])
    if len(chain_links) == 0:
        return True
    for chain_link in chain_links:
        action_name = chain_link['action_name']
        action_settings = chain_link.get('action_settings')
        if action_name == 'Chain Caller':
            target_chain_name = action_settings.get('chain_name')
            if not target_chain_name in all_chain_names:
                return _('Chain unavailable'), _(f'Chain ({target_chain_name}) is not available. stack({stack})')
            is_stack_valid = validate_stack(target_chain_name, stack)
            if is_stack_valid is not True:
                return is_stack_valid
    popped = stack.pop(-1)
    return True

def validate(settings, chains_config=[], parent_chain_name=None):
    if not chains_config:
        chains_config = cfg.get_chains_config()
    if not settings:
        return (_('Settings Error'), _('You must configure this action before running it'))
    target_chain_name = settings['chain_name']
    all_chain_names = [chain_config['menuText'] for chain_config in chains_config if chain_config['menuText']]
    if target_chain_name not in all_chain_names:
        return (_('Cahin Error'), _(f'Called chain ({target_chain_name}) is not currently available.'))
    # Validate the chain to make sure no infinite recursion occur
    stack = []
    if parent_chain_name: stack = [parent_chain_name]
    is_stack_valid = validate_stack(target_chain_name, stack=stack)
    if is_stack_valid is not True:
        return is_stack_valid
    return True

class ChainCallerConfigWidget(QWidget):
    def __init__(self, plugin_action, chain_name, chains_config, *args, **kwargs):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.gui = plugin_action.gui
        self.db = self.gui.current_db
        self.chain_name = chain_name
        self.chains_config = chains_config
        self._init_controls()

    def _init_controls(self):

        l = self.l = QVBoxLayout()
        self.setLayout(l)
        
        chain_groupbox = QGroupBox(_('Select chain'))
        l.addWidget(chain_groupbox)
        chain_groupbox_l = QVBoxLayout()
        chain_groupbox.setLayout(chain_groupbox_l)
        self.chain_combo = QComboBox()
        chain_groupbox_l.addWidget(self.chain_combo)
        
        opts_groupbox = QGroupBox(_('Options'))
        opts_groupbox_l = QVBoxLayout()
        opts_groupbox.setLayout(opts_groupbox_l)
        l.addWidget(opts_groupbox)
        
        vars_chk = self.vars_chk = QCheckBox(_('Pass variables from calling chain'))
        vars_chk.setChecked(False)
        opts_groupbox_l.addWidget(vars_chk)
        
        #all_chain_names = [chain_config['menuText'] for chain_config in cfg.get_chains_config()]
        all_chain_names = [chain_config['menuText'] for chain_config in self.chains_config if chain_config['menuText']]
        try:
            all_chain_names.remove(self.chain_name)
        except:
            pass
        self.chain_combo.addItems(all_chain_names)
        self.chain_combo.setCurrentIndex(-1)
        
        l.addStretch(1)

        self.setMinimumSize(300,300)

    def load_settings(self, settings):
        if settings:
            idx = self.chain_combo.findText(settings['chain_name'])
            self.chain_combo.setCurrentIndex(idx)
            self.vars_chk.setChecked(settings.get('pass_vars', False))

    def save_settings(self):
        settings = {}
        settings['chain_name'] = self.chain_combo.currentText()
        settings['pass_vars'] = self.vars_chk.isChecked()
        return settings

    def validate(self, settings):
        return validate(settings, chains_config=self.chains_config, parent_chain_name=self.chain_name)

class ChainCallerAction(ChainAction):

    name = 'Chain Caller'
    _is_builtin_ = True

    def run(self, gui, settings, chain):
        # Chain variables {
        add_to_vars = {}
        if settings.get('pass_vars'):
            add_to_vars = self.sanitize_chain_vars(chain)
        add_to_vars['_caller'] = chain.chain_name
        # }
        target_chain_config = cfg.get_chain_config(settings['chain_name'])
        target_chain = Chain(self.plugin_action, target_chain_config, show_progress=False,
                             chain_vars=add_to_vars, chain_caller=True)
        target_chain.run()
        del target_chain

    def run_as_job(self, settings):
        return False

    def validate(self, settings):
        return validate(settings, chains_config=[])

    def config_widget(self):
        return ChainCallerConfigWidget

    def sanitize_chain_vars(self, chain):
        sanitized_vars = {}
        for k, v in chain.chain_vars.items():
            if k in ['location']:
                continue
            elif k.startswith('_'):
                if k not in ['_event_name', '_event_args', '_variant_argument', '_book_vars']:
                    continue
            sanitized_vars[k] = v
        return sanitized_vars

    def run_as_job(self, settings):
        chain_name = settings['chain_name']
        chain_config = cfg.get_chain_config(chain_name)
        chain_settings = chain_config.get('chain_settings', {})
        chain_links = chain_settings.get('chain_links',[])
        if len(chain_links) == 0:
            return True
        for chain_link in chain_links:
            action_name = chain_link['action_name']
            action = self.plugin_action.actions[action_name]
            action_settings = chain_link.get('action_settings')
            if not action.run_as_job(action_settings):
                if DEBUG:
                    print(f'Action Chains: Cannot run as a job ({action_name})')
                return False
        return True
