#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import copy

from qt.core import (QApplication, Qt, QObject)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog

class ChainAction(QObject):

    # Actions must have a unique name attribute.
    name = 'Chain Action'
    
    # Set to True to enable scopes support
    support_scopes = False
    
    is_experimental = False

    '''
    This is the base class for all actions
    '''
    def __init__(self, plugin_action):
        '''
        All actions are intialized at startup
        The are re-initialized on library change, and on adding or modifying custom modules
        '''
        QObject.__init__(self)
        self.plugin_action = plugin_action

    def run(self, gui, settings, chain):
        '''
        This is the method that contain the logic of the action you want.
        It is called whenever a chain is activated.
        The settings is a dictionary with options configured for the specific
        action in the chain dialog.
        chain is the instance running the chain actions, it has info
        used by some actions (e.g. start time for each chain action)
        
        Actions that support scopes should call chain.scope().get_book_ids() to get the list
        of book_ids to act on.
        '''
        raise NotImplementedError

#    def shutting_down(self):
#        '''
#        Implement this if you want to do anything related to your action
#        when calibre shuts down (e.g cleaning)
#        '''
#        raise NotImplementedError

    def config_widget(self):
        '''
        If you want your action to have settings dialog, implement this method
        This should return a Qwidget (not dialog) with the following methods:
        [*] __init__(self, plugin_action)
        [*] save_settings(settings)
                This method is used to save the settings from the widget
                it should return a dictionary containing all the settings
        [*] load_settings(self, settings)
                This method is used to load the saved settings into the
                widget
        '''
        return None

    def validate(self, settings):
        '''
        Validate settings dict. This is called when you press the OK button in config dialog.
        changes are applied only if the this methods returns True.
        It is called also when verifying the chain validity on multiple occasions: startup,
        library change, chain dialog initialization .....
        If the setting in the dictionary are not valid, return a tuple
        of two strings (message, details), these will be displayed as a warning dialog to
        the user and the process will be aborted.
        '''
        return True

    def run_as_job(self, settings):
        '''
        The output of this function determines whether a chain containing
        the action can be run as a calibr job or not.
        '''
        return True

    def default_settings(self):
        '''
        default settings to be used if no settings are configured
        '''
        return {}

#    def on_modules_update(self):
#        '''
#        This method is called every time the modules editor is edited/updated
#        Implement only if you want to process custom functions related to your action
#        '''
#        raise NotImplementedError




