#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from collections import OrderedDict

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog
from calibre.utils.config import tweaks

from calibre_plugins.action_chains.actions.base import ChainAction

# builtins
from calibre_plugins.action_chains.actions.open_with import OpenWithAction
from calibre_plugins.action_chains.actions.single_field import SingleFieldAction
from calibre_plugins.action_chains.actions.search_replace import SearchReplaceAction
from calibre_plugins.action_chains.actions.calibre_actions import CalibreActions
from calibre_plugins.action_chains.actions.copy_to_clipboard import CopyToClipboardAction
from calibre_plugins.action_chains.actions.selection_modifier import SelectionModifierAction
from calibre_plugins.action_chains.actions.chain_variables import ChainVariablesAction
from calibre_plugins.action_chains.actions.formulas import Formulas
from calibre_plugins.action_chains.actions.template_search import TemplateSearchAction
from calibre_plugins.action_chains.actions.chain_caller import ChainCallerAction
from calibre_plugins.action_chains.actions.save_restore_format import SaveRestoreFormatAction
from calibre_plugins.action_chains.actions.code import CodeAction
from calibre_plugins.action_chains.actions.convert_books import ConvertBooksAction


#

try:
    load_translations()
except NameError:
    prints("ActionChains::actions/__int__.py - exception when loading translations")


def get_imported_actions(plugin_action):
    all_imported_actions = {}
    for plugin_name, imported_resources in plugin_action.imported_resources.items():
        imported_actions = imported_resources.get('actions', [])
        for imported_action in imported_actions:
            name = imported_action.name
            # must define a name attribute, must be set and not clash with builtin names
            # which can be imported into the module manager by custom actions
            if name in ['', 'Chain Action']:
                continue
            imported_action._source_plugin = plugin_name
            all_imported_actions[name] = imported_action          
    return all_imported_actions

def get_user_actions(plugin_action):
    user_actions = {}
    for cls in plugin_action.user_modules.get_classes(class_filters=[ChainAction]):
        name = cls.name
        # must define a name attribute, must be set and not clash with builtin names
        # which can be imported into the module manager by custom actions
        if name in ['', 'Chain Action']:
            continue
        user_actions[name] = cls            
    return user_actions

def get_all_actions(plugin_action):

    builtin_actions = OrderedDict()
    
    _builtin_actions = [
        OpenWithAction,
        SingleFieldAction,
        SearchReplaceAction,
        CalibreActions,
        CopyToClipboardAction,
        SelectionModifierAction,
        ChainVariablesAction,
        Formulas,
        TemplateSearchAction,
        SaveRestoreFormatAction,
        ChainCallerAction,
        CodeAction,
        ConvertBooksAction
    ]

    for action_cls in _builtin_actions:
        builtin_actions[action_cls.name] = action_cls

    _user_actions = get_user_actions(plugin_action)
    
    _imported_actions = get_imported_actions(plugin_action)

    all_actions = OrderedDict()
    user_actions = OrderedDict()
    imported_actions = OrderedDict()
    
    for action_name, action_cls in builtin_actions.items():
        action = action_cls(plugin_action)
        if action.is_experimental and not tweaks.get('action_chains_experimental', False):
            continue
        all_actions[action_name] = action

    # Note: imported actions can either be class or instantiated object
    for action_name, action_obj in _imported_actions.items():
        # dont override builtin actions
        if action_name in builtin_actions.keys():
            continue
        try:
            if isinstance(action_obj, ChainAction):
                action = action_obj
            elif issubclass(action_obj, ChainAction):
                action = action_obj(plugin_action)
            if action.is_experimental and not tweaks.get('action_chains_experimental', False):
                continue
            all_actions[action_name] = action
            imported_actions[action_name] = action
        except TypeError as e:
            # TypeError: issubclass() arg 1 must be a class
            import traceback
            if DEBUG:
                prints(f'Action Chains: Error intializing imported action: Un-reconized object: {action_obj}\n{traceback.format_exc()}')            
        except Exception as e:
            import traceback
            if DEBUG:
                prints(f'Action Chains: Error intializing imported action: {action_name}\n{traceback.format_exc()}')

    for action_name, action_cls in _user_actions.items():
        # dont override builtin actions
        if action_name in builtin_actions.keys():
            continue
        try:
            action = action_cls(plugin_action)
            if action.is_experimental and not tweaks.get('action_chains_experimental', False):
                continue
            all_actions[action_name] = action
            user_actions[action_name] = action
        except Exception as e:
            import traceback
            if DEBUG:
                prints(f'Action Chains: Error intializing user action: {action_name}\n{traceback.format_exc()}')

    return all_actions, builtin_actions, user_actions, imported_actions       
