#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from functools import partial
from collections import OrderedDict, defaultdict
import re
import traceback
import copy
import json
import operator
import inspect

from calibre import prints
from calibre.constants import DEBUG
from calibre.ebooks.metadata.book.base import Metadata
from calibre.ebooks.metadata.book.formatter import SafeFormat
from calibre.gui2 import error_dialog
from calibre.utils.formatter_functions import FormatterFunction, formatter_functions

from calibre_plugins.action_chains.common_utils import has_non_inherited_attribute
from calibre_plugins.action_chains.templates.functions.base import TemplateFunction
import calibre_plugins.action_chains.templates.functions

try:
    load_translations()
except NameError:
    prints("ActionsChain/templates/__init__.py - exception when loading translations")

TEMPLATE_PREFIX = 'TEMPLATE: '
TEMPLATE_ERROR = 'TEMPLATE_ERROR: '

BUILTIN_FUNCTIONS = [
    functions.SelectionCount,
    functions.ContainingFolder,
    functions.BookVars,
    functions.SetBookVars,
    functions.ListBookVars,
    functions.CategoryItems,
    functions.SanitizePath,
    functions.CoverPath,
    functions.LastModified,
    functions.BookField,
    functions.BookRawField
]

def get_imported_functions(plugin_action):
    all_imported_functions = {}
    for plugin_name, imported_resources in plugin_action.imported_resources.items():
        imported_functions = imported_resources.get('template_functions', [])
        for imported_function in imported_functions:
            name = imported_function.name
            # must define a name attribute, must be set and not clash with builtin names
            # which can be imported into the module manager by custom functions
            if name in ['', 'Chain function']:
                continue
            imported_function._source_plugin = plugin_name
            all_imported_functions[name] = imported_function          
    return all_imported_functions

def get_user_functions(plugin_action):
    user_functions = {}
    for cls in plugin_action.user_modules.get_classes(class_filters=[TemplateFunction]):
        name = cls.name
        # must define a name attribute
        if name in ['', 'no name provided']:
            continue
        user_functions[name] = cls            
    return user_functions


def get_action_functions(plugin_action):
    method_name = 'on_templates_update'
    action_functions = {}
    for action_name, action in plugin_action.actions.items():
        method = getattr(action, method_name, None)
        if method:
            try:
                functions = method()
                action_functions.update(functions)
            except:
                if DEBUG:
                    prints(f'Action Chains: Error running ({method_name}) for action ({action})')
    return action_functions

def get_template_functions(plugin_action):
    all_functions = OrderedDict()

    builtin_functions = OrderedDict()
    for cls in BUILTIN_FUNCTIONS:
        builtin_functions[cls.name] = cls

    imported_functions = get_imported_functions(plugin_action)

    user_functions = get_user_functions(plugin_action)
    
    action_functions = get_action_functions(plugin_action)
    
    calibre_funcs = formatter_functions().get_functions()

    # Note: imported functions can either be class or instantiated object
    for function_name, function_obj in imported_functions.items():
        # dont override builtin functions
        if function_name in builtin_functions.keys():
            continue
        try:
            if isinstance(function_obj, (TemplateFunction, FormatterFunction)):
                function = function_obj
            elif issubclass(function_obj, TemplateFunction):
                function = function_obj(plugin_action)
            elif issubclass(function_obj, FormatterFunction):
                function = function_obj()
            all_functions[function_name] = function
        except TypeError as e:
            # TypeError: issubclass() arg 1 must be a class
            import traceback
            if DEBUG:
                prints(f'Action Chains: Error intializing imported function: Un-reconized object: {function_obj}\n{traceback.format_exc()}')            
        except Exception as e:
            import traceback
            if DEBUG:
                prints(f'Action Chains: Error intializing imported function: {function_name}\n{traceback.format_exc()}')

    for function_name, cls in user_functions.items():
        try:
            function = cls(plugin_action)
            all_functions[function_name] = function
        except Exception as e:
            import traceback
            if DEBUG:
                prints(f'Action Chains: Error intializing user function: {function_name}\n{traceback.format_exc()}')

    for function_name, function in action_functions.items():
        try:
            all_functions[function_name] = function
        except Exception as e:
            import traceback
            if DEBUG:
                prints(f'Action Chains: Error intializing action function: {function_name}\n{traceback.format_exc()}')

    for name, func in calibre_funcs.items():
        all_functions[name] = func

    # we keep this last in case we want to override some calibre formatter functions
    for function_name, cls in builtin_functions.items():
        function = cls(plugin_action)
        all_functions[function_name] = function

    return all_functions, builtin_functions, user_functions, action_functions, imported_functions

###################################################

def dummy_metadata(db):
    fm = db.new_api.field_metadata
    mi = Metadata(_('Title'), [_('Author')])
    mi.author_sort = _('Author Sort')
    mi.series = ngettext('Series', 'Series', 1)
    mi.series_index = 3
    mi.rating = 4.0
    mi.tags = [_('Tag 1'), _('Tag 2')]
    mi.languages = ['eng']
    mi.id = 1
    mi.set_all_user_metadata(fm.custom_field_metadata())
    for col in mi.get_all_user_metadata(False):
        mi.set(col, (col,), 0)
    return mi

def get_metadata_object(gui):
    db = gui.current_db
    try:
        current_row = gui.library_view.currentIndex()
        book_id = gui.library_view.model().id(current_row)
        mi = db.new_api.get_proxy_metadata(book_id)
    except Exception as e:
        if DEBUG:
            prints('Action Chains: get_metadata_object: exception trying to get mi from current row')
        try:
            book_id = list(db.all_ids())[0]
            mi = db.new_api.get_proxy_metadata(book_id)
        except:
            mi = dummy_metadata(db)
    return mi

def check_template(template, plugin_action, print_error=True, template_functions=None):
    gui = plugin_action.gui
    db = gui.current_db
    if not template_functions:
        template_functions = plugin_action.template_functions
    error_msgs = [
        TEMPLATE_ERROR,
        'unknown function',
        'unknown identifier',
        'unknown field',
        'assign requires the first parameter be an id',
        'missing closing parenthesis',
        'incorrect number of arguments for function',
        'expression is not function or constant'
    ]
    mi = get_metadata_object(gui)

    output = get_template_output(template, mi, TEMPLATE_ERROR, mi, template_functions=template_functions)
    for msg in error_msgs:
        if output.lower().find(msg.lower()) != -1:
            error = _(f'Running the template: {template} returned an error:\n{output.replace(TEMPLATE_ERROR, "")}')
            if print_error:
                return error_dialog(gui, _('Template Error'), error, show=True)
            return _('Template Error'), error
    return True


try:
    from calibre.utils.formatter import PythonTemplateContext
except:
    PythonTemplateContext = object

class ACTemplateContext(PythonTemplateContext):
    pass

def get_template_output(template, kwargs, TEMPLATE_ERROR, book, global_vars={},
                        template_functions=None, context_attrs={}):
    if not template_functions:
        template_functions = get_template_functions[0]
    try:
        # calibre >= 6.7.0. support for python template context object
        context = ACTemplateContext()
        context.set_values(**context_attrs)
        output = SafeFormat().safe_format(template, kwargs, TEMPLATE_ERROR, book,
                                          global_vars=global_vars,
                                          template_functions=template_functions,
                                          python_context_object=context)
    except:
        output = SafeFormat().safe_format(template, kwargs, TEMPLATE_ERROR, book,
                                          global_vars=global_vars,
                                          template_functions=template_functions)
    return output
