#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from qt.core import (QApplication, Qt, QVBoxLayout, QGroupBox, QCheckBox, QRadioButton)

from calibre import prints
from calibre.constants import DEBUG
from calibre.utils.search_query_parser import ParseException

from calibre_plugins.action_chains.scopes.base import ActionScope
from calibre_plugins.action_chains.templates import check_template, get_metadata_object
from calibre_plugins.action_chains.templates.dialogs import TemplateBox

try:
    load_translations()
except NameError:
    prints("ActionChains::scopes/template_scopes.py - exception when loading translations")

class TemplateScopeConfigWidget(TemplateBox):

    def __init__(self, plugin_action):
        self.plugin_action = plugin_action
        self.gui = plugin_action.gui
        self.db = self.gui.current_db
        placeholder_text = _('Depending on the option selected below, the template entered here should return a comma separated list '
                             'of book_ids, or a calibre search')
        TemplateBox.__init__(
            self,
            self.gui,
            plugin_action,
            template_text='',
            placeholder_text=placeholder_text,
            show_buttonbox=False
        )
        vl_chk = self.vl_chk = QCheckBox(_('Exclude books not in current virtual libarary'))
        vl_chk.setChecked(False)
        self.user_layout_1.addWidget(vl_chk)
        
        options_groupbox = QGroupBox(_('Options'))
        self.user_layout_2.addWidget(options_groupbox)
        options_groupbox_layout = QVBoxLayout()
        options_groupbox.setLayout(options_groupbox_layout)
        ids_opt = self.ids_opt = QRadioButton(_('Template output is a list of book ids'))
        options_groupbox_layout.addWidget(ids_opt)
        ids_opt.setChecked(True)
        search_opt = self.search_opt = QRadioButton(_('Template output is a calibre search'))
        options_groupbox_layout.addWidget(search_opt)

    def load_settings(self, settings):
        if settings:
            template = settings['template']
            self.textbox.insertPlainText(template)
            self.vl_chk.setChecked(settings.get('vl'))
            if settings.get('opt') == 'search':
                self.search_opt.setChecked(True)

    def save_settings(self):
        settings = {}
        settings['template'] = str(self.textbox.toPlainText()).rstrip()
        settings['vl'] = self.vl_chk.isChecked()
        if self.ids_opt.isChecked():
            settings['opt'] = 'ids'
        elif self.search_opt.isChecked():
            settings['opt'] = 'search'
        return settings


class TemplateScope(ActionScope):

    name = 'Template Scopes'

    def get_book_ids(self, gui, settings, chain):
        db = gui.current_db
        template = settings.get('template', '')
        template_output = chain.evaluate_template(template, book_id=None)
        if settings.get('opt') == 'ids':
            try:
                book_ids = [int(x.strip()) for x in template_output.split(',')]
            except:
                book_ids = []
        elif settings.get('opt') == 'search':
            try:
                book_ids = db.data.search_getting_ids(template_output, '', use_virtual_library=False)
            except ParseException:
                book_ids = []

        # discard book_ids not present in library
        book_ids = list(set(gui.current_db.all_ids()).intersection(set(book_ids)))

        if settings.get('vl'):
            vl_ids = self.get_current_restriction_book_ids()
            book_ids = list(set(vl_ids).intersection(set(book_ids)))
        return book_ids

    def validate(self, settings):
        gui = self.plugin_action.gui
        db = gui.current_db
        if not settings:
            return (_('Scope errors'), _('You must configure this scope before running it'))
        mi = get_metadata_object(gui)
        is_template_valid = check_template(settings['template'], self.plugin_action, print_error=False)
        if is_template_valid is not True:
            return is_template_valid
        return True

    def config_widget(self):
        return TemplateScopeConfigWidget
