#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import copy

from qt.core import (QApplication, Qt)

from calibre import prints
from calibre.constants import DEBUG
from calibre.utils.date import now, as_local_time

import calibre_plugins.action_chains.config as cfg
from calibre_plugins.action_chains.chains import Chain
from calibre_plugins.action_chains.gui.events_dialogs import EventTimer
from calibre_plugins.action_chains.common_utils import call_method

try:
    load_translations()
except NameError:
    prints("ActionChains::events/event_tools.py - exception when loading translations")

class EventLoop(object):

    def __init__(self, plugin_action, event, event_args, qt_processEvents=True, is_refresh_gui=True):
        self.plugin_action = plugin_action
        self.gui = plugin_action.gui
        self.db = self.gui.current_db
        self.event_args = event_args
        self.qt_processEvents = qt_processEvents
        self.is_refresh_gui = is_refresh_gui
        self.event = event
        self.event_name = event.name
        # copy as to not overwrite mutable data
        self.event_config = copy.deepcopy(cfg.get_event_config(self.event_name))
        if self.event_config:
            self.event_members = self.event_config.get('event_settings', {}).get('event_members', [])
        self.start_time = now()
        self.step_length = 1
        self.current_step = -1
        self.last_step = None

    def _run_loop(self):
        
        # if event has no entry in event manager, stop
        if not self.event_config:
            return

        # if event is unchecked in even manager, stop
        if not self.event_config['active']:
            if DEBUG:
                prints(f'Action Chain: Event ({self.event_name}) entry is unchecked, not proceeding')
            return
        
        # event_loop should not run in response to any events while other events or chains are running
        if (self.plugin_action.chainStack or self.plugin_action.eventRunnig):
            return

        # validate member chains (only if the user ticks the option), abort event if a chain has errors
        res = self.validate()
        if not res:
            if DEBUG:
                prints(f'Action Chain: Event ({self.event_name}) aborting because errors in member chain(s)')
            return

        res = self.start_countdown()
        if not res:
            if DEBUG:
                prints(f'Action Chain: Event ({self.event_name}) aborted by user')
            return

        QApplication.setOverrideCursor(Qt.WaitCursor)
        try:
            # block event manager
            self.plugin_action.eventRunnig = True
            #
            
            self.start_time = now()
            
            # event options
            
            event_opts = self.event_config.get('event_settings', {}).get('event_options', {})
            self.event.pre_chains_event_actions(self.event_args, event_opts)
            
            if DEBUG:
                prints(f'Action Chains: Starting event: {self.event_name}')

            while True:                
                self.last_step = self.current_step
                self.current_step += self.step_length if self.step_length > 0 else 1
                # now that we read the step lenght that might have been modified by the previous action, restore it back
                self.step_length = 1
                if self.current_step > len(self.event_members) - 1:
                    break
                #
                event_member = self.event_members[self.current_step]
                # store start time for each chain link, used by some actions to change selections based on last_modified date.
                event_member['start_time'] = now()
                chain_name = event_member['chain_name']
                chain_config = cfg.get_chain_config(chain_name)

                try:
                    if chain_config:
                        chain = Chain(self.plugin_action, chain_config, qt_processEvents=self.qt_processEvents, event=self, event_args=self.event_args)
                        chain.run(check_conditions=True)
                        del chain
                    else:
                        if DEBUG:
                            prints(f'Action Chains: Chain ({chain_name}) is not available')

                    QApplication.processEvents()
                except Chain.UserInterrupt:
                    if DEBUG:
                        prints(f'Action Chains: Aborting chain ({self.chain_name}) in event ({self.event_name})')
                    break
                except Exception as e:
                    if DEBUG:
                        prints(f'Action Chains: chain "{chain_name}" terminated with exception: {e}')
                    import traceback
                    traceback.print_exc()
                    raise e

            if self.is_refresh_gui:
                self.refresh_gui()
            end = now()
            if DEBUG:
                prints(f'Action Chains: event ({self.event_name}) finished in: {end - self.start_time}')
        finally:
            QApplication.restoreOverrideCursor()
            self.plugin_action.eventRunnig = False

    def refresh_gui(self):
        # refresh only if at least one chain members have refresh_gui set to true
        refresh = False
        db_last_modified = as_local_time(self.gui.current_db.last_modified())
        db_event_modified = db_last_modified > self.start_time
        if DEBUG:
            prints(f'Action Chains: Is db modified by event: {db_event_modified}')
        
        if db_event_modified:
            for chain_config in self.event_members:
                chain_refresh = chain_config.get('refresh_gui')
                if chain_refresh:
                    refresh = True
                    break
        if refresh:
            lm_map = {book_id: self.db.new_api.field_for('last_modified', book_id) for book_id in self.db.all_ids() }
            refresh_books = [ book_id for book_id, lm in lm_map.items() if (lm and lm > self.start_time ) ]
            cr = self.gui.library_view.currentIndex().row()
            self.gui.library_view.model().refresh_ids(refresh_books, cr)
            self.gui.tags_view.recount()
            #self.gui.current_view().resort()
            
    def start_countdown(self):
        countdown_settings = self.event_config.get('event_settings', {}).get('countdown_settings', {})
        enabled = countdown_settings.get('enabled')
        message = countdown_settings.get('message')
        duration = countdown_settings.get('duration')
        if enabled:
            d = EventTimer(self.gui, ceiling=duration, message=message)
            if d.exec_() == d.Rejected:
                return False
        return True

    def validate(self):
        '''
        if the user settings validate is on, validate all memeber chains before running
        '''
        do_validate = self.event_config.get('event_settings', {}).get('validate')
        if do_validate:
            # init_cache for template functions, we do it here instead of in the chain.validate
            # to avoid flusing it after each chain is checked
            call_method(self.plugin_action.template_functions, 'init_cache')
            try:
                for event_member in self.event_members:
                    chain_name = event_member['chain_name']
                    chain_config = cfg.get_chain_config(chain_name)
                    if not chain_config:
                        if DEBUG:
                            prints(f'Action Chains: validating chain ({chain_name}) failed: chain is not available')
                        return False
                    chain = Chain(self.plugin_action, chain_config)
                    is_true = chain.validate(use_cache=False)
                    if is_true is not True:
                        msg, details = is_true
                        if DEBUG:
                            prints(f'Action Chains: validating chain ({chain_name}) failed: {msg}, {details}')
                        return False
            finally:
                call_method(self.plugin_action.template_functions, 'flush_cache')
        return True

