#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from qt.core import (QApplication, Qt, QObject)

from calibre import prints
from calibre.constants import DEBUG

from calibre_plugins.action_chains.events.event_tools import EventLoop
import calibre_plugins.action_chains.config as cfg

try:
    load_translations()
except NameError:
    prints("ActionChains::events/base.py - exception when loading translations")

class ChainEvent(QObject):

    name = 'Chain Event'
    support_variants = False

    def __init__(self, plugin_action):
        QObject.__init__(self)
        self.plugin_action = plugin_action
        self.gui = plugin_action.gui
        try:
            signal = self.get_event_signal()
            signal.connect(self.run)
        except Exception as e:
            if DEBUG:
                print(f'Action Chains: failed to initialize Chain Event: {self.name}')
            raise e

    def get_event_signal(self):
        '''
        This method should return the signal for the event
        '''
        raise NotImplementedError

    def pre_chains_event_actions(self, event_args, event_opts):
        '''
        Implement in sub-classes if any of the event options needs it.
        '''
        pass

    def config_widget(self):
        '''
        If you want your action to have settings dialog, implement this method
        This should return a Qwidget (not dialog) with the following methods:
        [*] __init__(self, plugin_action)
        [*] save_settings(settings)
                This method is used to save the settings from the widget
                it should return a dictionary containing all the settings
        [*] load_settings(self, settings)
                This method is used to load the saved settings into the
                widget
        '''
        return None

    def validate(self, settings):
        '''
        Validate settings dict. This is called when you press the OK button in config dialog.
        changes are applied only if the this methods returns True.
        If the setting in the dictionary are not valid, return a tuple
        of two strings (message, details), these will be displayed as a warning dialog to
        the user and the process will be aborted.
        '''
        return True

    def default_settings(self):
        '''
        default settings to be used if no settings are configured
        '''
        return {}

    def get_options(self):
        '''
        Get event specific options
        '''
        event_config = cfg.get_event_config(self.name)
        if event_config:
            return event_config.get('event_settings', {}).get('event_options', {})
        else:
            return {}

    def run(self, *args):
        event_args = args
        event = EventLoop(self.plugin_action, self, event_args)
        event._run_loop()
        del event
