#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__   = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import copy

from calibre.utils.config import JSONConfig

KEY_SCHEMA_VERSION = 'schemaVersion'
DEFAULT_SCHEMA_VERSION = 1.9
STORE_MENUS_NAME = 'ActionsChainMenus'
KEY_MENUS = 'Menus'
STORE_EVENTS_NAME = 'events'
EVENT_ENTRIES = 'eventEntries'
KEY_EVENT_VARIANTS = 'eventVariants'
#TIMER_EVENT_TIMEOUT = 'timerEventTimeout'
KEY_GPREFS = 'gprefs'
KEY_CHAINS_TABLE_STATE = 'chainsTableState'
KEY_ACTIONS_TABLE_STATE = 'actionsTableState'
KEY_EVENTS_TABLE_STATE = 'eventsTableState'
KEY_EVENT_MEMBERS_TABLE_STATE = 'eventMembersTableState'
KEY_VARS_TABLE_STATE = 'varsTableState'
KEY_EVENT_VARIANTS_TABLE_STATE = 'eventVariantsTableState'
DEFAULT_VALUES = {
    STORE_MENUS_NAME: {
        KEY_MENUS: []
    },
    STORE_EVENTS_NAME: {
        EVENT_ENTRIES: []
    },
    KEY_EVENT_VARIANTS: [],
    KEY_GPREFS: {
        KEY_CHAINS_TABLE_STATE: None,
        KEY_ACTIONS_TABLE_STATE: None,
        KEY_EVENTS_TABLE_STATE: None,
        KEY_EVENT_MEMBERS_TABLE_STATE: None,
        KEY_VARS_TABLE_STATE: None,
        KEY_EVENT_VARIANTS_TABLE_STATE: None
    }
}

# This is where all preferences for this plugin will be stored
plugin_prefs = JSONConfig('plugins/Action Chains')

## Set defaults
#plugin_prefs.defaults[STORE_MENUS_NAME] = DEFAULT_VALUES

PLUGIN_ICONS = [
    'action_chains.png',
    'image_add.png',
    'import.png',
    'export.png',
    'condition.png',
    'no_condition.png',
    'scope.png',
    'no_scope.png',
    'multiple.png'
]

def get_missing_values_from_defaults(default_settings, settings):
    '''add keys present in default_settings and absent in setting'''
    for k, default_value in default_settings.items():
        try:
            setting_value = settings[k]
            if isinstance(default_value, dict):
                get_missing_values_from_defaults(default_value, setting_value)
        except KeyError:
            settings[k] = copy.deepcopy(default_value)

def migrate_plugin_prefs_if_required(plugin_prefs, commit=True):
    schema_version = plugin_prefs.get(KEY_SCHEMA_VERSION, 0)
    if schema_version == DEFAULT_SCHEMA_VERSION:
        return

    # We have changes to be made - mark schema as updated
    plugin_prefs[KEY_SCHEMA_VERSION] = DEFAULT_SCHEMA_VERSION

    # Any migration code in future will exist in here.
    if schema_version < 1.1:
        pass

    if schema_version < 1.2:
        # remove KEY_CHAINS_TABLE_STATE since new columns added would miss it up
        plugin_prefs[KEY_CHAINS_TABLE_STATE] = None
        
        c = plugin_prefs[STORE_MENUS_NAME]
        chains = c.get(KEY_MENUS, [])
        for chain in chains:
            chain_name = chain.get('menuText','')
            chain_settings = chain.get('chain_settings', {})
            chain_links = chain_settings.get('chain_links', [])
            for chain_link in chain_links:
                action_name = chain_link['action_name']
                if action_name == 'Calibre Actions':
                    action_settings = chain_link['action_settings']
                    #
                    # This segment is inserted to deal with exported zip archives
                    # that did not have a schema version (versions prior to 1.4.0)
                    # so we have to detect it here to avoid missing up settings
                    if action_settings.get('disable_busy_cursor') is not None:
                        continue
                    #
                    new_settings = {
                        'disable_busy_cursor': False,
                        'selection': action_settings
                    }
                    chain_link['action_settings'] = new_settings
                    
    if schema_version < 1.4:
        plugin_prefs[KEY_GPREFS] = {
            KEY_EVENTS_TABLE_STATE: plugin_prefs.get(KEY_EVENTS_TABLE_STATE),
            KEY_EVENT_MEMBERS_TABLE_STATE: plugin_prefs.get(KEY_EVENT_MEMBERS_TABLE_STATE),
            KEY_VARS_TABLE_STATE: plugin_prefs.get(KEY_VARS_TABLE_STATE)
        }
        for key in [
            KEY_EVENTS_TABLE_STATE,
            KEY_EVENT_MEMBERS_TABLE_STATE,
            'configTableState',
            'chainTableState'
            ]:
            try:
                del plugin_prefs[key]
            except:
                pass

    if schema_version < 1.5:
        del plugin_prefs[KEY_GPREFS]
        for key in [
            KEY_VARS_TABLE_STATE,
            KEY_EVENTS_TABLE_STATE
            ]:
            try:
                del plugin_prefs[key]
            except:
                pass

    if schema_version < 1.6:
        
        c = plugin_prefs[STORE_MENUS_NAME]
        chains = c.get(KEY_MENUS, [])
        for chain in chains:
            chain_name = chain.get('menuText','')
            chain_settings = chain.get('chain_settings', {})
            chain_links = chain_settings.get('chain_links', [])
            for chain_link in chain_links:
                action_name = chain_link['action_name']
                if action_name.lower() == 'search and replace':
                    chain_link['action_name'] = 'Search And Replace'

    if schema_version < 1.7:
        
        c = plugin_prefs[STORE_EVENTS_NAME]
        events = c.get(EVENT_ENTRIES, [])
        for event in events:
            event_name = event.get('event_name','')
            if event_name == 'Library View Double Clicked':
                event['event_name'] = 'Book List Double Clicked'

    if schema_version < 1.8:
        
        try:
            del plugin_prefs['timerEventTimeout']
        except:
            pass

    if schema_version < 1.9:
        c = plugin_prefs[STORE_MENUS_NAME]
        chains = c.get(KEY_MENUS, [])
        for chain in chains:
            chain_name = chain.get('menuText','')
            chain_settings = chain.get('chain_settings', {})
            chain_links = chain_settings.get('chain_links', [])
            for chain_link in chain_links:
                action_name = chain_link['action_name']
                if action_name == 'Open With':
                    action_settings = chain_link['action_settings']
                    use_default = action_settings.get('use_default_app', False)
                    if not use_default:
                        args = action_settings.get('args', '').replace(',', ' ')
                        action_settings['args'] = args
                        chain_link['action_settings'] = action_settings
        
    # Update: add defaults for new keys {
    get_missing_values_from_defaults(DEFAULT_VALUES, plugin_prefs)
    if commit:
        plugin_prefs.commit()

if not plugin_prefs:
    get_missing_values_from_defaults(DEFAULT_VALUES, plugin_prefs)
migrate_plugin_prefs_if_required(plugin_prefs)

def get_chains_config(exclude_separators=True):
    c = plugin_prefs[STORE_MENUS_NAME]
    chains = c.get(KEY_MENUS, [])
    if exclude_separators:
        chains = [chain for chain in chains if chain.get('menuText')]
    return chains

def get_chain_config(chain_name, chains_config=[]):
    if not chains_config:
        chains_config = get_chains_config()
    for chain_config in chains_config:
        name = chain_config['menuText']
        if name == chain_name:
            return chain_config
    return {}

def chains_have_conditions():
    c = []
    for chain in get_chains_config():
        c = [chain for chain in get_chains_config() if chain.get('condition_settings', {}).get('affect_menu')]
    return len(c) > 0

def get_events_config():
    c = plugin_prefs[STORE_EVENTS_NAME]
    events = c.get(EVENT_ENTRIES, [])
    return events

def get_event_config(event_name):
    events = get_events_config()
    for event in events:
        name = event['event_name']
        if name == event_name:
            return event
    return {}

def get_event_variants(event_name):
    variants = []
    for variant_dict in plugin_prefs[KEY_EVENT_VARIANTS]:
        variant_name = variant_dict['variant_name']
        parent_event = variant_dict['parent_event']
        if parent_event == event_name:
            variants.append(variant_name)
    return variants
    
