#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__   = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from collections import defaultdict
from functools import partial
import os
import copy
import importlib

from qt.core import Qt, QApplication, QMenu, QToolButton, QUrl, pyqtSignal, QTimer

from calibre import prints
from calibre.constants import iswindows, isosx, DEBUG
from calibre.constants import numeric_version as calibre_version
from calibre.gui2 import open_url, error_dialog
from calibre.gui2.actions import InterfaceAction
from calibre.utils.config import config_dir
from calibre.utils.date import now
from calibre.utils.config import tweaks
from calibre.customize.ui import initialized_plugins, builtin_names

import calibre_plugins.action_chains.config as cfg
from calibre_plugins.action_chains.common_utils import (set_plugin_icon_resources, get_icon, KeyboardConfigDialog,
                                        create_menu_item, create_menu_action_unique, call_method)
from calibre_plugins.action_chains.actions import get_all_actions
from calibre_plugins.action_chains.events import get_all_events
from calibre_plugins.action_chains.scopes import get_all_scopes
from calibre_plugins.action_chains.chains import Chain
from calibre_plugins.action_chains.modules import ManageModulesDialog, UserModules
from calibre_plugins.action_chains.gui.events_dialogs import EventsDialog
from calibre_plugins.action_chains.gui import IconCache
from calibre_plugins.action_chains.gui.chains_dialog import ChainsDialog
from calibre_plugins.action_chains.templates import get_metadata_object, get_template_functions, get_template_output, TEMPLATE_ERROR
from calibre_plugins.action_chains.templates.functions import ChainVariant

try:
    load_translations()
except NameError:
    prints("ActionChains::action.py - exception when loading translations")


def get_plugin_resources(plugin):
    from calibre_plugins.action_chains.actions.base import ChainAction
    from calibre_plugins.action_chains.events.base import ChainEvent
    from calibre_plugins.action_chains.scopes.base import ActionScope
    from calibre_plugins.action_chains.templates.functions.base import TemplateFunction
    res = {}
    try:
        main = importlib.import_module(plugin.__class__.__module__+'.action_chains')
    except ModuleNotFoundError:
        pass
    except ImportError:
        import traceback
        traceback.print_exc()
    else:
        actions = []
        events = []
        scopes = []
        template_functions = []
        for k,v in vars(main).items():
            if isinstance(v, type) and issubclass(v, ChainAction):
                actions.append(v)
            if isinstance(v, type) and issubclass(v, ChainEvent):
                events.append(v)
            if isinstance(v, type) and issubclass(v, ActionScope):
                scopes.append(v)
            if isinstance(v, type) and issubclass(v, TemplateFunction):
                template_functions.append(v)
            if k == 'on_modules_update' and callable(v):
                res['on_modules_update'] = v
        if actions:
            res['actions'] = actions
        if events:
            res['events'] = events
        if scopes:
            res['scopes'] = scopes
        if template_functions:
            res['template_functions'] = template_functions
    return res

class ChainInterfaceAction(InterfaceAction):

    name = 'Action Chains'
    action_spec = ('Action Chains', None, None, None)
    popup_type = QToolButton.InstantPopup
    action_type = 'current'
    dont_add_to = frozenset(['context-menu-device'])

    plugin_initialization_complete = pyqtSignal()
    plugin_library_changed = pyqtSignal()
    plugin_location_selected = pyqtSignal()
    plugin_gui_layout_complete = pyqtSignal()
    plugin_shutting_down = pyqtSignal()
    plugin_modules_updated = pyqtSignal()
    plugin_restart_required = pyqtSignal(object, object)
    plugin_library_about_to_change = pyqtSignal(object, object)

    def genesis(self):
        self.is_library_selected = True
        self.loc = 'library'
        self.menu = QMenu(self.gui)

        # Read the plugin icons and store for potential sharing with the config widget
        icon_names = ['images/'+i for i in cfg.PLUGIN_ICONS]
        icon_resources = self.load_resources(icon_names)
        set_plugin_icon_resources(self.name, icon_resources)


        # Assign our menu to this action and an icon
        self.qaction.setMenu(self.menu)
        self.qaction.setIcon(get_icon('images/'+cfg.PLUGIN_ICONS[0]))
        # Setup hooks so that we only enable the relevant submenus for available formats for the selection.
        self.menu.aboutToShow.connect(self.about_to_show_menu)
        self.menu.aboutToHide.connect(self.about_to_hide_menu)
        
    def initialization_complete(self):
        # we implement here to have access to current_db
        # if we try this in genesis() we get the error:
        # AttributeError: 'Main' object has no attribute 'current_db'
        #
        self.api_version = 1.0
        self.book_vars = defaultdict(dict)
        # these variables are used to block events when chains and events are running
        self.chainStack = []
        self.eventRunnig = False
        # counter for how many times a chain has run
        self.chain_iterations = defaultdict(lambda: 0)
        
        self.menu_actions = []
        #self.rebuild_menus()
        self.resources_dir = os.path.join(config_dir, 'resources/images')
        if iswindows:
            self.resources_dir = os.path.normpath(self.resources_dir)
        self.icon_cache = IconCache(self)
        # Use QTimer to defeer this method after all other plugins run their initialization_complete()
        QTimer.singleShot(0, self.post_initialization)

    def post_initialization(self):
        '''
        Operations that should happen after all plugins are initialized should be here.
        '''
        # Get imported resources first before calling on_modules_update()
        self.imported_resources = self._get_plugins_resources()
        self.on_modules_update()
        # rebuild_menus should be called only after on_modules_update (required for chain variations)
        self.rebuild_menus()
        self.plugin_initialization_complete.emit()

    def library_changed(self, db):
        # Reset actions
        # Re-import resources because instance can be re-initialized on library_changed
        # to get rid of stale references like db
        self.imported_resources = self._get_plugins_resources()
        self.on_modules_update()
        self.rebuild_menus()
        QTimer.singleShot(0, self.plugin_library_changed.emit)

    def gui_layout_complete(self):
        QTimer.singleShot(0, self.plugin_gui_layout_complete.emit)

    def location_selected(self, loc):
        self.is_library_selected = loc == 'library'
        self.loc = loc        
        QTimer.singleShot(0, self.plugin_location_selected.emit)

    def shutting_down(self):
        # for actions that implement the shutdown method, run it
        self.call_actions_method('shutting_down')
        self.plugin_shutting_down.emit()

    def library_about_to_change(self, olddb, db):
        self.call_actions_method('library_about_to_change', olddb, db)
        self.plugin_library_about_to_change.emit(olddb, db)

    def on_modules_update(self):
        self.user_modules = UserModules()
        self.actions, self.builtin_actions, self.user_actions, self.imported_actions = get_all_actions(self)
        self.events, self.builtin_events, self.user_events, self.imported_events = get_all_events(self)
        self.update_template_functions()
        self.scopes, self.builtin_scopes, self.user_scopes, self.imported_scopes = get_all_scopes(self)
        self.call_actions_method('on_modules_update', self.user_modules)
        self._call_plugins_on_modules_update(self.user_modules)
        self.plugin_modules_updated.emit()

    def update_template_functions(self):
        self.all_template_functions, self.builtin_template_functions, self.user_template_function, self.action_functions, self.imported_template_functions = get_template_functions(self)

    @property
    def template_functions(self):
        self.update_template_functions()
        return self.all_template_functions

    def _get_plugins_resources(self):
        '''
        Look in plugins for action chains resources (e.g. actions, events, scopes)
        Resources should be defined in action_chains.py module in the root of the plugin.
        resources will be a dictionary as follows
        {
            'actions': [],
            'events': [],
            'template_functions: []
        }
        '''
        if DEBUG:
            prints('Action chains: _get_plugins_resources(): start')
        all_resources = {}

        # Collect resources from plugins
        for plugin in initialized_plugins():
            if plugin.name in builtin_names:
                pass
            else:
                try:
                    resources = get_plugin_resources(plugin)
                except:
                    resources = {}
                    import traceback
                    traceback.print_exc()
                if resources:
                    all_resources[plugin.name] = resources

        if DEBUG:
            prints('Action Chains: _get_plugins_resources(): finished')
        return all_resources

    def _call_plugins_on_modules_update(self, user_modules):
        '''
        This method makes the module editor available for plugins that
        implement on_modules_update(). It passes user_modules instance
        which allows plugins to access different objects and classes
        defined in the module editor.
        '''
        if DEBUG:
            prints('Action chains: _call_plugins_on_modules_update(): start')
        for plugin_name, resources in self.imported_resources.items():
            if plugin_name == 'Action Chains': continue
            on_modules_update = resources.get('on_modules_update')
            if on_modules_update:
                try:
                    if DEBUG:
                        prints(f'Action Chains: Running on_modules_update() for plugin: {plugin_name}')
                    on_modules_update(user_modules)
                except Exception as e:
                    if DEBUG:
                        prints(f'Action Chains: Error running on_modules_update from plugin: {plugin_name}')
                        import traceback
                        print(traceback.format_exc())
        if DEBUG:
            prints('Action Chains: _call_plugins_on_modules_update(): finished')

    def rebuild_menus(self):
        c = cfg.plugin_prefs[cfg.STORE_MENUS_NAME]
        chains_config = c.get(cfg.KEY_MENUS, [])
        m = self.menu
        m.clear()
        self.chain_menu = []
        sub_menus = {}
        
        for action in self.menu_actions:
            self.gui.keyboard.unregister_shortcut(action.calibre_shortcut_unique_name)
            # starting in calibre 2.10.0, actions are registers at
            # the top gui level for OSX' benefit.
            if calibre_version >= (2,10,0):
                self.gui.removeAction(action)
        self.menu_actions = []

        for chain_config in chains_config:
            active = chain_config['active']
            if active:
                menu_text = chain_config['menuText']
                sub_menu_text = chain_config['subMenu']
                image_name = chain_config['image']
                self.create_menu_item_ex(m, sub_menus, menu_text, sub_menu_text,
                                         image_name, chain_config)

        m.addSeparator()
        create_menu_action_unique(self, m, _('&Event Manager')+'...', 'auto-reload.png',
                                  triggered=self.event_manager)

        create_menu_action_unique(self, m, _('&Manage Modules')+'...', 'code.png',
                                  triggered=self.manage_modules)

        create_menu_action_unique(self, m, _('&Add/Modify Chains')+'...', 'images/action_chains.png',
                                  triggered=self.add_chains)
#        m.addSeparator()
#        create_menu_action_unique(self, m, _('&Customize plugin')+'...', 'config.png',
#                                  shortcut=False, triggered=self.show_configuration)
        
        # Hide submenus that contain certain keywords
        for path, submenu in sub_menus.items():
            name = path.split(':::')[-1].lower().strip()
            if name in ['invisible']:
                submenu.menuAction().setVisible(False)
        
        self.gui.keyboard.finalize()

    def create_menu_item_ex(self, m, sub_menus, menu_text, sub_menu_text,
                            image_name, chain_config):
        parent_menu = m
        if sub_menu_text:
            parent_menu = self.create_sub_menus(parent_menu, sub_menus, sub_menu_text, image_name)

        if not menu_text:
            ac = parent_menu.addSeparator()
        else:
            ac = self.create_all_menu_actions_for_chain(chain_config, parent_menu, sub_menus, menu_text, image_name)
        return ac

    def create_sub_menus(self, parent_menu, sub_menus, sub_menu_text, image_name):
        # Create the sub-menu(s) if it does not exist
        # Multiple sub-menus can be nested by using ':::'. e.g sub_menu_text == 'level1:::level2:::level3'
        tokens = sub_menu_text.split(':::')
        for idx in range(len(tokens)):
            sub_menu_path = ':::'.join(tokens[:idx+1])
            if sub_menu_path not in sub_menus:
                icon = self.get_sub_menu_icon(sub_menu_path, image_name)
                ac = create_menu_item(self, parent_menu, tokens[idx], image=icon, shortcut=None)
                sm = QMenu()
                ac.setMenu(sm)
                sub_menus[sub_menu_path] = sm
            # Now set our menu variable so the parent menu item will be the sub-menu
            parent_menu = sub_menus[sub_menu_path]
        return parent_menu

    def get_sub_menu_icon(self, sub_menu_path, first_action_icon):
        if tweaks.get('action_chains_default_sub_menu_icon', 'first_action') == 'first_action':
            return first_action_icon
        return None

    def create_all_menu_actions_for_chain(self, chain_config, parent_menu, sub_menus, menu_text, image_name):
        variation_config = chain_config.get('variation_settings', {})
        if variation_config.get('template'):
            # If the user configured chain variants, we make multiple menu entries,
            # one for each configured variant. Each chain variant is a copy of
            # the original chain, except it has different argument that will be 
            # stored in a chain var called _variant_argument
            try:
                return self.create_variants_menu_actions(chain_config, variation_config, menu_text, parent_menu, sub_menus)
            except Exception as e:
                if DEBUG:
                    import traceback
                    prints(f'Action Chains: Unable to create variant for chain ({menu_text})\n{traceback.format_exc()}')
        else:
            # If the user has not configured any variants for the chain, we can one entry with no _variant_argument
            return self.create_chain_menu_action(chain_config, menu_text, parent_menu, menu_text, image_name)

    def create_variants_menu_actions(self, chain_config, variation_config, menu_text, parent_menu, sub_menus):
        variants = self.variants_from_config(variation_config)
        all_ac = []
        for key, variant in variants.items():
            icon = variant.get('icon', '')
            sub_menu_text = variant.get('sub_menu_text', '')
            if sub_menu_text:
                sm = self.create_sub_menus(parent_menu, sub_menus, sub_menu_text, icon)
            else:
                sm = parent_menu
            # Ensure the name cannot clash with any other menu item
            unique_name = menu_text + '__chain_variant__' + variant['menu_text']
            variant_chain_config = copy.deepcopy(chain_config)
            variant_chain_config['variant_argument'] = variant['argument']
            if key.startswith('__separator__'):
                sm.addSeparator()
            else:
                ac = self.create_chain_menu_action(variant_chain_config, unique_name, sm, variant['menu_text'], icon)
                all_ac.append(ac)
        return all_ac

    def create_chain_menu_action(self, chain_config, unique_name, parent_menu, menu_text, image_name):
        icon = self.icon_cache.get_icon(image_name)
        ac = create_menu_action_unique(self, parent_menu, menu_text, icon,
                       unique_name=unique_name,
                       triggered=partial(self.run_chain, chain_config, True, True))
        # Maintain our list of menus by chain references so we can easily enable/disable menus when user right-clicks.
        self.menu_actions.append(ac)
        self.chain_menu.append((chain_config, ac))
        return ac

    def variants_from_config(self, variation_config, template_functions=None):
        template = variation_config.get('template', '')
        if not template:
            return []
        mi = get_metadata_object(self.gui)

        # Must be initialized here to create a new dictionary
        chain_variant_function = ChainVariant(self)

        if not template_functions:
            template_functions = self.template_functions

        # We will add a special function, so we copy first to avoid adding it other places in the plugin
        template_functions = copy.copy(template_functions)

        # Add special function
        template_functions[chain_variant_function.name] = chain_variant_function

        template_output = get_template_output(template, mi, TEMPLATE_ERROR, mi,
                                              template_functions=template_functions)
        if template_output.startswith(TEMPLATE_ERROR):
            return {}

        return chain_variant_function.chain_variants

    def about_to_show_menu(self):
        # Enable/disable chains based on their conditions evaluation
        #
        # If no conditions set for any chain, no need to waste time calculating and delaying the menu
        c = cfg.chains_have_conditions()
        if not c:
            return
        # mi is calculated once and passed to save time, we also initialize the cache for template
        # function selection_count, since selections are not chaing during the lifespan of this method,
        # and flush it before leaving
        mi = get_metadata_object(self.gui)
        call_method(self.template_functions, 'init_cache')
        QApplication.setOverrideCursor(Qt.WaitCursor)
        try:
            for chain_config, menu_action in self.chain_menu:
                is_enabled = self.is_menu_enabled(chain_config, mi)
                menu_action.setEnabled(is_enabled)
        finally:
            QApplication.restoreOverrideCursor()
            call_method(self.template_functions, 'flush_cache')

    def about_to_hide_menu(self):
        # When hiding menus we must re-enable all selections in case a shortcut key for the
        # action gets pressed after moving to a new row.
        self.set_enabled_for_all_menu_actions(True)

    def set_enabled_for_all_menu_actions(self, is_enabled):
        for chain_config, menu_action in self.chain_menu:
            menu_action.setEnabled(is_enabled)

    def is_menu_enabled(self, chain_config, mi):
        '''
        Determine whether menu item for the chain is enabled or not
        '''
        # If the user opted not to disable menu if conditions are not met
        if not chain_config.get('condition_settings', {}).get('affect_menu'):
            return True
        #
        loop = Chain(self, chain_config, mi=mi)
        is_true = loop.check_chain_conditions()
        del loop
        return is_true

    def run_chain(self, chain_config, check_conditions=True, print_errors=True):
        chain = Chain(self, chain_config, chain_vars={})
        chain.run(check_conditions=check_conditions,
                  print_errors=print_errors)
        del chain

    def call_actions_method(self, meth_name, *args):
        '''
        loop over all actions and call the supplied method if present
        '''
        call_method(self.actions, meth_name, *args)

    def edit_shortcuts(self):
        # Force the menus to be rebuilt immediately, so we have all our actions registered
        self.rebuild_menus()
        d = KeyboardConfigDialog(self.gui, self.action_spec[0])
        if d.exec_() == d.Accepted:
            self.gui.keyboard.finalize()

    def manage_modules(self):
        d = ManageModulesDialog(self.gui, self)
        d.exec_()

    def event_manager(self):
        d = EventsDialog(self.gui, self)
        d.exec_()

    def add_chains(self):
        d = ChainsDialog(self)
        d.exec_()

    def show_configuration(self):
        self.interface_action_base_plugin.do_user_config(self.gui)
