#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from collections import OrderedDict

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog

from calibre_plugins.action_chains.scopes.base import ActionScope
from calibre_plugins.action_chains.scopes.fallback import FallbackScope

# builtins
from calibre_plugins.action_chains.scopes.basic import BasicScope
from calibre_plugins.action_chains.scopes.template_scopes import TemplateScope
#

try:
    load_translations()
except NameError:
    prints("ActionChains::scopes/__int__.py - exception when loading translations")

def get_imported_scopes(plugin_action):
    all_imported_scopes = {}
    for plugin_name, imported_resources in plugin_action.imported_resources.items():
        imported_scopes = imported_resources.get('scopes', [])
        for imported_scope in imported_scopes:
            name = imported_scope.name
            # must define a name attribute, must be set and not clash with builtin names
            # which can be imported into the module manager by custom scopes
            if name in ['', 'Action Scope']:
                continue
            imported_scope._source_plugin = plugin_name
            all_imported_scopes[name] = imported_scope          
    return all_imported_scopes

def get_user_scopes(plugin_action):
    user_scopes = {}
    for cls in plugin_action.user_modules.get_classes(class_filters=[ActionScope]):
        name = cls.name
        # must define a name attribute, must be set and not clash with builtin names
        # which can be imported into the module manager by custom scopes
        if name in ['', 'Action Scope']:
            continue
        user_scopes[name] = cls            
    return user_scopes

def get_all_scopes(plugin_action):
    _builtin_scopes = [
        FallbackScope,
        BasicScope,
        TemplateScope
    ]
    
    builtin_scopes = OrderedDict()

    for scope_cls in _builtin_scopes:
        builtin_scopes[scope_cls.name] = scope_cls

    _user_scopes = get_user_scopes(plugin_action)
    
    _imported_scopes = get_imported_scopes(plugin_action)

    all_scopes = OrderedDict()
    user_scopes = OrderedDict()
    imported_scopes = OrderedDict()
    
    for scope_name, scope_cls in builtin_scopes.items():
        scope = scope_cls(plugin_action)
        all_scopes[scope_name] = scope

    # Note: imported scopes can either be class or instantiated object
    for scope_name, scope_obj in _imported_scopes.items():
        # dont override builtin scopes
        if scope_name in builtin_scopes.keys():
            continue
        try:
            if isinstance(scope_obj, ActionScope):
                scope = scope_obj
            elif issubclass(scope_obj, ActionScope):
                scope = scope_obj(plugin_action)
            all_scopes[scope_name] = scope
            imported_scopes[scope_name] = scope
        except TypeError as e:
            # TypeError: issubclass() arg 1 must be a class
            import traceback
            if DEBUG:
                prints(f'Action Chains: Error intializing imported scope: Un-reconized object: {scope_obj}\n{traceback.format_exc()}')            
        except Exception as e:
            import traceback
            if DEBUG:
                prints(f'Action Chains: Error intializing imported scope: {scope_name}\n{traceback.format_exc()}')

    for scope_name, scope_cls in _user_scopes.items():
        # dont override builtin scopes
        if scope_name in builtin_scopes.keys():
            continue
        try:
            scope = scope_cls(plugin_action)
            all_scopes[scope_name] = scope
            user_scopes[scope_name] = scope
        except Exception as e:
            import traceback
            if DEBUG:
                prints(f'Action Chains: Error intializing user scope: {scope_name}\n{traceback.format_exc()}')

    return all_scopes, builtin_scopes, user_scopes, imported_scopes            
