#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import copy
from functools import partial

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                     QLabel, QToolButton, QPushButton, QSizePolicy, QSpacerItem, 
                     QFont, QIcon, QModelIndex, QCheckBox, QGroupBox, QMenu, QTimer,
                     QDialog, QDialogButtonBox, QSpinBox, QTextEdit, QFont)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog, question_dialog
from calibre.gui2.widgets2 import Dialog

from calibre_plugins.action_chains.common_utils import get_icon
from calibre_plugins.action_chains.gui.models import EventsModel, EventMembersModel, EventVariantsModel, UP, DOWN
from calibre_plugins.action_chains.gui.views import EventsTable, EventMembersTable, EventVariantsTable
import calibre_plugins.action_chains.config as cfg

try:
    load_translations()
except NameError:
    prints("ActionsChain::gui/events_dialogs.py - exception when loading translations")

class EventTimer(QDialog):
    def __init__(self, parent, ceiling=5, message=_('Action chains event will start shortly')):
        QDialog.__init__(self, parent)
        self.setWindowTitle(_('Action chains event countdown'))
        self.message = message
        self.ceiling = ceiling
        self.timer = QTimer()
        self.timer.timeout.connect(self._on_timeout)
        self.counter = self.ceiling
        self.setup_ui()

    def setup_ui(self):
        l = QVBoxLayout()
        self.setLayout(l)

        self.lcd = QLabel(f'{self.counter}')
        self.lcd.setStyleSheet("border : 4px solid black;")
        self.lcd.setFont(QFont('Arial', 25))
        self.lcd.setAlignment(Qt.AlignCenter)

        self.text_label = QLabel(self.message, self)
        self.text_label.setAlignment(Qt.AlignCenter)
        
        self.bb = QDialogButtonBox(QDialogButtonBox.Ok|QDialogButtonBox.Cancel, self)
        self.bb.accepted.connect(self.accept)
        self.bb.rejected.connect(self.cancel)

        l.addWidget(self.lcd)        
        l.addWidget(self.text_label)
        l.addWidget(self.bb)

    def _on_timeout(self):
        self.counter = self.counter - 1
        self.lcd.setText(f'{self.counter}')
        if self.counter == 0:
            self.timer.stop()
            self.accept()

    def showEvent(self, ev):
        QDialog.showEvent(self, ev)
        self.start_countdown()

    def start_countdown(self):
        self.counter = self.ceiling
        self.timer.start(1000)

    def cancel(self):
        self.timer.stop()
        self.reject()

class CountdownSettings(QDialog):
    def __init__(self, parent, duration=5, message=_('Action chains event starting shortly')):
        QDialog.__init__(self, parent)
        l = QVBoxLayout()
        self.setLayout(l)

        hl = QHBoxLayout()
        label1 = QLabel(_('Duration: '))
        self.spin = QSpinBox(self)
        self.spin.setValue(duration)
        hl.addWidget(label1)
        hl.addWidget(self.spin)
        l.addLayout(hl)

        label2 = QLabel(_('Message:'))
        self.tb = QTextEdit()
        self.tb.insertPlainText(message)
        l.addWidget(label2)
        l.addWidget(self.tb)

        self.bb = QDialogButtonBox(QDialogButtonBox.Ok|QDialogButtonBox.Cancel, self)  
        self.bb.accepted.connect(self.accept)
        self.bb.rejected.connect(self.reject)      
        l.addWidget(self.bb)

    def accept(self):
        self.message = self.tb.toPlainText()
        self.duration = self.spin.value()
        QDialog.accept(self)

class EventOptionsDialog(Dialog):

    def __init__(self, name, parent, plugin_action, event, widget_cls, title=_('Settings')):
        self.plugin_action = plugin_action
        self.widget_cls = widget_cls
        self.event = event
        Dialog.__init__(self, title, name, parent)

    def setup_ui(self):
        self.widget = self.widget_cls(self.plugin_action)
        l = QVBoxLayout()
        self.setLayout(l)
        l.addWidget(self.widget)
        l.addWidget(self.bb)

    def load_settings(self, settings):
        self.widget.load_settings(settings)

    def save_settings(self):
        return self.widget.save_settings()

    def validate(self, settings):
        if hasattr(self.widget, 'validate'):
            return self.widget.validate(settings)
        else:
            return self.event.validate(settings)
    
    def accept(self):
        self.settings = self.save_settings()
        # Validate settings
        #is_valid = self.event.validate(self.settings)
        is_valid = self.validate(self.settings)
        if is_valid is not True:
            msg, details = is_valid
            error_dialog(self, msg, details, show=True)
            return
        Dialog.accept(self)

class EventMembersDialog(Dialog):
    def __init__(self, parent, plugin_action, event_config):
        self.plugin_action = plugin_action
        self.gui = plugin_action.gui
        self.db = self.gui.current_db
        self.event_config = event_config
        self.event_settings = event_config['event_settings']
        self.countdown_settings = self.event_settings.get('countdown_settings', {})
        self.event_name = self.event_config['event_name']
        self.event = self.plugin_action.events[self.event_name]
        self.widget_cls = self.event.config_widget()
        self.event_opts = self.event_settings.get('event_options', {})
        Dialog.__init__(self, _('Event Members Dialog'), 'action-chains-event-members-dialog', parent)

    def setup_ui(self):
        self.setWindowTitle(_('Add chains'))
        l = QVBoxLayout()
        self.setLayout(l)

        settings_l = QGridLayout()
        l.addLayout(settings_l)

        gb1 = QGroupBox(_('General Options'))
        l1 = QVBoxLayout()
        gb1.setLayout(l1)
        l.addWidget(gb1)

        hl1 = QHBoxLayout()

        countdown_chk = self.countdown_chk = QCheckBox(_('Run a countdown dialog before invoking event'))
        countdown_chk.setChecked(self.countdown_settings.get('enabled', False))
        countdown_button = self.countdown_button = QPushButton(_('Countdown settings'))
        countdown_button.clicked.connect(self._on_countdown_button_clicked)
        hl1.addWidget(countdown_chk)
        hl1.addWidget(countdown_button)
        l1.addLayout(hl1)

        validate_chk = self.validate_chk = QCheckBox(_('Validate chains before running and abort in case of errors'))
        validate_chk.setChecked(self.event_settings.get('validate', True))
        l1.addWidget(validate_chk)

        hl2 = QHBoxLayout()
        l.addLayout(hl2)
        self.event_opts_button = QPushButton(_('Event specific options'))
        self.event_opts_button.clicked.connect(self._on_event_opts_button_clicked)
        self.event_opts_button.setEnabled(bool(self.widget_cls))
        hl2.addWidget(self.event_opts_button)

        _table_gb = QGroupBox(_('Event members'))
        _table_l = QHBoxLayout()
        _table_gb.setLayout(_table_l)
        l.addWidget(_table_gb)
        
        self._table = EventMembersTable(self)
        _table_l.addWidget(self._table)
        
        _model = EventMembersModel(self.event_settings.get('event_members', []), self.plugin_action)
        _model.validate()
        self._table.set_model(_model)
        self._table.selectionModel().selectionChanged.connect(self._on_table_selection_change)
        
        # restore table state
        state = cfg.plugin_prefs[cfg.KEY_GPREFS][cfg.KEY_EVENT_MEMBERS_TABLE_STATE]
        if state:
            self._table.apply_state(state)

        # Add a vertical layout containing the the buttons to move up/down etc.
        button_layout = QVBoxLayout()
        _table_l.addLayout(button_layout)
        
        move_up_button = self.move_up_button = QToolButton(self)
        move_up_button.setToolTip(_('Move row up'))
        move_up_button.setIcon(get_icon('arrow-up.png'))
        button_layout.addWidget(move_up_button)
        spacerItem1 = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem1)

        add_button = self.add_button = QToolButton(self)
        add_button.setToolTip(_('Add row'))
        add_button.setIcon(get_icon('plus.png'))
        button_layout.addWidget(add_button)
        spacerItem2 = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem2)

        delete_button = self.delete_button = QToolButton(self)
        delete_button.setToolTip(_('Delete row'))
        delete_button.setIcon(get_icon('minus.png'))
        button_layout.addWidget(delete_button)
        spacerItem4 = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem4)

        move_down_button = self.move_down_button = QToolButton(self)
        move_down_button.setToolTip(_('Move row down'))
        move_down_button.setIcon(get_icon('arrow-down.png'))
        button_layout.addWidget(move_down_button)

        move_up_button.clicked.connect(partial(self._table.move_rows,UP))
        move_down_button.clicked.connect(partial(self._table.move_rows,DOWN))
        add_button.clicked.connect(self._table.add_row)
        delete_button.clicked.connect(self._table.delete_rows)
        
        l.addWidget(self.bb)
        
        self._on_table_selection_change()

#        self.setMinimumSize(400, 300)        
#        self.resize(500, 600)
#        self.setSizePolicy(QSizePolicy.Preferred, QSizePolicy.Preferred)

    def _on_table_selection_change(self):
        sm = self._table.selectionModel()
        selection_count = len(sm.selectedRows())
        self.delete_button.setEnabled(selection_count > 0)
        self.move_up_button.setEnabled(selection_count > 0)
        self.move_down_button.setEnabled(selection_count > 0)

    def _on_countdown_button_clicked(self):
        duration = self.countdown_settings.get('duration', 5)
        message = self.countdown_settings.get('message', _('Action chain event starting shortly'))
        d = CountdownSettings(self, duration=duration, message=message)
        if d.exec_() == d.Accepted:
            self.countdown_settings = {
                'enabled': self.countdown_chk.isChecked(),
                'message': d.message,
                'duration': d.duration
            }

    def _on_event_opts_button_clicked(self):
        name = f'action_chains_{self.event_name}_event_settings'
        d = EventOptionsDialog(name, self, self.plugin_action, self.event, self.widget_cls)
        if self.event_opts:
            d.load_settings(self.event_opts)
        if d.exec_() == d.Accepted:
            self.event_opts = d.settings

    def save_table_state(self):
        # save table state
        cfg.plugin_prefs[cfg.KEY_GPREFS][cfg.KEY_EVENT_MEMBERS_TABLE_STATE] = self._table.get_state()

    def reject(self):
        self.save_table_state()
        Dialog.reject(self)

    def accept(self):
        self.save_table_state()
        
#        is_valid = self._table.model().validate()
#        if is_valid is not True:
#            message = _('Event contains some empty rows. '
#                        'You must complete the missing items before proceeding')
#            error_dialog(
#                self,
#                _('Event invalid'),
#                message,
#                show=True
#            )
#            return

        self.event_settings = {}        
        event_members = self._table.model().event_members
        # remove error keys from event_members
        for event_member in event_members:
            try:
                del event_member['errors']
            except:
                pass
        self.event_settings['event_members'] = event_members
        if self.event_opts:
            self.event_settings['event_options'] = self.event_opts
        if not self.countdown_chk.isChecked():
            self.countdown_settings = {'enabled': False}
        else:
            self.countdown_settings['enabled'] = True
        self.event_settings['countdown_settings'] = self.countdown_settings
        self.event_settings['validate'] = self.validate_chk.isChecked()
        Dialog.accept(self)

class EventsDialog(Dialog):
    def __init__(self, parent, plugin_action):
        self.plugin_action = plugin_action
        self.gui = plugin_action.gui
        self.db = self.gui.current_db
        self.events_config = cfg.get_events_config()
        Dialog.__init__(self, _('Events Dialog'), 'action-chains-events-dialog', parent)
        self.restart_required = []
        self.plugin_action.plugin_restart_required.connect(self._on_restart_required_signal)

    def setup_ui(self):
        self.setWindowTitle(_('Add events'))
        l = QVBoxLayout()
        self.setLayout(l)

        settings_l = QGridLayout()
        l.addLayout(settings_l)

        _table_gb = QGroupBox(_('Events'))
        _table_l = QHBoxLayout()
        _table_gb.setLayout(_table_l)
        l.addWidget(_table_gb)
        
        self._table = EventsTable(self)
        _table_l.addWidget(self._table)
        
        _model = EventsModel(copy.deepcopy(self.events_config), self.plugin_action)
        _model.validate()
        self._table.set_model(_model)
        self._table.selectionModel().selectionChanged.connect(self._on_table_selection_change)
        
        # restore table state
        state = cfg.plugin_prefs.get(cfg.KEY_EVENTS_TABLE_STATE, None)
        if state:
            self._table.apply_state(state)

        # Add a vertical layout containing the the buttons to move up/down etc.
        button_layout = QVBoxLayout()
        _table_l.addLayout(button_layout)
        
        move_up_button = self.move_up_button = QToolButton(self)
        move_up_button.setToolTip(_('Move row up'))
        move_up_button.setIcon(get_icon('arrow-up.png'))
        button_layout.addWidget(move_up_button)
        spacerItem1 = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem1)

        add_button = self.add_button = QToolButton(self)
        add_button.setToolTip(_('Add row'))
        add_button.setIcon(get_icon('plus.png'))
        button_layout.addWidget(add_button)
        spacerItem2 = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem2)

        delete_button = self.delete_button = QToolButton(self)
        delete_button.setToolTip(_('Delete row'))
        delete_button.setIcon(get_icon('minus.png'))
        button_layout.addWidget(delete_button)
        spacerItem4 = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem4)

        move_down_button = self.move_down_button = QToolButton(self)
        move_down_button.setToolTip(_('Move row down'))
        move_down_button.setIcon(get_icon('arrow-down.png'))
        button_layout.addWidget(move_down_button)

        move_up_button.clicked.connect(partial(self._table.move_rows,UP))
        move_down_button.clicked.connect(partial(self._table.move_rows,DOWN))
        add_button.clicked.connect(self._table.add_row)
        delete_button.clicked.connect(self._table.delete_rows)

        button_layout_2 = QHBoxLayout()
        l.addLayout(button_layout_2)
        self.event_variants_button = QPushButton(_('Event Variants'))
        self.event_variants_button.clicked.connect(self._on_event_variants_button_clicked)
        button_layout_2.addWidget(self.event_variants_button)
        button_layout_2.addStretch(1)
        
        l.addWidget(self.bb)
        
        self._on_table_selection_change()

#        self.setMinimumSize(400, 300)        
#        self.resize(500, 600)
#        self.setSizePolicy(QSizePolicy.Preferred, QSizePolicy.Preferred)

    def _on_table_selection_change(self):
        sm = self._table.selectionModel()
        selection_count = len(sm.selectedRows())
        self.delete_button.setEnabled(selection_count > 0)
        self.move_up_button.setEnabled(selection_count > 0)
        self.move_down_button.setEnabled(selection_count > 0)

    def _on_event_variants_button_clicked(self):
        variants_config = copy.deepcopy(cfg.plugin_prefs[cfg.KEY_EVENT_VARIANTS])
        d = EventVariantsDialog(self, self.plugin_action, variants_config)
        if d.exec_() == d.Accepted:
            cfg.plugin_prefs[cfg.KEY_EVENT_VARIANTS] = d.variants_config
            # Update
            self.plugin_action.on_modules_update()

    def save_table_state(self):
        # save table state
        cfg.plugin_prefs[cfg.KEY_EVENTS_TABLE_STATE] = self._table.get_state()

    def _on_restart_required_signal(self, target, msg):
        if target == 'events_dialog':
            self.restart_required.append(msg)

    def reject(self):
        self.save_table_state()
        Dialog.reject(self)

    def accept(self):
        self.save_table_state()
    
        events_config = self._table.model().events_config
        # remove error keys from event_members
        for event_config in events_config:
            try:
                del event_config['errors']
            except:
                pass
        self.events_config = events_config

        data = {}
        data[cfg.EVENT_ENTRIES] = events_config
        cfg.plugin_prefs[cfg.STORE_EVENTS_NAME] = data

        Dialog.accept(self)
        from calibre.gui2 import show_restart_warning
        if self.restart_required:
            details = _('You need to restart for the following settings to take effect:\n') + '\n'.join(self.restart_required)
            do_restart = show_restart_warning(details)
            if do_restart:
                self.gui.quit(restart=True)


class EventVariantsDialog(Dialog):
    def __init__(self, parent, plugin_action, variants_config):
        self.plugin_action = plugin_action
        self.gui = plugin_action.gui
        self.db = self.gui.current_db
        self.variants_config = variants_config
        Dialog.__init__(self, _('Event Variants Dialog'), 'action-chains-event-variants-dialog', parent)

    def setup_ui(self):
        self.setWindowTitle(_('Add event variants'))
        l = QVBoxLayout()
        self.setLayout(l)

        settings_l = QGridLayout()
        l.addLayout(settings_l)

        _table_gb = QGroupBox(_('Event Variants'))
        _table_l = QHBoxLayout()
        _table_gb.setLayout(_table_l)
        l.addWidget(_table_gb)
        
        self._table = EventVariantsTable(self)
        _table_l.addWidget(self._table)
        
        _model = EventVariantsModel(copy.deepcopy(self.variants_config), self.plugin_action)
        _model.validate()
        self._table.set_model(_model)
        self._table.selectionModel().selectionChanged.connect(self._on_table_selection_change)
        
        # restore table state
        state = cfg.plugin_prefs.get(cfg.KEY_GPREFS, {}).get(cfg.KEY_EVENT_VARIANTS_TABLE_STATE, None)
        if state:
            self._table.apply_state(state)

        # Add a vertical layout containing the the buttons to move up/down etc.
        button_layout = QVBoxLayout()
        _table_l.addLayout(button_layout)
        
        move_up_button = self.move_up_button = QToolButton(self)
        move_up_button.setToolTip(_('Move row up'))
        move_up_button.setIcon(get_icon('arrow-up.png'))
        button_layout.addWidget(move_up_button)
        spacerItem1 = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem1)

        add_button = self.add_button = QToolButton(self)
        add_button.setToolTip(_('Add row'))
        add_button.setIcon(get_icon('plus.png'))
        button_layout.addWidget(add_button)
        spacerItem2 = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem2)

        delete_button = self.delete_button = QToolButton(self)
        delete_button.setToolTip(_('Delete row'))
        delete_button.setIcon(get_icon('minus.png'))
        button_layout.addWidget(delete_button)
        spacerItem4 = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem4)

        move_down_button = self.move_down_button = QToolButton(self)
        move_down_button.setToolTip(_('Move row down'))
        move_down_button.setIcon(get_icon('arrow-down.png'))
        button_layout.addWidget(move_down_button)

        move_up_button.clicked.connect(partial(self._table.move_rows,UP))
        move_down_button.clicked.connect(partial(self._table.move_rows,DOWN))
        add_button.clicked.connect(self._table.add_row)
        delete_button.clicked.connect(self._table.delete_rows)
        
        l.addWidget(self.bb)
        
        self._on_table_selection_change()

#        self.setMinimumSize(400, 300)        
#        self.resize(500, 600)
#        self.setSizePolicy(QSizePolicy.Preferred, QSizePolicy.Preferred)

    def _on_table_selection_change(self):
        sm = self._table.selectionModel()
        selection_count = len(sm.selectedRows())
        self.delete_button.setEnabled(selection_count > 0)
        self.move_up_button.setEnabled(selection_count > 0)
        self.move_down_button.setEnabled(selection_count > 0)

    def save_table_state(self):
        # save table state
        cfg.plugin_prefs[cfg.KEY_GPREFS][cfg.KEY_EVENT_VARIANTS_TABLE_STATE] = self._table.get_state()

    def reject(self):
        self.save_table_state()
        Dialog.reject(self)

    def accept(self):
        self.save_table_state()
    
        variants_config = self._table.model().variants_config
        # remove error keys from event_members
        for variant_config in variants_config:
            try:
                del variant_config['errors']
            except:
                pass
        self.variants_config = variants_config

        Dialog.accept(self)
