#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from qt.core import (QApplication, Qt, QTimer, QWidget, QVBoxLayout,
                     QSpinBox, QGroupBox, pyqtSignal)

from calibre import prints
from calibre.constants import DEBUG

from calibre_plugins.action_chains.events.base import ChainEvent

try:
    load_translations()
except NameError:
    prints("ActionChains::events/timer.py - exception when loading translations")

class ConfigWidget(QWidget):
    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.gui = plugin_action.gui
        self.db = self.gui.current_db
        self._init_controls()

    def _init_controls(self):

        l = self.l = QVBoxLayout()
        self.setLayout(l)

        timer_group_box = QGroupBox('Timer event interval in seconds')
        timer_group_box_l = QVBoxLayout()
        timer_group_box.setLayout(timer_group_box_l)
        l.addWidget(timer_group_box)
        self.timer_spin = QSpinBox()
        self.timer_spin.setMaximum(100000)
        self.timer_spin.setMinimum(60)
        self.timer_spin.setSingleStep(1)
        timer_group_box_l.addWidget(self.timer_spin)
        
        l.addStretch(1)

        self.setMinimumSize(300,300)

    def load_settings(self, settings):
        if settings:
            self.timer_spin.setValue(settings['timeout_interval'])

    def save_settings(self):
        settings = {}
        settings['timeout_interval'] = self.timer_spin.value()
        self.plugin_action.plugin_restart_required.emit('events_dialog', 'Timer event interval')
        return settings

class TimerEvent(ChainEvent):

    name = 'Timer'
    support_variants = True
    timer = pyqtSignal()

    def __init__(self, plugin_action):
        ChainEvent.__init__(self, plugin_action)
        QTimer.singleShot(self.get_interval_in_seconds() * 1000, self._on_timeout)

    def get_interval_in_seconds(self):
        event_options = self.get_options()
        interval = event_options.get('timeout_interval', 300)
        return interval
        
    def _on_timeout(self):
        # Signal is not emitted if another chain is running
        if not self.plugin_action.chainStack:
            QTimer.singleShot(0, self.timer.emit)
        
        # keep the timer runnig after re-reading interval value
        QTimer.singleShot(self.get_interval_in_seconds() * 1000, self._on_timeout) 
    
    def get_event_signal(self):
        return self.timer

    def config_widget(self):
        return ConfigWidget
