#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import os
import copy
import time
from collections import defaultdict
import contextlib
from threading import Event
from queue import Queue

from qt.core import (QApplication, Qt, QProgressBar, QPushButton, QWidget, QHBoxLayout,
                     QModelIndex)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog, Dispatcher
from calibre.gui2.threaded_jobs import ThreadedJob
from calibre.utils.date import now, as_local_time, as_utc, format_date
from calibre.ebooks.metadata.book.base import Metadata

import calibre_plugins.action_chains.config as cfg
from calibre_plugins.action_chains.templates import TEMPLATE_ERROR, get_metadata_object, get_template_output
from calibre_plugins.action_chains.conditions import check_conditions
from calibre_plugins.action_chains.common_utils import call_method, truncate
from calibre_plugins.action_chains.scopes.scope_tools import ScopeWrapper, validate_scope
from calibre_plugins.action_chains.export_import import chains_config_from_archive, pick_archive_to_import

try:
    load_translations()
except NameError:
    prints("ActionChains::chain_tools.py - exception when loading translations")

class Chain(object):

    class UserInterrupt(Exception):
        pass

    def __init__(self, plugin_action, chain_config, event=None,
                    mi=None, qt_processEvents=True, show_progress=None,
                    event_args=(), chain_vars={}, chain_caller=False):
        self.plugin_action = plugin_action
        self.gui = plugin_action.gui
        self.db = self.gui.current_db
        self.event = event
        self.event_args = event_args
        self.qt_processEvents = qt_processEvents
        # copy as to not overwrite mutable chain data
        self.chain_config = copy.deepcopy(chain_config)
        self.chain_name = self.chain_config['menuText']
        self.chain_links = self.chain_config['chain_settings'].get('chain_links')
        if show_progress is None:
            self.show_progress = self.chain_config['chain_settings'].get('show_progress', True)
        else:
            self.show_progress = show_progress
        self.start_time = now()
        self.step_length = 1
        self.current_step = -1
        self.last_step = None
        self.pbar = None
        self.chain_caller = chain_caller
        self.jobify = self.is_job()
        # mi is calculated once and passed to each chain instance to save time, used only
        # for testing conditions for chains when building menus, when checking conditions for actions, other updated
        # objects are used
        if mi:
            self.mi = mi
        else:
            self.mi = get_metadata_object(self.gui)

        self._initialize_vars(chain_vars=chain_vars)

    def is_job(self):
        if not self.chain_config['chain_settings'].get('jobify', False):
            return False
        if self.chain_caller:
            return False
        chain_name = self.chain_config['menuText']
        chain_settings = self.chain_config.get('chain_settings', {})
        chain_links = chain_settings.get('chain_links', [])
        for chain_link in chain_links:
            action_name = chain_link['action_name']
            action = self.plugin_action.actions[action_name]
            action_settings = chain_link.get('action_settings')
            if not action.run_as_job(action_settings):
                if DEBUG:
                    print(f'Action Chains: Cannot run as a job ({chain_name} > {action_name})')
                return False
        return True

    def validate(self, use_cache=True):
        all_actions = self.plugin_action.actions
        msg = _('Chain Error')
        details = _('Validating the chain settings before running failed.\
                        You can see the detailed errors by opening the chain in the config dialog')
        if use_cache:
            # init_cache for template functions
            call_method(self.plugin_action.template_functions, 'init_cache')
        try:
            chain_name = self.chain_config['menuText']
            chain_settings = self.chain_config['chain_settings']
            chain_links = chain_settings.get('chain_links',[])
            if len(chain_links) == 0:
                return (_('Empty chain'), _('A chain must contain at least one action'))
            for chain_link in chain_links:
                action_name = chain_link['action_name']
                if action_name in all_actions.keys():
                    action = all_actions[action_name]
                    action_settings = chain_link.get('action_settings')
                    if not action_settings:
                        action_settings = action.default_settings()
                    # Sometimes user can opt to make action not be validated if the action
                    # conditions are not met.
                    action_conditions = chain_link.get('condition_settings', {})
                    if action_conditions and action_conditions.get('affect_validation', False):
                        do_run = self.check_action_conditions(action_conditions)
                        if not do_run:
                            continue
                    #
                    is_valid = action.validate(action_settings)
                    if is_valid is not True:
                        if DEBUG:
                            prints(f'Action Chains: chain ({chain_name}): action ({action_name}) settings are not valid')
                        return msg, details
                    # validate scope settings
                    is_scope_valid = validate_scope(self.plugin_action, chain_link)
                    if is_scope_valid is not True:
                        if DEBUG:
                            print(f'Action Chains: chain ({chain_name}): action ({action_name}) settings are not valid')
                        return msg, details   
                else:
                    if DEBUG:
                        print(f'Action Chains: chain ({chain_name}): action ({action_name}) is not currently available')
                    return msg, details
        except Exception as e:
            if DEBUG:
                import traceback
                print(f'Action Chains: Exception when checking chain:\nchain: {self.chain_config}\nexception: {traceback.format_exc()}')
                raise e
        finally:
            if use_cache:
                call_method(self.plugin_action.template_functions, 'flush_cache')
        return True

    def _run_loop(self, log=print, abort=Event(), notifications=Queue()):
        count = len(self.chain_links)
        if self.jobify:
            pass
        else:
            QApplication.setOverrideCursor(Qt.WaitCursor)
        try:

            #
            # chain_start_time has to be reset here in case someone initializes this object once, 
            # and and runs the loop mulitple times
            self.chain_start_time = now()            
            # initiate previous action start time, will be reset after each action
            self.previous_action_start_time = self.chain_start_time
            
            self.set_chain_vars({'_chain_start_time': format_date(self.chain_start_time, 'iso')})

            # Add to stack. Non empty stack will block event manager
            self.plugin_action.chainStack.append(self.chain_name)
            #
            
            if DEBUG:
                prints(f'Action Chains: Starting chain: {self.chain_name}')
            if self.jobify:
                pass
            else:
                # set progress bar
                pbar = self.pbar = StatusBarProgress(truncate(self.chain_name, 10), count, abort=abort)
                if count > 1 and self.show_progress:
                    self.gui.status_bar.addWidget(pbar)
                    pbar.pb.setValue(0)

            # Put Last Modified plugin into hibernation mode durating lifetime of the chain
            last_modified_hibernate = getattr(self.gui.iactions.get('Last Modified'), 'hibernate', contextlib.nullcontext())
            with last_modified_hibernate:
                while True:
                    self.last_step = self.current_step
                    self.current_step += self.step_length if self.step_length > 0 else 1
                    # now that we read the step lenght that might have been modified by the previous action, restore it back
                    self.step_length = 1
                    if self.current_step > len(self.chain_links) - 1:
                        break
                    #
                    chain_link = self.chain_links[self.current_step]
                    # store start time for each chain link, used by some actions to change selections based on last_modified date.
                    chain_link['start_time'] = now()
                    action_name = chain_link['action_name']
                    action = self.plugin_action.actions[action_name]
                    action_settings = chain_link.get('action_settings')
                    if not action_settings:
                        action_settings = action.default_settings()

                    self.update_vars_for_action(chain_link)                
                    # test conditions for running action
                    do_run = self.check_action_conditions(chain_link.get('condition_settings', {}))
                    #
                    try:
                        if do_run:
                            if DEBUG:
                                prints(f'Action Chains: {self.chain_name}: starting action No. {self.current_step+1}: {action_name}')
                            action.run(self.gui, action_settings, self)
                        else:
                            if DEBUG:
                                msg = f'Action Chains: {self.chain_name}: Conditions are not met for action No. {self.current_step+1} ({action_name})'
                                tooltip = chain_link.get('condition_settings', {}).get('tooltip', '')
                                if tooltip:
                                    msg += f'\ntooltip: {tooltip}'
                                prints(msg)

                        # check if the user pressed abort button
                        self.stop_if_canceled(abort)
                        if self.pbar is not None:
                            pbar.pb.setValue(self.current_step + 1)
                        if self.jobify:
                            if notifications:
                                notifications.put(((self.current_step+1)/count, f'Running Chain: {self.chain_name}'))
                        #
                        # This is set false when called from library view double clicked event as it
                        # makes the active cell enter into edit mode.
                        if self.qt_processEvents:
                            QApplication.processEvents()
                    except self.UserInterrupt:
                        if DEBUG:
                            prints(f'Action Chains: Aborting chain ({self.chain_name})')
                        break
                    except Exception as e:
                        if DEBUG:
                            prints(f'Action Chains: action "{action_name}" terminated with exception: {e}')
                        import traceback
                        traceback.print_exc()
                        raise e
                    finally:
                        # reset previous action start time
                        self.previous_action_start_time = chain_link['start_time']

            if self.jobify:
                # refresh_gui() is a job callback using Dispatcher to ensure
                # it is run from the main gui thread
                pass
            else:
                self.refresh_gui()

            end = now()
            if DEBUG:
                prints(f'Action Chains: chain ({self.chain_name}) finished in: {end - self.chain_start_time}')
        finally:
            if self.jobify:
                pass
            else:
                QApplication.restoreOverrideCursor()
                self.gui.status_bar.removeWidget(pbar)
                pbar.destroy()
            self.plugin_action.chainStack.pop(-1)
            # persist the chain_iteration var, we defered to this point to avoid test initializations adding to iterations
            chain_iteration = getattr(self, 'chain_iteration', 1)
            self.plugin_action.chain_iterations[self.chain_name] = chain_iteration
            # re-initialize chain vars for a possible next runs (in case the chain object initialized by extrenal action or plugin and deleted)
            self._initialize_vars(chain_vars={})
        return True

    def stop_if_canceled(self, abort):
        if abort.is_set():
            if self.pbar:
                self.gui.status_bar.removeWidget(self.pbar)
                self.pbar.destroy()
            raise self.UserInterrupt

    def _event_args_vars(self):
        event_args = []
        try:
            for arg in self.event_args:
                if isinstance(arg, (tuple, list, set)):
                    arg = [str(x).strip() for x in arg]
                    arg = ':::'.join(arg)
                else:
                    try:
                        arg = str(arg)
                    except:
                        continue
                event_args.append(arg)
        except Exception as e:
            import traceback
            print(traceback.format_exc())
        return event_args

    def _initialize_vars(self, chain_vars={}):
        # as opposed to self.chain_vars, _book_vars is per book vars (book_id lookup)
        # it has to be copied outsite this object, because it is accessed by template functions:
        # book_vars, set_book_vars, that needs to find it in a permanat storage.
        # It also have to be cleaned by re-assigning to remove any leftover values from previous chains
        _book_vars = chain_vars.get('_book_vars', defaultdict(dict))
        self.plugin_action.book_vars = _book_vars
        #
        self.chain_vars = chain_vars
        # chain iteration is saved, will be presisted only if the chain runs
        self.chain_iteration = self.plugin_action.chain_iterations[self.chain_name] + 1
        # Event variables {
        event_name = ''
        event_args = []
        if self.event:
            event_name = self.event.event_name
            event_args = self._event_args_vars()
        # }
        # Allow _variant_argument to be passed by chain caller
        default_variant_argument = chain_vars.get('_variant_argument', '')
        vars_dict = {
            '_chain_name': self.chain_name,
            '_chain_iteration': str(self.chain_iteration),
            'location': self.plugin_action.loc,
            '_variant_argument': self.chain_config.get('variant_argument', default_variant_argument),
            '_event_name': event_name,
            '_event_args': ','.join(event_args),
            '_book_vars': _book_vars
        }
        self.set_chain_vars(vars_dict)

    def set_chain_vars(self, vars_dict):
        for k, v in vars_dict.items():
            if k.startswith('_') or ( isinstance(k, (str)) and isinstance(v, (str)) ):
                self.chain_vars[k] = v
            else:
                if DEBUG:
                    prints(f'Action Chains: variable ({k}:{v}) cannot be set, variables must be of string type')

    def set_book_var(self, book_id, var_name, var_value):
        self.chain_vars['_book_vars'][book_id][var_name] = var_value

    def set_book_vars(self, book_id, vars_dict):
        for k, v in vars_dict.items():
            if isinstance(k, (str)) and isinstance(v, (str)):
                self.chain_vars['_book_vars'][book_id][k] = v
            else:
                if DEBUG:
                    prints(f'Action Chains: variable ({k}:{v}) cannot be set, variables must be of string type')

    def get_book_var(self, book_id, var_name, default=None):
        return self.chain_vars['_book_vars'].get(book_id, {}).get(var_name, default)

    def update_vars_for_action(self, chain_link):
        vars_dict = {
            '_action_name': chain_link['action_name'],
            '_action_comment': chain_link.get('comment', ''),
            '_action_index': str(self.current_step),
            '_previous_action_start_time': format_date(self.previous_action_start_time, 'iso')
        }
        self.set_chain_vars(vars_dict)

    def get_book_var(self, book_id, var_name, default=None):
        return self.chain_vars['_book_vars'].get(book_id, {}).get(var_name, default)

    def set_book_var(self, book_id, var_name, var_value):
        self.chain_vars['_book_vars'][book_id][var_name] = var_value

    def check_conditions(self, condition_settings, global_vars={}, mi=None):
        if not global_vars:
            global_vars = self.chain_vars
        
        if not condition_settings.get('template'):
            # No conditions set, the chain/action can run
            return True
        else:
            is_true = check_conditions(self.plugin_action, condition_settings, mi=mi, global_vars=global_vars)
            return is_true

    def check_chain_conditions(self, global_vars={}):
        condition_settings = self.chain_config.get('condition_settings', {})
        return self.check_conditions(condition_settings, global_vars, mi=self.mi)

    def check_action_conditions(self, condition_settings, global_vars={}):
        return self.check_conditions(condition_settings, global_vars)

    def evaluate_template(self, template, book_id=None, template_functions=None):
        if book_id:
            mi = self.db.new_api.get_proxy_metadata(book_id)
        else:
            mi = get_metadata_object(self.gui)

        if not template_functions:
            template_functions = self.template_functions
        template_output = get_template_output(template, mi, TEMPLATE_ERROR, mi,
                                             global_vars=self.chain_vars,
                                             template_functions=template_functions,
                                             context_attrs={"chain": self})
        return template_output

    @property
    def template_functions(self):
        return self.plugin_action.template_functions

    def refresh_gui(self, *args):
        do_refresh = self.chain_config['chain_settings'].get('refresh_gui', True)
        if do_refresh and not (self.event or self.chain_caller):
            db_last_modified = as_local_time(self.gui.current_db.last_modified())
            db_chain_modified = db_last_modified > self.start_time
            if DEBUG:
                prints(f'Action Chains: Is db modified by chain: {db_chain_modified}')
            
            if db_chain_modified:
                lm_map = {book_id: self.db.new_api.field_for('last_modified', book_id) for book_id in self.db.all_ids() }
                refresh_books = [ book_id for book_id, lm in lm_map.items() if (lm and lm > self.start_time ) ]
                cr = self.gui.library_view.currentIndex().row()
                self.gui.library_view.model().refresh_ids(refresh_books, cr)
                self.gui.tags_view.recount()
                #self.gui.current_view().resort()

    def run(self, check_conditions=True,
                  print_errors=False):
        # check chains for errors
        is_chain_valid = self.validate()
        if is_chain_valid is not True:
            msg, details = is_chain_valid
            if print_errors:
                return error_dialog(self.gui,
                                    msg,
                                    details,
                                    show=True)
            else:
                if DEBUG:
                    print(f'Action Chains: {msg}: {details}')
                    return

        # disabled chains can have keyboard shortcut, so check chain before running
        if check_conditions:
            do_run = self.check_chain_conditions()
            if not do_run:
                name = self.chain_config['menuText']
                tooltip = self.chain_config.get('condition_settings', {}).get('tooltip', '')
                msg = _(f'Action Chains: conditions for chain ({name}) are not met.\ntooltip: {tooltip}')
                if self.event:
                    self.gui.status_bar.showMessage(msg)
                    if DEBUG:
                        prints(msg)
                else:
                    error_dialog(self.gui,
                                _('Cannot run chain'),
                                msg,
                                show=True)
                return

        if self.jobify:
            job = ThreadedJob(
                'run chain',
                _(f'Run chain: {self.chain_name}'),
                self.run_threaded, (), {}, Dispatcher(self.refresh_gui))
            self.gui.job_manager.run_threaded_job(job)
            self.gui.status_bar.show_message(_('Chain started'), 3000)
        else:
            self._run_loop()

    def run_threaded(self, log=None, abort=None, notifications=None):
        res = False
        try:
            res = self._run_loop(log, abort, notifications=notifications)
        except Exception as e:
            import traceback
            traceback.print_exc()
            log.error('Exception when running chain:', e)
            pass
        if res:
            log.warn('  Chain ran successfully')
        else:
            log.error('  Failed to complete chain')
        return res

    def scope(self):
        chain_link = self.chain_links[self.current_step]
        scope_config = chain_link.get('scope_settings', {})
        scope_manager = ScopeWrapper(self.plugin_action, scope_config, self)
        return scope_manager

class StatusBarProgress(QWidget):

    def __init__(self, chain_name, maximum, minimum=0, cancelable=True, abort=Event(), hide_cancel=True):
        QWidget.__init__(self)
        self._layout = l = QHBoxLayout()
        self.setLayout(l)
        self.pb = QProgressBar(self)
        l.addWidget(self.pb)
        self.pb.setRange(minimum, maximum)
        self.pb.setFormat(_(f"(Action Chains) {chain_name}: %p % (%v of %m)"))
        self.cancel_button = QPushButton(_('Abort'))
        self.cancel_button.clicked.connect(self._on_canceled_clicked)
        l.addWidget(self.cancel_button)
        if not cancelable:
            self.cancel_button.setVisible(False)
        self.cancelable = cancelable
        self.abort = abort
        if hide_cancel:
            self.cancel_button.hide()

    def _on_canceled_clicked(self, *args):
        self.abort.set()
        self.cancel_button.setDisabled(True)
        self.cancel_button.setText(_('Aborting...'))

    def destroy(self):
        #self.hide()
        self.setParent(None)
        self.deleteLater()
