#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'


from qt.core import (QApplication, Qt, QModelIndex, QCheckBox, QGroupBox,
                     QRadioButton, QVBoxLayout)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog
from calibre.ebooks.metadata.book.formatter import SafeFormat
from calibre.utils.search_query_parser import ParseException

from calibre_plugins.action_chains.actions.base import ChainAction
from calibre_plugins.action_chains.templates import check_template, get_metadata_object
from calibre_plugins.action_chains.templates.dialogs import TemplateBox

try:
    load_translations()
except NameError:
    prints("ActionsChain::action/template_search.py - exception when loading translations")

class TemplateSearchWidget(TemplateBox):
    def __init__(self, parent, plugin_action, action, name, title):
        self.plugin_action = plugin_action
        self.action = action
        self.gui = plugin_action.gui
        self.db = self.gui.current_db
        mi = get_metadata_object(self.gui)
        TemplateBox.__init__(
                self,
                parent,
                plugin_action,
                template_text='',
                placeholder_text = _('Depending on the option selected below, '
                                     'the template entered here should return a calibre search, or a comma separated list of book_ids'),
                mi=mi
            )
        self.setWindowTitle(title)

        self.select_chk = QCheckBox(_('Select all books resulting from this template'))
        self.select_chk.setChecked(True)
        self.user_layout_1.addWidget(self.select_chk)
        self.select_chk.stateChanged.connect(self._on_select_chk_change)

        options_groupbox = QGroupBox(_('Options'))
        self.user_layout_3.addWidget(options_groupbox)
        options_groupbox_layout = QVBoxLayout()
        options_groupbox.setLayout(options_groupbox_layout)
        search_opt = self.search_opt = QRadioButton(_('Template output is a calibre search'))
        search_opt.setChecked(True)
        ids_opt = self.ids_opt = QRadioButton(_('Template output is a list of book ids'))
        options_groupbox_layout.addWidget(search_opt)
        options_groupbox_layout.addWidget(ids_opt)
        self._on_select_chk_change()

    def _on_select_chk_change(self):
        state = self.select_chk.isChecked()
        if not state:
            self.search_opt.setChecked(True)
        self.ids_opt.setEnabled(state)

    def load_settings(self, settings):
        if settings:
            template = settings['template']
            self.textbox.insertPlainText(template)
            self.select_chk.setChecked(settings['select_books'])
            if settings.get('opt') == 'ids':
                self.ids_opt.setChecked(True)

    def save_settings(self):
        settings = {}
        settings['template'] = str(self.textbox.toPlainText()).rstrip()
        if self.ids_opt.isChecked():
            settings['opt'] = 'ids'
        elif self.search_opt.isChecked():
            settings['opt'] = 'search'
        settings['select_books'] = self.select_chk.isChecked()
        return settings

    def accept(self):
        self.settings = self.save_settings()
        # validate settings
        is_valid = self.action.validate(self.settings)
        if is_valid is not True:
            msg, details = is_valid
            error_dialog(
                self,
                msg,
                details,
                show=True
            )
            return
        TemplateBox.accept(self)

class TemplateSearchAction(ChainAction):

    name = 'Search Using Templates'
    _is_builtin = True

    def run(self, gui, settings, chain):
        db = gui.current_db
        template = settings['template']
        template_output = chain.evaluate_template(template, book_id=None)
        if settings['opt'] == 'search':
            gui.search.clear()
            gui.search.set_search_string(template_output)

            # we select books returned by this search, instead of selecting all books in current view
            # because if the search is invalid this can select all books in library.
            try:
                book_ids = db.data.search_getting_ids(template_output, '')
            except ParseException:
                # empty book_ids will make sure previously selected items will be de-selected
                # since the search failed and did not change the view
                book_ids = []
        elif settings['opt'] == 'ids':
            try:
                book_ids = [int(x.strip()) for x in template_output.split(',')]
            except:
                book_ids = []

        if settings['select_books']:
            # get the books in current view to select only within current limitation
            row_count = gui.library_view.model().rowCount(QModelIndex())
            current_view_rows = list(range(row_count))
            current_view_book_ids = list(map(gui.library_view.model().id, current_view_rows))
            
            book_ids = set(book_ids).intersection(current_view_book_ids)
        else:
            book_ids = []
        gui.library_view.select_rows(book_ids)

    def run_as_job(self, settings):
        return False

    def validate(self, settings):
        gui = self.plugin_action.gui
        db = gui.current_db
        if not settings:
            return (_('Settings Error'), _('You must configure this action before running it'))
        mi = get_metadata_object(gui)
        is_template_valid = check_template(settings['template'], self.plugin_action, print_error=False)
        if is_template_valid is not True:
            return is_template_valid
        return True

    def config_widget(self):
        return TemplateSearchWidget

