#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import copy
import os

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                     QLabel, QGroupBox, QPushButton, QLineEdit, QToolButton,
                     QRadioButton, QDialog, QDialogButtonBox, QCheckBox, QPlainTextEdit)

from calibre import prints
from calibre.constants import DEBUG, iswindows
from calibre.gui2 import error_dialog, question_dialog
from calibre.gui2.dialogs.message_box import MessageBox, Icon
from calibre.ptempfile import PersistentTemporaryDirectory
from calibre.gui2.widgets2 import Dialog

from calibre_plugins.action_chains.actions.base import ChainAction

try:
    load_translations()
except NameError:
    prints("ActionChains::actions/save_restore_format.py - exception when loading translations")


def copy_to_format(gui, book_ids, source_fmt, destination_fmt, remove_source=False, confirm_overwrite=True):
    db = gui.current_db
    tdir = PersistentTemporaryDirectory('_copy_format') 
    if confirm_overwrite:
        QApplication.setOverrideCursor(Qt.ArrowCursor)
        try:
            if not overwrite_confirmation(gui, book_ids, source_fmt, destination_fmt):
                return
        finally:
            QApplication.restoreOverrideCursor()

    for book_id in book_ids:
        fmts_string = db.formats(book_id, index_is_id=True)
        if fmts_string:
            available_fmts = [ fmt.strip() for fmt in fmts_string.split(',') ]
        else:
            available_fmts = []
        if source_fmt.upper() in available_fmts:
            path = os.path.join(tdir, str(book_id) + '.' + destination_fmt)
            with open(path, 'wb') as f:
                db.copy_format_to(book_id, source_fmt, f, index_is_id=True)
                db.new_api.add_format(book_id, destination_fmt, path)
            if remove_source:
                db.remove_format(book_id, source_fmt, index_is_id=True, notify=False)
            try:
                os.remove(path)
            except:
                pass

def overwrite_confirmation(gui, book_ids, source_fmt, destination_fmt):
    db = gui.current_db
    l = []
    for book_id in book_ids:
        fmts_string = db.formats(book_id, index_is_id=True)
        if fmts_string:
            available_fmts = [ fmt.strip() for fmt in fmts_string.split(',') ]
        else:
            available_fmts = []
        if not source_fmt.upper() in available_fmts:
            continue
        if destination_fmt.upper() in available_fmts:
            l.append(f'{db.title(book_id, index_is_id=True)} ({book_id})')
    if l:
        msg = _('Copy to format will overwrite already existing files for some books. '
                'Are you sure you want to proceed?')
        details = '\n'.join(l)
        msg = _('Some books have the same format you are trying to copy. Are you sure you want to overwrite existing formats?')
        return question_dialog(gui, _('Are you sure?'), msg, det_msg=details)
    return True

    return d.exec_() == d.Accepted

class OuterConfigWidget(QWidget):
    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.gui = plugin_action.gui
        self.db = self.gui.current_db
        self._init_controls()

    def _init_controls(self):

        l = self.l = QVBoxLayout()
        self.setLayout(l)


        opt_l = QHBoxLayout()
        l.addLayout(opt_l)

        self.predefined_opt = QRadioButton(_('Predefind settings'))
        self.predefined_opt.setChecked(True)
        opt_l.addWidget(self.predefined_opt)
        
        self.runtime_opt = QRadioButton(_('Ask for setting at runtime'))
        opt_l.addWidget(self.runtime_opt)

        for opt in [self.predefined_opt, self.runtime_opt]:
            opt.toggled.connect(self._on_opt_toggled)  

        self.widget_groupbox = QGroupBox(_('Settings'))
        l.addWidget(self.widget_groupbox)
        widget_groupbox_l = QVBoxLayout()
        self.widget_groupbox.setLayout(widget_groupbox_l)
        self.widget = InnerConfigWidget(self.plugin_action)
        widget_groupbox_l.addWidget(self.widget)
        
        self.setMinimumSize(200,300)
        l.addStretch(1)
        self._on_opt_toggled()

    def _on_opt_toggled(self, *args):
        self.widget_groupbox.setEnabled(self.predefined_opt.isChecked())

    def load_settings(self, settings):
        if settings:
            if settings['opt'] == 'predefined':
                self.predefined_opt.setChecked(True)
                self.widget.load_settings(settings)
            elif settings['opt'] == 'runtime':
                self.runtime_opt.setChecked(True)

    def save_settings(self):
        settings = {}
        if self.predefined_opt.isChecked():
            settings['opt'] = 'predefined'
            widget_settings = self.widget.save_settings()
            settings.update(widget_settings)
        elif self.runtime_opt.isChecked():
            settings['opt'] = 'runtime'
        return settings

class InnerConfigWidget(QWidget):
    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.gui = plugin_action.gui
        self.db = self.gui.current_db
        self._init_controls()

    def _init_controls(self):

        l = self.l = QGridLayout()
        self.setLayout(l)

        source_label = QLabel(_('Source format:'))
        self.source_fmt_ledit = QLineEdit()
        l.addWidget(source_label, 0, 0, 1, 1)
        l.addWidget(self.source_fmt_ledit, 0, 1, 1, 1)

        dest_label = QLabel(_('Destination format:'))
        self.dest_fmt_ledit = QLineEdit()
        l.addWidget(dest_label, 1, 0, 1, 1)
        l.addWidget(self.dest_fmt_ledit, 1, 1, 1, 1)
        
        self.confirm_overwrite_chk = QCheckBox(_('Confirm overwriting existing format confirmation dialog'))
        self.confirm_overwrite_chk.setChecked(True)
        l.addWidget(self.confirm_overwrite_chk, 2, 0, 1, 2)

        self.remove_source_chk = QCheckBox(_('Remove source format after copying'))
        self.remove_source_chk.setChecked(False)
        l.addWidget(self.remove_source_chk, 3, 0, 1, 2)

    def load_settings(self, settings):
        if settings:
            self.source_fmt_ledit.setText(settings['source_fmt'])
            self.dest_fmt_ledit.setText(settings['destination_fmt'])
            self.confirm_overwrite_chk.setChecked(settings['confirm_overwrite'])
            self.remove_source_chk.setChecked(settings['remove_source'])

    def save_settings(self):
        settings = {}
        settings['source_fmt'] = self.source_fmt_ledit.text()
        settings['destination_fmt'] = self.dest_fmt_ledit.text()
        settings['confirm_overwrite'] = self.confirm_overwrite_chk.isChecked()
        settings['remove_source'] = self.remove_source_chk.isChecked()
        return settings


class SettingsDialog(Dialog):

    def __init__(self, name, parent, plugin_action, widget_cls, title=_('Settings')):
        self.plugin_action = plugin_action
        self.widget_cls = widget_cls
        Dialog.__init__(self, title, name, parent)

    def setup_ui(self):
        self.widget = self.widget_cls(self.plugin_action)
        l = QVBoxLayout()
        self.setLayout(l)
        l.addWidget(self.widget)
        l.addWidget(self.bb)

    def load_settings(self, settings):
        self.widget.load_settings(settings)

    def save_settings(self):
        return self.widget.save_settings()
    
    def accept(self):
        self.settings = self.save_settings()
        Dialog.accept(self)

class SaveRestoreFormatAction(ChainAction):

    name = 'Save/Restore Formats'
    _is_builtin_ = True
    support_scopes = True
            
    def run(self, gui, settings, chain):
        book_ids = chain.scope().get_book_ids()        
        db = gui.current_db
        if settings['opt'] == 'runtime':
            d = SettingsDialog('action-chains-save-restore-format-dialog', gui, gui.iactions['Action Chains'], InnerConfigWidget)
            QApplication.setOverrideCursor(Qt.ArrowCursor)
            try:
                if d.exec_() == d.Accepted:
                    settings = d.settings
                else:
                    return
            finally:
                QApplication.restoreOverrideCursor()
        source_fmt = settings['source_fmt']
        destination_fmt = settings['destination_fmt']
        remove_source = settings['remove_source']
        confirm_overwrite = settings['confirm_overwrite']
        copy_to_format(gui, book_ids, source_fmt, destination_fmt, remove_source=remove_source, confirm_overwrite=confirm_overwrite)

    def validate(self, settings):
        if not settings:
            return (_('Settings Error'), _('You must configure this action before running it'))
        if settings.get('opt') == 'predefined':
            if not settings.get('source_fmt'):
                return (_('No format'), _('You must choose source format'))
            if not settings.get('destination_fmt'):
                return (_('No format'), _('You must choose destination format'))
        return True

    def config_widget(self):
        return OuterConfigWidget
