#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2022, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import os

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                     QLabel, QComboBox, QCheckBox, QLineEdit, QPushButton, QDialog, QSize,
                     QFontMetrics)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog
from calibre.gui2.dialogs.confirm_delete import confirm
from calibre.gui2.tweak_book.widgets import Dialog
from calibre.gui2.complete2 import EditWithComplete
from calibre.gui2.tweak_book.editor.text import TextEdit
from calibre.ebooks.oeb.polish.container import get_container


from calibre_plugins.action_chains.actions.base import ChainAction
from calibre_plugins.action_chains.modules import compile_code
from calibre_plugins.action_chains.common_utils import reverse_lookup

try:
    load_translations()
except NameError:
    prints("ActionChains::actions/actions/code.py - exception when loading translations")

CODE_TEMPLATE = '''
def run(gui, settings, chain):
    # Enter you code here
    pass
'''

FORMAT_TEMPLATE = '''
def run_for_book(gui, settings, chain, container):
    # Enter you code here
    # Note: The function must return True for whatever changes you make to the book to be saved
    pass
'''

class CodeEditor(Dialog):

    def __init__(self, parent, plugin_action, action, name, title):
        #self.template = CODE_TEMPLATE
        self.language = 'python'
        #self.show_name = False
        self.action = action
        self.modes = {
            'library': _('Run Python Code'),
            'book': _('Run Python Code For Book Format')
        }
        self.template = {
            'library': CODE_TEMPLATE,
            'book': FORMAT_TEMPLATE
        }
        Dialog.__init__(self, name, title, parent=parent)

    def setup_ui(self):
        self.l = l = QVBoxLayout(self)
        self.h = h = QHBoxLayout()
        l.addLayout(h)

        self.mode_combo = QComboBox()
        self.mode_combo.addItems(list(self.modes.values()))
        self.mode_combo.currentIndexChanged.connect(self._on_mode_change)
        h.addWidget(self.mode_combo, 1)

        self.format_combo = QComboBox()
        self.format_combo.addItems(['epub','azw3'])
        h.addWidget(self.format_combo)
        self.format_combo.setVisible(False)

        self.la3 = la = QLabel(_('&Code:'))
        self.source_code = TextEdit(self)
        self.source_code.load_text(self.template['library'], self.language)
        la.setBuddy(self.source_code)
        l.addWidget(la), l.addWidget(self.source_code)

        self.la2 = la = QLabel()
        la.setOpenExternalLinks(True)
        l.addWidget(la)

        l.addWidget(self.bb)

    def sizeHint(self):
        fm = QFontMetrics(self.font())
        return QSize(fm.averageCharWidth() * 120, 600)

    @property
    def source(self):
        return self.source_code.toPlainText()

    def _on_mode_change(self):
        ct = self.mode_combo.currentText()
        mode = reverse_lookup(self.modes, ct)
        previous_mode = 'book' if mode == 'library' else 'library'
        self.template[previous_mode] = self.source
        self.source_code.setPlainText(self.template[mode])
        self.format_combo.setVisible(mode == 'book')

    def load_settings(self, settings):
        if settings:
            mode = settings.get('mode', 'library')
            self.mode_combo.setCurrentText(self.modes[mode])
            self.source_code.setPlainText(settings['source'] or self.template[mode])
            if mode == 'book':
                self.format_combo.setCurrentText(settings['book_format'])
                self.format_combo.setVisible(True)

    def save_settings(self):
        settings = {}
        mode = reverse_lookup(self.modes, self.mode_combo.currentText())
        settings['mode'] = mode
        settings['source'] = self.source
        if mode == 'book':
            settings['book_format'] = self.format_combo.currentText()
        return settings

    def accept(self):
        source = self.source
        try:
            mod = compile_code(source)
        except Exception as err:
            return error_dialog(self, _('Invalid Python code'), _(
                f'The code you created is not valid Python code, with error: {err}'), show=True)

        mode = reverse_lookup(self.modes, self.mode_combo.currentText())
        func_name = 'run' if mode == 'library' else 'run_for_book'
        try:
            func = mod[func_name]
        except Exception as err:
            return error_dialog(self, _('Function not defined'), _(
                f'You must define a functino called "{func_name}"'), show=True)
        self.settings = self.save_settings()
        # validate settings
        is_valid = self.action.validate(self.settings)
        if is_valid is not True:
            msg, details = is_valid
            error_dialog(
                self,
                msg,
                details,
                show=True
            )
            return
        Dialog.accept(self)

def create_container(book_path):
    if not book_path:
        raise FileNotFoundError
    if not os.access(book_path, os.W_OK):
        raise PermissionError
    container = get_container(book_path)
    return container

class CodeAction(ChainAction):

    name = 'Run Python Code'
    _is_builtin = True
    support_scopes = True

    def run(self, gui, settings, chain):
        source = settings['source']
        mod = compile_code(source, 'module')
        mode = settings.get('mode', 'library')
        func_name = 'run' if mode == 'library' else 'run_for_book'
        func = mod.get(func_name)
        if settings.get('mode', 'library') == 'library':
            return func(gui, settings, chain)
        else:
            updated = set()
            db = gui.current_db
            book_format = settings.get('book_format', 'epub')
            book_ids = chain.scope().get_book_ids()

            if len(book_ids) == 0:
                return

            for book_id in book_ids:
                book_title = db.title(book_id, index_is_id=True)
                print(f'Processing book: {book_title}')
                fmts_string = db.formats(book_id, index_is_id=True)
                if fmts_string:
                    available_fmts = [ fmt.strip() for fmt in fmts_string.split(',') ]
                    if book_format.upper() in available_fmts and ( book_format.upper() in ['EPUB', 'AZW3'] ):
                        book_path = db.format_abspath(book_id, book_format, index_is_id=True)

                        container =  create_container(book_path)
                        container.book_id = book_id
                        res = func(gui, settings, chain, container)
                        if res is True:
                            container.commit()
                            updated.add(book_id)
                            x = db.new_api.format_metadata(book_id, book_format.upper(), update_db=True)
            db.update_last_modified(list(updated))


    def validate(self, settings):
        if not settings:
            return (_('Settings Error'), _('You must configure this action'))
        source = settings['source']
        mod = compile_code(source, 'module')
        validate_func = mod.get('validate', None)
        if validate_func:
            return validate_func(settings)
        return True

    def config_widget(self):
        return CodeEditor
