#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__ = 'GPL v3'
__copyright__ = '2020, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from functools import partial
import re
from uuid import uuid4

from qt.core import (QApplication, Qt, QWidget, QGridLayout, QHBoxLayout, QVBoxLayout,
                      QDialog, QDialogButtonBox, QMenu, QPainter, QPoint, QPixmap,
                      QSize, QIcon, QTreeWidgetItem, QTreeWidget, QAbstractItemView,
                      QGroupBox, QCheckBox, QLabel)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog
from calibre.gui2.dialogs.progress import ProgressDialog
from calibre.gui2.jobs import JobsDialog

from calibre_plugins.action_chains.actions.base import ChainAction
from calibre_plugins.action_chains.common_utils import get_icon, responsive_wait, responsive_wait_until, unfinished_job_ids

try:
    load_translations()
except NameError:
    prints("ActionChains::actions/calibre_actions.py - exception when loading translations")

# Code for this is based on code from the Favourites Plugin by Grant Drake AKA kiwidude

EXCLUDED_PLUGINS = [
#    'Add Books',
#    'Add To Library',
#    'Remove Books',
#    'Convert Books',
#    'Choose Library',
#    'Favourites Menu',
#    'Choose Library'
]

ICON_SIZE = 32

def translation_reverse_lookup(translated_string):
    '''
    get untranslated_string (msgid) from translated_string (msgstr) by
    doing a reverse lookup in gettext catalog
    '''
    if not hasattr(_.__self__, '_catalog'):
        return translated_string
    if not hasattr(translation_reverse_lookup, 'cache'):
        try:
            translation_reverse_lookup.cache = {v: k for k, v in _.__self__._catalog.items()}
        except:
            #AttributeError: 'NullTranslations' object has no attribute '_catalog'
            translation_reverse_lookup.cache = {}
    ans = translation_reverse_lookup.cache.get(translated_string, translated_string)
    if isinstance(ans, tuple):
          ans  = ans[1]
    return ans

def get_safe_title(action):
    if hasattr(action, 'favourites_menu_unique_name'):
        text = str(action.favourites_menu_unique_name)
    else:
        text = str(translation_reverse_lookup(str(action.text())))
    return text.replace('&&', '—').replace('&', '').replace('—', '&')

def get_title(action, plugin_name):
    '''
    New method that wraps get_safe_title to handle some special cases
    '''
    safe_title = get_safe_title(action)
    # handle a special case of "Show marked books": https://www.mobileread.com/forums/showpost.php?p=4062478&postcount=46
    if safe_title.startswith('Show marked book') and ( plugin_name == 'Mark Books'):
        safe_title = 'Show marked books'
    return safe_title

def _find_action_for_menu(parent, paths, plugin_name):
    if parent is not None:
        # call translation_reverse_lookup() to convert older config that used to store
        # entries in interface language
        translated_find_text = paths[0]
        find_text = translation_reverse_lookup(translated_find_text)
        for ac in QMenu.actions(parent):
            if ac.isSeparator():
                continue
            #print('Looking at action:',str(ac.text()))
            safe_title = get_title(ac, plugin_name)
            if safe_title == find_text:
                if len(paths) == 1:
                    return ac
                return _find_action_for_menu(ac.menu(), paths[1:], plugin_name)

def get_qaction_from_settings(gui, settings):
    setting = settings[0]
    if setting is None:
        return
    ac = None
    paths = list(setting['path'])
    plugin_name = paths[0]
    is_device_only_plugin = False

    if plugin_name == 'Location Manager':
        # Special case handling since not iaction instances
        is_device_only_plugin = True
        paths = paths[1:]
        for loc_action in gui.location_manager.all_actions[1:]:
            translated_text = str(loc_action.text())
            untranslated_text = translation_reverse_lookup(translated_text)
            if untranslated_text == paths[0]:
                if len(paths) > 1:
                    # This is an action on the menu for this plugin or its submenus
                    ac = _find_action_for_menu(loc_action.menu(), paths[1:], plugin_name)
                else:
                    # This is a top-level plugin being added to the menu
                    ac = loc_action
                return ac
    else:
        iaction = gui.iactions.get(plugin_name, None)
        if iaction is not None:
            is_device_only_plugin = 'toolbar' in iaction.dont_add_to and 'toolbar-device' not in iaction.dont_add_to
            if len(paths) > 1:
                # This is an action on the menu for this plugin or its submenus
                ac = _find_action_for_menu(iaction.qaction.menu(), paths[1:], plugin_name)
            else:
                # This is a top-level plugin
                ac = iaction.qaction
            return ac
        else:
            return None

class Item(QTreeWidgetItem):
    pass


def validate(gui, settings, check_device_active=True):
    if not settings:
        return (_('Settings Error'), _('You must configure this action before running it'))
    selection = settings['selection']
    if not selection:
        return _('Settings Error'), _('You must check one entry')
    ac = get_qaction_from_settings(gui, selection)
    paths = list(selection[0]['path'])
    plugin_name = paths[0]
    if not ac:
        return (_('Settings Error'), _(f'Menu entry: {paths[-1]} (plugin_name: {plugin_name}) cannot be found'))
    if plugin_name == 'Location Manager':
        if check_device_active:
            if not gui.location_manager.has_device:
                return (_('Device Error'), _(f'Menu entry: {paths[-1]} (plugin_name: {plugin_name}) This action is only available when device view is active'))
    return True

class CalibreActionsWidget(QWidget):
    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.gui = plugin_action.gui
        self.db = self.gui.current_db
        # used to keep track of selected item and allow only one selection
        self.checked_item = None
        
        self._init_controls()

    def _init_controls(self):
        self.blank_icon = get_icon('blank.png')
        l = self.l = QVBoxLayout()
        self.setLayout(l)

        opts_groupbox = QGroupBox()
        l.addWidget(opts_groupbox)
        opts_groupbox_layout = QVBoxLayout()
        opts_groupbox.setLayout(opts_groupbox_layout)
        cursor_chk = self.cursor_chk = QCheckBox(_('Disable wait cursor for this action.'))
        opts_groupbox_layout.addWidget(cursor_chk)
        cursor_chk.setChecked(False)
        job_wait_chk = self.job_wait_chk = QCheckBox(_('Wait until any jobs stared by the selected action finishes.'))
        opts_groupbox_layout.addWidget(job_wait_chk)
        job_wait_chk.setChecked(False)
        progress_wait_chk = self.progress_wait_chk = QCheckBox(_('Wait until any progress bar shown by action finishes.'))
        opts_groupbox_layout.addWidget(progress_wait_chk)
        progress_wait_chk.setChecked(False)
        warning = QLabel(_('<b>Warning:</b> The above option will fail whenever the option'
                ' "Hide main window" is chosen from the tray dropown menu.'))
        warning.setWordWrap(True)
        opts_groupbox_layout.addWidget(warning)
        
        self.tv = QTreeWidget(self.gui)
        self.tv.setIconSize(QSize(ICON_SIZE, ICON_SIZE))
        self.tv.header().hide()
        self.tv.setSelectionMode(QAbstractItemView.SingleSelection)
        l.addWidget(self.tv, 1)
        self.tv.itemChanged.connect(self._tree_item_changed)
           
        self.fav_menus = []
        # Rebuild this into a map for comparison purposes
        lookup_menu_map = self._build_lookup_menu_map(self.fav_menus)
        self._populate_actions_tree(lookup_menu_map)

    def _build_lookup_menu_map(self, fav_menus):
        m = {}
        for fav_menu in fav_menus:
            if fav_menu is None:
                continue
            path = fav_menu['path']
            plugin = path[0]
            if plugin not in m:
                m[plugin] = []
            fav_menu['paths_text'] = '|'.join(path[1:])
            m[plugin].append(fav_menu)
        return m

    def _get_scaled_icon(self, icon):
        if icon.isNull():
            return self.blank_icon
        # We need the icon scaled to 16x16
        src = icon.pixmap(ICON_SIZE, ICON_SIZE)
        if src.width() == ICON_SIZE and src.height() == ICON_SIZE:
            return icon
        # Need a new version of the icon
        pm = QPixmap(ICON_SIZE, ICON_SIZE)
        pm.fill(Qt.transparent)
        p = QPainter(pm)
        p.drawPixmap(QPoint(int((ICON_SIZE - src.width()) / 2), int((ICON_SIZE - src.height()) / 2)), src)
        p.end()
        return QIcon(pm)

    def set_checked_state(self, item, col, state):
        item.setCheckedState(col, state)
        if state == Qt.Checked:
            self.checked_item = item
        else:
            self.checked_item = None

    def _populate_actions_tree(self, lookup_menu_map):
        # Lets re-sort the keys so that items will appear on screen sorted
        # by their display name (not by their key)
        skeys_map = {}
        for plugin_name, iaction in self.gui.iactions.items():
            if plugin_name in [self.plugin_action.name] + EXCLUDED_PLUGINS:
                continue
            if 'toolbar' in iaction.dont_add_to and 'toolbar-device' in iaction.dont_add_to:
                print(('Not adding:', plugin_name))
                continue
            display_name = str(iaction.qaction.text())
            skeys_map[display_name] = (plugin_name, iaction.qaction)
        # Add a special case item for the location manager
        skeys_map['Location Manager'] = ('Location Manager', None)

        self.top_level_items_map = {}
        for display_name in sorted(skeys_map.keys()):
            plugin_name, qaction = skeys_map[display_name]
            possible_menus = lookup_menu_map.get(plugin_name, [])

            # Create a node for our top level plugin name
            tl = Item()
            translated_display_name = _(display_name)
            tl.setText(0, translated_display_name)
            tl.setData(0, Qt.UserRole, plugin_name)

            if plugin_name == 'Location Manager':
                # Special case handling
                tl.setFlags(Qt.ItemIsEnabled)
                tl.setCheckState(0, Qt.PartiallyChecked)
                tl.setIcon(0, self._get_scaled_icon(get_icon('reader.png')))
                # Put all actions except library within this node.
                actions = self.gui.location_manager.all_actions[1:]
                self._populate_action_children(actions, tl, possible_menus, [], plugin_name,
                                               is_location_mgr_child=True)
            else:            
                # Normal top-level checkable plugin iaction handling
                tl.setFlags(Qt.ItemIsEnabled | Qt.ItemIsUserCheckable)
                tl.setCheckState(0, Qt.Unchecked)
                tl.setIcon(0, self._get_scaled_icon(qaction.icon()))

                # Lookup to see if we have a menu item for this top-level plugin
                if possible_menus:
                    fav_menu = self._is_in_menu(possible_menus)
                    if fav_menu is not None:
                        #fav_menu['icon'] = tl.icon(0)
                        tl.setCheckState(0, Qt.Checked)
                        ####
                        self.checked_item = tl
                        ####
                m = qaction.menu()
                if m:
                    # Iterate through all the children of this node
                    self._populate_action_children(QMenu.actions(m), tl,
                                                   possible_menus, [], plugin_name)

            self.tv.addTopLevelItem(tl)
            self.top_level_items_map[plugin_name] = tl
        ####
        if self.checked_item:
            self.tv.scrollToItem(self.checked_item)
        ####

    def _populate_action_children(self, children, parent, possible_menus, paths,
                                  plugin_name, is_location_mgr_child=False):
        for ac in children:
            if ac.isSeparator():
                continue
            if not ac.isVisible() and not is_location_mgr_child:
                # That is special case of location mgr visibility, since it has child
                # actions that will not be visible if device not plugged in at the
                # moment but we want to always be able to configure them.
                continue
            text = get_title(ac, plugin_name)

            it = Item(parent)
            translated_text = _(text)
            it.setText(0, translated_text)
            it.setFlags(Qt.ItemIsEnabled | Qt.ItemIsUserCheckable)
            it.setCheckState(0, Qt.Unchecked)
            it.setIcon(0, self._get_scaled_icon(ac.icon()))

            new_paths = list(paths)
            new_paths.append(text)
            if possible_menus:
                fav_menu = self._is_in_menu(possible_menus, new_paths)
                if fav_menu is not None:
                    #fav_menu['icon'] = it.icon(0)
                    it.setCheckState(0, Qt.Checked)
                    ####
                    self.checked_item = it
                    ####
            if ac.menu():
                self._populate_action_children(QMenu.actions(ac.menu()), it,
                                               possible_menus, new_paths, plugin_name)

    def _is_in_menu(self, possible_menus, paths=[]):
        path_text = '|'.join(paths)
        for x in range(0, len(possible_menus)):
            fav_menu = possible_menus[x]
            if fav_menu['paths_text'] == path_text:
                del possible_menus[x]
                return fav_menu
        return None

    def _tree_item_changed(self, item, column):
        # Checkstate has been changed

        is_checked = item.checkState(column) == Qt.Checked
        paths = []
        translated_display_name = _(str(item.text(column)))
        fav_menu = {'display': translated_display_name,
                    'path':    paths}
        
        temp_item = item
        while True:
            parent = temp_item.parent()
            if parent is None:
                untranslated_text = translation_reverse_lookup(temp_item.data(column, Qt.UserRole))
                paths.insert(0, untranslated_text)
                break
            else:
                untranslated_text = translation_reverse_lookup(str(temp_item.text(column)))
                paths.insert(0, untranslated_text)
            temp_item = parent

        if is_checked:
            ####
            # un-check previously checked item if any
            if self.checked_item:
                self.tv.blockSignals(True)
                self.checked_item.setCheckState(0, Qt.Unchecked)
                self.tv.blockSignals(False)
            
            self.checked_item = item
            self.fav_menus = [fav_menu]
            ####
        else:
            ####
            self.checked_item = None
            self.fav_menus = []
            ####

    def _repopulate_actions_tree(self):
        self.tv.clear()
        lookup_menu_map = self._build_lookup_menu_map(self.fav_menus)
        self._populate_actions_tree(lookup_menu_map)

    def load_settings(self, settings):
        if settings:
            self.cursor_chk.setChecked(settings.get('disable_busy_cursor', False))
            self.job_wait_chk.setChecked(settings.get('wait_jobs', False))
            self.progress_wait_chk.setChecked(settings.get('wait_progress', False))
            self.fav_menus = settings['selection']
            self._repopulate_actions_tree()

    def save_settings(self):
        settings = {}
        settings['selection'] = self.fav_menus
        settings['disable_busy_cursor'] = self.cursor_chk.isChecked()
        settings['wait_jobs'] = self.job_wait_chk.isChecked()
        settings['wait_progress'] = self.progress_wait_chk.isChecked()
        return settings

    def validate(self, settings):
        gui = self.plugin_action.gui
        return validate(gui, settings, check_device_active=False)

def get_killed_or_failed_jobs(gui, action_job_ids):
    l = set()
    for j in gui.job_manager.jobs:
        if j.id in action_job_ids:
            if j.failed or j.killed:
                l.add(j.id)
    return l

class CalibreActions(ChainAction):

    name = 'Calibre Actions'
    _is_builtin = True

    def run(self, gui, settings, chain):
        rows = gui.current_view().selectionModel().selectedRows()
        book_ids = [ gui.library_view.model().db.id(row.row()) for row in rows ]

        selection = settings['selection']
        
        db = gui.current_db

        paths = list(selection[0]['path'])
        plugin_name = paths[0]

        if DEBUG:
            prints(f'Action Chains: Calibre Actions: paths > {paths}')        

        if plugin_name == 'Location Manager':
            if not gui.location_manager.has_device:
                if DEBUG:
                    prints(f'(plugin_name: {plugin_name}) This action is only available when device view is active')
                return
        
        ac = get_qaction_from_settings(gui, selection)
        if ac:
            ###
            if DEBUG:
                prints(f'Action Chains: Calibre Actions: Found action: {ac.text()} | isEnabled: {ac.isEnabled()}')

            cursor = Qt.WaitCursor
            if settings.get('disable_busy_cursor'):
                cursor = Qt.ArrowCursor
            
            QApplication.setOverrideCursor(cursor)
            try:
                wait_jobs = settings.get('wait_jobs', False)
                wait_progress = settings.get('wait_progress', False)
                
                jobs_before_ids = unfinished_job_ids(gui)
                
                ac.trigger()
                
                if wait_jobs or wait_progress:
                    
                    # wait for progress bar and jobs spawned by action to kick in
                    responsive_wait(1)

                if wait_progress:
                    # Wait for any dialogs (e.g progress bar) activated by action to disappear
                    modal_widget = QApplication.instance().activeModalWidget()
                    try:
                        if modal_widget:
                            # Testing for the presence of active modal widget can fail if the main window is hidden from
                            # tray. We insert a uuid.
                            uuid = str(uuid4())
                            # This might raise exception of modal_dialog is deleted
                            setattr(modal_widget, '__calibre_actions_uuid', uuid)

                            def widget_from_uuid(uuid):
                                for widget in QApplication.topLevelWidgets():
                                    widget_uuid = getattr(widget, '__calibre_actions_uuid', '')
                                    if widget_uuid == uuid:
                                        return widget
                            
                            responsive_wait_until(lambda: not QApplication.instance().activeModalWidget() and not widget_from_uuid(uuid))
                    except RuntimeError as e:
                        # Catch RuntimeError: underlying C/C++ object has been deleted
                        if DEBUG:
                            prints('Action Chains: Error: Modal widget has been deleted')
                        pass
                    
                if wait_jobs:
                    # save ids of jobs started after running the action                    
                    action_job_ids = unfinished_job_ids(gui).difference(jobs_before_ids)

                    # wait for jobs to finish
                    responsive_wait_until(lambda: action_job_ids.intersection(unfinished_job_ids(gui)) == set())
                    failed_or_killed = get_killed_or_failed_jobs(gui, action_job_ids)
                    if failed_or_killed:
                        print('debug1: calibre actions: failed_or_killed: {}'.format(failed_or_killed))
            finally:
                QApplication.restoreOverrideCursor()
            ####
            
        if DEBUG:
            prints('Action Chains: Calibre Actions: Finishing run action') 

    def run_as_job(self, settings):
        return False

    def validate(self, settings):
        gui = self.plugin_action.gui
        return validate(gui, settings, check_device_active=True)

    def config_widget(self):
        return CalibreActionsWidget


