#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__   = 'GPL v3'
__copyright__ = '2021 additions by Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from qt.core import QWizard, QApplication, QWidget, QStyle

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import gprefs

import calibre_plugins.category_tags.config as cfg
from calibre_plugins.category_tags.common_utils import get_icon, debug_print
from calibre_plugins.category_tags.user_categories.page_import import ImportPage
from calibre_plugins.category_tags.user_categories.page_resolve import ResolvePage

try:
    load_translations()
except NameError:
    prints("Category Tags::user_categories/wizards.py - exception when loading translations")

class ImportCategoriesWizard(QWizard):

    def __init__(self, plugin_action, is_modal=False):
        self.gui = plugin_action.gui
        self.plugin_action = plugin_action
        QWizard.__init__(self, self.gui)
        self.unique_pref_name = 'Category Tags plugin:import categories wizard'
        self.setModal(is_modal)
        self.setWindowTitle(_('Import User Categories'))
        self.setWindowIcon(get_icon('images/category_tags.png'))
        self.setMinimumSize(600, 0)

        self.is_closed = False
        self.db = self.gui.current_db
        self.info = {}
        self.info['state'] = gprefs.get(self.unique_pref_name+':state', {})
        self.library_config = cfg.get_library_config(self.db)

        self.addPages()

        geom = gprefs.get(self.unique_pref_name, None)
        if geom is None:
            self.resize(self.sizeHint())
        else:
            self.restoreGeometry(geom)
        self.finished.connect(self._on_dialog_closing)

    def addPages(self):
        for attr, cls in [
                ('import_page',  ImportPage),
                ('resolve_page', ResolvePage)
                ]:
            setattr(self, attr, cls(self, self.gui, self.plugin_action.import_action))
            self.setPage(getattr(cls, 'ID'), getattr(self, attr))

    def _save_window_state(self):
        geom = bytearray(self.saveGeometry())
        gprefs[self.unique_pref_name] = geom

        state = {}
        state['cat_resolve_splitter_state'] = bytearray(self.resolve_page.splitter.saveState())
        state['cat_resolve_search_column_widths'] = self.resolve_page.get_search_matches_table_column_widths()
        gprefs[self.unique_pref_name+':state'] = state

    def _on_dialog_closing(self, result):
        if self.is_closed:
            return
        self.is_closed = True
        # We will always save window position/save
        self._save_window_state()
        # Also persist our settings into the library, even though user cancelled
        cfg.set_library_config(self.db, self.library_config)

        # Make sure no memory leaks (not that there should be anyway)
        self.db = self.gui = None
        self.library_config = self.info = None

