#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

from functools import partial

from qt.core import (Qt, QApplication, QTableView, QAbstractItemView, QModelIndex)

from calibre import prints
from calibre.constants import DEBUG

from calibre_plugins.category_tags.common_utils import ViewLog
from calibre_plugins.category_tags.advanced_matching.rules_widget import AlgorithmsDialog
from calibre_plugins.category_tags.advanced_matching.views import TableView
from calibre_plugins.category_tags.advanced_matching.delegates import ButtonDelegate, ComboDelegate
from calibre_plugins.category_tags.user_categories import get_cols

try:
    load_translations()
except NameError:
    prints("Category Tags::user_categories/views.py - exception when loading translations")

class ListItemView(QTableView):

    def __init__(self, parent):
        QTableView.__init__(self, parent)
        self.setSortingEnabled(False)
        self.setAlternatingRowColors(True)
        self.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.setSelectionMode(QAbstractItemView.ExtendedSelection)
        self.verticalHeader().setDefaultSectionSize(24)
        self.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOn)

    def set_model(self, proxy_model):
        self.setModel(proxy_model)
        self.books_model = proxy_model.sourceModel()
        self.col_map = self.books_model.col_map

        # Hide columns
        for col_name in self.books_model.hidden_cols_1:
            col = self.col_map.index(col_name)
            self.setColumnHidden(col, True)

        self.resizeColumnsToContents()
        # Make sure every other column has a minimum width
        for col_name, width in self.books_model.col_min_width.items():
            col = self.col_map.index(col_name)
            self._set_minimum_column_width(col, width)

    def _set_minimum_column_width(self, col, minimum):
        if self.columnWidth(col) < minimum:
            self.setColumnWidth(col, minimum)

class MatchedItemView(QTableView):

    def __init__(self, parent):
        QTableView.__init__(self, parent)
        self.setSortingEnabled(False)
        self.setAlternatingRowColors(True)
        self.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.setSelectionMode(QAbstractItemView.ExtendedSelection)
        self.verticalHeader().setDefaultSectionSize(24)
        self.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOn)

    def set_model(self, proxy_model):
        self.setModel(proxy_model)
        self.books_model = proxy_model.sourceModel()
        self.col_map = self.books_model.col_map

        # Hide columns
        for col_name in self.books_model.hidden_cols_2:
            col = self.col_map.index(col_name)
            self.setColumnHidden(col, True)

        self.resizeColumnsToContents()
        # Make sure every other column has a minimum width
        for col_name, width in self.books_model.col_min_width.items():
            col = self.col_map.index(col_name)
            self._set_minimum_column_width(col, width)

    def _set_minimum_column_width(self, col, minimum):
        if self.columnWidth(col) < minimum:
            self.setColumnWidth(col, minimum)


class MatchTable(TableView):

    def __init__(self, parent):
        TableView.__init__(self, parent)
        self.gui = parent.gui
        self.db = self.gui.current_db
        self.doubleClicked.connect(self._on_double_clicked)
        self.column_header.setSectionsClickable(True)
        self.column_header.setContextMenuPolicy(Qt.ContextMenuPolicy.CustomContextMenu)
        self.column_header.customContextMenuRequested.connect(partial(self.show_column_header_context_menu))

    def set_model(self, _model):
        self.setModel(_model)
        self.col_map = _model.col_map

        # Hide columns
        for col_name in _model.hidden_cols:
            col = self.col_map.index(col_name)
            self.setColumnHidden(col, True)

        self.categories_delegate = ComboDelegate(self, get_cols(self.db))
        self.setItemDelegateForColumn(self.col_map.index('name'), self.categories_delegate)

        self.button_delegate = ButtonDelegate(self)
        self.setItemDelegateForColumn(self.col_map.index('settings'), self.button_delegate)
        self.button_delegate.clicked.connect(self._on_button_clicked)

        self.resizeColumnsToContents()
        # Make sure every other column has a minimum width
        for col_name, width in _model.col_min_width.items():
            col = self.col_map.index(col_name)
            self._set_minimum_column_width(col, width)

    def _on_button_clicked(self, index):
        m = self.model()
        col_name = m.col_map[index.column()]
        if col_name == 'settings':
            category_col = m.col_map.index('name')
            category_index = m.index(index.row(), category_col, QModelIndex())
            category = m.data(category_index, Qt.DisplayRole)
            category_match_rules = m.categories_match_rules[index.row()]
            name = category_match_rules['name']
            match_rules = category_match_rules.get('settings', [])
            if match_rules:
                match_rule = match_rules[0]
                algorithms_config = match_rule['algos']
            else:
                algorithms_config = []
            algorithms = self.gui.iactions['Category Tags'].algorithms
            d = AlgorithmsDialog(self, self.gui, algorithms_config, algorithms)
            if d.exec_() == d.Accepted:
                algorithms_config = d.algorithms_config
                match_rule = {'field': category, 'algos': algorithms_config}
                category_match_rules['settings'] = [match_rule]
                # reset any previous error if present
                category_match_rules['errors'] = ''
                m.dataChanged.emit(index, index)

    def _on_double_clicked(self, index):
        m = self.model()
        col_name = m.col_map[index.column()]
        if col_name == 'errors':
            category_match_rules = m.categories_match_rules[index.row()]
            details = category_match_rules.get('errors', '')
            self._view_error_details(details)
            

    def _view_error_details(self, details):
        ViewLog(_('Errors details'), details, self)
