
# Modified version of calibre/gui2/dialogs/tag_editor.py

__license__   = 'GPL v3'
__copyright__ = '2008, Kovid Goyal <kovid at kovidgoyal.net>'

from operator import itemgetter

from qt.core import Qt, QApplication, QDialog, QAbstractItemView, QtCore, QtGui, QtWidgets

from calibre.gui2.dialogs.confirm_delete import confirm
from calibre.gui2 import question_dialog, error_dialog, gprefs
from calibre.constants import islinux
from calibre.utils.icu import sort_key, primary_contains

from calibre.gui2.widgets import EnLineEdit

class UserCatEditor(QDialog):

    def __init__(self, gui, category_item, item_type, current_tags=None):
        QDialog.__init__(self, gui)
        self.category_item = category_item
        self.item_type = item_type
        self.current_tags = current_tags
        self.gui = gui
        self.db = gui.current_db
        self.setupUi(self)
        self.initialize()

    def setupUi(self, TagEditor):
        TagEditor.setObjectName("UserCatEditor")
        TagEditor.resize(640, 666)
        self.verticalLayout = QtWidgets.QVBoxLayout(TagEditor)
        self.verticalLayout.setContentsMargins(11, 11, 11, 11)
        self.verticalLayout.setSpacing(6)
        self.verticalLayout.setObjectName("verticalLayout")
        self.horizontalLayout_4 = QtWidgets.QHBoxLayout()
        self.horizontalLayout_4.setContentsMargins(0, -1, 0, -1)
        self.horizontalLayout_4.setSpacing(6)
        self.horizontalLayout_4.setObjectName("horizontalLayout_4")
        spacerItem = QtWidgets.QSpacerItem(40, 20, QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Minimum)
        self.horizontalLayout_4.addItem(spacerItem)
        self.label_3 = QtWidgets.QLabel(TagEditor)
        self.label_3.setObjectName("label_3")
        self.horizontalLayout_4.addWidget(self.label_3)
        self.add_tag_input = EnLineEdit(TagEditor)
        self.add_tag_input.setObjectName("add_tag_input")
        self.horizontalLayout_4.addWidget(self.add_tag_input)
        self.add_tag_button = QtWidgets.QToolButton(TagEditor)
        icon = QtGui.QIcon()
        icon.addPixmap(QtGui.QPixmap(I("plus.png")), QtGui.QIcon.Mode.Normal, QtGui.QIcon.State.Off)
        self.add_tag_button.setIcon(icon)
        self.add_tag_button.setObjectName("add_tag_button")
        self.horizontalLayout_4.addWidget(self.add_tag_button)
        spacerItem1 = QtWidgets.QSpacerItem(40, 20, QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Minimum)
        self.horizontalLayout_4.addItem(spacerItem1)
        self.verticalLayout.addLayout(self.horizontalLayout_4)
        self.line_2 = QtWidgets.QFrame(TagEditor)
        self.line_2.setFrameShape(QtWidgets.QFrame.HLine)
        self.line_2.setFrameShadow(QtWidgets.QFrame.Sunken)
        self.line_2.setObjectName("line_2")
        self.verticalLayout.addWidget(self.line_2)
        self.middleBoxLayout = QtWidgets.QGridLayout()
        self.middleBoxLayout.setSpacing(6)
        self.middleBoxLayout.setObjectName("middleBoxLayout")
        self.available_filter_input = QtWidgets.QLineEdit(TagEditor)
        self.available_filter_input.setClearButtonEnabled(True)
        self.available_filter_input.setObjectName("available_filter_input")
        self.middleBoxLayout.addWidget(self.available_filter_input, 1, 1, 1, 1)
        self.available_tags = QtWidgets.QListWidget(TagEditor)
        self.available_tags.setAlternatingRowColors(True)
        self.available_tags.setSelectionMode(QtWidgets.QAbstractItemView.MultiSelection)
        self.available_tags.setSelectionBehavior(QtWidgets.QAbstractItemView.SelectRows)
        self.available_tags.setObjectName("available_tags")
        self.middleBoxLayout.addWidget(self.available_tags, 2, 1, 1, 1)
        self.buttonverticalLayout = QtWidgets.QVBoxLayout()
        self.buttonverticalLayout.setSpacing(6)
        self.buttonverticalLayout.setObjectName("buttonverticalLayout")
        spacerItem2 = QtWidgets.QSpacerItem(10, 20, QtWidgets.QSizePolicy.Minimum, QtWidgets.QSizePolicy.Expanding)
        self.buttonverticalLayout.addItem(spacerItem2)
        self.delete_button = QtWidgets.QToolButton(TagEditor)
        icon1 = QtGui.QIcon()
        icon1.addPixmap(QtGui.QPixmap(I("trash.png")), QtGui.QIcon.Mode.Normal, QtGui.QIcon.State.Off)
        self.delete_button.setIcon(icon1)
        self.delete_button.setObjectName("delete_button")
        self.buttonverticalLayout.addWidget(self.delete_button)
        spacerItem3 = QtWidgets.QSpacerItem(10, 20, QtWidgets.QSizePolicy.Minimum, QtWidgets.QSizePolicy.Expanding)
        self.buttonverticalLayout.addItem(spacerItem3)
        self.middleBoxLayout.addLayout(self.buttonverticalLayout, 2, 0, 1, 1)
        self.label = QtWidgets.QLabel(TagEditor)
        self.label.setObjectName("label")
        self.middleBoxLayout.addWidget(self.label, 0, 1, 1, 2)
        self.buttonverticalLayout1 = QtWidgets.QVBoxLayout()
        self.buttonverticalLayout1.setSpacing(6)
        self.buttonverticalLayout1.setObjectName("buttonverticalLayout1")
        spacerItem4 = QtWidgets.QSpacerItem(10, 20, QtWidgets.QSizePolicy.Minimum, QtWidgets.QSizePolicy.Expanding)
        self.buttonverticalLayout1.addItem(spacerItem4)
        self.apply_button = QtWidgets.QToolButton(TagEditor)
        icon2 = QtGui.QIcon()
        icon2.addPixmap(QtGui.QPixmap(I("forward.png")), QtGui.QIcon.Mode.Normal, QtGui.QIcon.State.Off)
        self.apply_button.setIcon(icon2)
        self.apply_button.setObjectName("apply_button")
        self.buttonverticalLayout1.addWidget(self.apply_button)
        self.unapply_button = QtWidgets.QToolButton(TagEditor)
        icon3 = QtGui.QIcon()
        icon3.addPixmap(QtGui.QPixmap(I("back.png")), QtGui.QIcon.Mode.Normal, QtGui.QIcon.State.Off)
        self.unapply_button.setIcon(icon3)
        self.unapply_button.setObjectName("unapply_button")
        self.buttonverticalLayout1.addWidget(self.unapply_button)
        spacerItem5 = QtWidgets.QSpacerItem(10, 20, QtWidgets.QSizePolicy.Minimum, QtWidgets.QSizePolicy.Expanding)
        self.buttonverticalLayout1.addItem(spacerItem5)
        self.middleBoxLayout.addLayout(self.buttonverticalLayout1, 2, 2, 1, 1)
        self.applied_filter_input = QtWidgets.QLineEdit(TagEditor)
        self.applied_filter_input.setClearButtonEnabled(True)
        self.applied_filter_input.setObjectName("applied_filter_input")
        self.middleBoxLayout.addWidget(self.applied_filter_input, 1, 3, 1, 1)
        self.applied_tags = QtWidgets.QListWidget(TagEditor)
        self.applied_tags.setAlternatingRowColors(True)
        self.applied_tags.setSelectionMode(QtWidgets.QAbstractItemView.MultiSelection)
        self.applied_tags.setSelectionBehavior(QtWidgets.QAbstractItemView.SelectRows)
        self.applied_tags.setObjectName("applied_tags")
        self.middleBoxLayout.addWidget(self.applied_tags, 2, 3, 1, 1)
        self.label_2 = QtWidgets.QLabel(TagEditor)
        self.label_2.setObjectName("label_2")
        self.middleBoxLayout.addWidget(self.label_2, 0, 3, 1, 1)
        self.verticalLayout.addLayout(self.middleBoxLayout)
        self.line = QtWidgets.QFrame(TagEditor)
        self.line.setFrameShape(QtWidgets.QFrame.HLine)
        self.line.setFrameShadow(QtWidgets.QFrame.Sunken)
        self.line.setObjectName("line")
        self.verticalLayout.addWidget(self.line)
        self.buttonBox = QtWidgets.QDialogButtonBox(TagEditor)
        self.buttonBox.setStandardButtons(QtWidgets.QDialogButtonBox.Cancel|QtWidgets.QDialogButtonBox.Ok)
        self.buttonBox.setObjectName("buttonBox")
        self.verticalLayout.addWidget(self.buttonBox)
        self.label_3.setBuddy(self.add_tag_input)
        self.label.setBuddy(self.available_tags)
        self.label_2.setBuddy(self.applied_tags)

        self.retranslateUi(TagEditor)
        self.buttonBox.accepted.connect(TagEditor.accept)
        self.buttonBox.rejected.connect(TagEditor.reject)
        QtCore.QMetaObject.connectSlotsByName(TagEditor)

    def retranslateUi(self, TagEditor):

        TagEditor.setWindowTitle(_("Manage user categories"))
        self.label_3.setText(_("&Add user category:"))
        self.add_tag_button.setToolTip(_("Apply user category(s) to the current item"))
        self.available_filter_input.setPlaceholderText(_("Filter the available user categories"))
        self.delete_button.setToolTip(_("Delete the selected user categories from database. This will unapply the user categories from all items and then remove them from the database."))
        self.label.setText(_("Ava&ilable user categories"))
        self.apply_button.setToolTip(_("Apply user category to current item"))
        self.unapply_button.setToolTip(_("Unapply (remove) the user category from current item"))
        self.applied_filter_input.setPlaceholderText(_("Filter the applied user categories"))
        self.label_2.setText(_("Applied &user categories"))

    def initialize(self):
        self.added_set = set([])
        self.unapplied_set = set([])
        self.sep = ','
        self.setWindowTitle(_('Edit User Categories For: {} ({})'.format(self.category_item, self.item_type)))

        self.add_tag_input.setToolTip('<p>' +
                    _('If the user category you want is not in the available list, '
                      'you can add it here. Accepts a comma-separated '
                      'list of user categories. The user categories will be applied to '
                      'the item.') + '</p>')
        tags = self.original_tags = self.get_tags()

        if self.current_tags is not None:
            tags = sorted(set(self.current_tags), key=sort_key)
        if tags:
            tags.sort(key=sort_key)
            for tag in tags:
                self.applied_tags.addItem(tag)
        else:
            tags = []

        all_tags = self.get_all_tags()

        all_tags = sorted(set(all_tags), key=sort_key)
        q = set(tags)
        for tag in all_tags:
            if tag not in q:
                self.available_tags.addItem(tag)

        connect_lambda(self.apply_button.clicked, self, lambda self: self.apply_tags())
        connect_lambda(self.unapply_button.clicked, self, lambda self: self.unapply_tags())
        self.add_tag_button.clicked.connect(self.add_tag)
        connect_lambda(self.delete_button.clicked, self, lambda self: self.delete_tags())
        self.add_tag_input.returnPressed[()].connect(self.add_tag)
        # add the handlers for the filter input fields
        connect_lambda(self.available_filter_input.textChanged, self, lambda self, text: self.filter_tags(text))
        connect_lambda(self.applied_filter_input.textChanged, self, lambda self, text: self.filter_tags(text, which='applied_tags'))

        # Restore the focus to the last input box used (typed into)
        for x in ('add_tag_input', 'available_filter_input', 'applied_filter_input'):
            ibox = getattr(self, x)
            ibox.setObjectName(x)
            connect_lambda(ibox.textChanged, self, lambda self: self.edit_box_changed(self.sender().objectName()))
        getattr(self, gprefs.get('tag_editor_last_filter', 'add_tag_input')).setFocus()

        if islinux:
            self.available_tags.itemDoubleClicked.connect(self.apply_tags)
        else:
            self.available_tags.itemActivated.connect(self.apply_tags)
        self.applied_tags.itemActivated.connect(self.unapply_tags)

        geom = gprefs.get('tag_editor_geometry', None)
        if geom is not None:
            QApplication.instance().safe_restore_geometry(self, geom)

    def edit_box_changed(self, which):
        gprefs['tag_editor_last_filter'] = which

    def delete_tags(self, item=None):
        confirms, deletes = [], []
        items = self.available_tags.selectedItems() if item is None else [item]
        if not items:
            error_dialog(self, 'No user categoriess selected', 'You must select at least one user category from the list of Available user cagtegories.').exec_()
            return
        if not confirm(
            _('Deleting user categories is done immediately and there is no undo.'),
            'category_tags_editor_delete'):
            return
        pos = self.available_tags.verticalScrollBar().value()
        for item in items:
            user_categories = dict.copy(self.db.prefs.get('user_categories', {}))
            used = len(user_categories.get(str(item.text()), [])) > 0
            if used:
                confirms.append(item)
            else:
                deletes.append(item)
        if confirms:
            ct = ', '.join([str(item.text()) for item in confirms])
            if question_dialog(self, _('Are your sure?'),
                '<p>'+_('The following user categories are used by one or more items. '
                    'Are you certain you want to delete them?')+'<br>'+ct):
                deletes += confirms

        self.delete_user_categories([str(item.text()) for item in deletes])
        for item in deletes:
            self.available_tags.takeItem(self.available_tags.row(item))
            self.unapplied_set.discard(str(item.text()))
            #
            remove = set()
            for tag in self.unapplied_set:
                if tag.startswith(str(item.text()) + '.'):
                    remove.add(tag)
            self.unapplied_set = self.unapplied_set.difference(remove)
#            for row in reversed(range(self.available_tags.count())):
#                avail_item = self.available_tags.item(row)
#                if str(avail_item.text()).startswith(str(item.text()) + '.'):
#                    self.available_tags.takeItem(self.available_tags.row(avail_item))
            #
        self.available_tags.verticalScrollBar().setValue(pos)

    def apply_tags(self, item=None):
        items = self.available_tags.selectedItems() if item is None else [item]
        rows = [self.available_tags.row(i) for i in items]
        if not rows:
            text = self.available_filter_input.text()
            if text and text.strip():
                self.add_tag_input.setText(text)
                self.add_tag_input.setFocus(Qt.FocusReason.OtherFocusReason)
            return
        row = max(rows)
        tags = self._get_applied_tags_box_contents()
        for item in items:
            tag = str(item.text())
            tags.append(tag)
            self.available_tags.takeItem(self.available_tags.row(item))


        tags.sort(key=sort_key)
        self.applied_tags.clear()
        for tag in tags:
            self.applied_tags.addItem(tag)

        if row >= self.available_tags.count():
            row = self.available_tags.count() - 1

        if row > 2:
            item = self.available_tags.item(row)
            self.available_tags.scrollToItem(item)

        # use the filter again when the applied tags were changed
        self.filter_tags(self.applied_filter_input.text(), which='applied_tags')

    def _get_applied_tags_box_contents(self):
        tags = []
        for i in range(0, self.applied_tags.count()):
            tags.append(str(self.applied_tags.item(i).text()))
        return tags

    def unapply_tags(self, item=None):
        tags = self._get_applied_tags_box_contents()
        items = self.applied_tags.selectedItems() if item is None else [item]
        for item in items:
            tag = str(item.text())
            tags.remove(tag)
            if tag not in self.added_set:
                self.available_tags.addItem(tag)

        tags.sort(key=sort_key)
        self.applied_tags.clear()
        for tag in tags:
            self.applied_tags.addItem(tag)

        items = [str(self.available_tags.item(x).text()) for x in
                range(self.available_tags.count())]
        items.sort(key=sort_key)
        self.available_tags.clear()
        for item in items:
            self.available_tags.addItem(item)
            if item in self.original_tags:
                self.unapplied_set.add(item)

        # use the filter again when the applied tags were changed
        self.filter_tags(self.applied_filter_input.text(), which='applied_tags')
        self.filter_tags(self.available_filter_input.text())

    def add_tag(self):
        tags = str(self.add_tag_input.text()).split(self.sep)
        tags_in_box = self._get_applied_tags_box_contents()
        for tag in tags:
            tag = tag.strip()
            if not tag:
                continue
            for item in self.available_tags.findItems(tag, Qt.MatchFlag.MatchFixedString):
                self.available_tags.takeItem(self.available_tags.row(item))
            if tag not in tags_in_box:
                tags_in_box.append(tag)
                self.added_set.add(tag)

        tags_in_box.sort(key=sort_key)
        self.applied_tags.clear()
        for tag in tags_in_box:
            self.applied_tags.addItem(tag)

        self.add_tag_input.setText('')
        # use the filter again when the applied tags were changed
        self.filter_tags(self.applied_filter_input.text(), which='applied_tags')

    # filter tags
    def filter_tags(self, filter_value, which='available_tags'):
        collection = getattr(self, which)
        q = icu_lower(str(filter_value))
        for i in range(collection.count()):  # on every available tag
            item = collection.item(i)
            item.setHidden(bool(q and not primary_contains(q, str(item.text()))))

    def get_tags(self):
        user_cat_vals = []
        user_categories = dict.copy(self.db.prefs.get('user_categories', {}))
        for ucat, categories in user_categories.items():
            for category_item, item_type, ign in categories:
                if item_type == self.item_type:
                    if category_item == self.category_item:
                        user_cat_vals.append(ucat)
        return user_cat_vals                

    def get_all_tags(self):
        user_categories = dict.copy(self.db.prefs.get('user_categories', {}))
        return user_categories.keys()

    def delete_user_categories(self, categories):
        user_cats = self.db.new_api.pref('user_categories', {})
        cat_keys = sorted(user_cats.keys(), key=sort_key)
        for category_name in categories:
            if category_name.startswith('@'):
                category_name = category_name[1:]
            has_children = False
            found = False
            for k in cat_keys:
                if k == category_name:
                    found = True
                    has_children = len(user_cats[k])
                elif k.startswith(category_name + '.'):
                    has_children = True
            if not found:
                continue
            for k in cat_keys:
                if k == category_name:
                    del user_cats[k]
                elif k.startswith(category_name + '.'):
                    del user_cats[k]
        self.db.new_api.set_pref('user_categories', user_cats)
        self.gui.tags_view.recount()
        self.db.new_api.clear_search_caches()
        self.gui.library_view.model().refresh()

    def get_tags_to_remove(self):
        '''
        Get unapplied tags and make sure to remove any tag whose parent category is
        also in unapplied tag to avoid any problems with processing the tag after deleting
        the parent category e.g. cat.sub_cat & cat
        '''
        remove = set()
        for tag1 in self.unapplied_set:
            search = tag1 + '.'
            for tag2 in self.unapplied_set:
                if tag2.startswith(search):
                    remove.add(tag2)
        return self.unapplied_set.difference(remove)

    def accept(self):
        self.new_tags = set(self._get_applied_tags_box_contents()).difference(set(self.original_tags))
        self.remove_tags = self.get_tags_to_remove()
        self.save_state()
        return QDialog.accept(self)

    def reject(self):
        self.save_state()
        return QDialog.reject(self)

    def save_state(self):
        gprefs['tag_editor_geometry'] = bytearray(self.saveGeometry())

