#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

from collections import defaultdict
from functools import partial

from qt.core import (Qt, QApplication, QWidget, QVBoxLayout, QHBoxLayout, QGridLayout,
                     QDialog, QDialogButtonBox, QCheckBox, QComboBox, QPushButton,
                     QSizePolicy, QSize, QModelIndex, pyqtSignal, QGroupBox, QLabel,
                     QAbstractItemView, QRadioButton, QSpacerItem, QAction, QToolButton,
                     QTextBrowser, QLineEdit, QSplitter, QTableWidget, QWizardPage,
                     QTabWidget, QScrollArea)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2.dialogs.confirm_delete import confirm
from calibre.gui2.widgets2 import Dialog
from calibre.ebooks.metadata.book.base import Metadata
from calibre.utils.icu import contains, primary_contains

from calibre_plugins.category_tags.common_utils import (get_icon, ReadOnlyTableWidgetItem)
from calibre_plugins.category_tags.user_categories.models import ResolveModel, ResolveSortFilterModel
from calibre_plugins.category_tags.user_categories.views import ListItemView, MatchedItemView
import calibre_plugins.category_tags.config as cfg

try:
    load_translations()
except NameError:
    prints("Category Tags::user_categories/page_resolve.py - exception when loading translations")

class ItemTypeFilterWidget(QWidget):

    def __init__(self, parent, label, slot=None):
        QWidget.__init__(self, parent)
        self._parent = parent
        self.slot = slot
        self.label = label
        self._init_controls()

    def _init_controls(self):
        l = QHBoxLayout()
        self.setLayout(l)
        label = QLabel(self.label)
        self.combo = QComboBox()
        self.combo.setCurrentIndex(-1)
        if self.slot:
            self.combo.activated.connect(self.slot)
        l.addWidget(label)
        l.addWidget(self.combo)

    def update(self):
        item_types = self._parent.library_matches_for_imported_items_map.keys()
        # Clear combo in case the user presses previous and then next
        self.combo.clear()
        self.combo.addItems([''] + list(item_types))

    def currentText(self):
        return self.combo.currentText()

class SearchMatchesTableWidget(QTableWidget):

    def __init__(self, parent):
        QTableWidget.__init__(self, parent)
        self.setSortingEnabled(False)
        self.setAlternatingRowColors(True)
        self.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.setSelectionMode(QAbstractItemView.SingleSelection)
        self.verticalHeader().setDefaultSectionSize(24)

    def initialise(self, column_widths, cols_map):
        self.cols = cols_map
        self.setColumnCount(len(self.cols))
        self.setHorizontalHeaderLabels(self.cols.values())
        self.populate_table([])

        if column_widths is None:
            for i, key in enumerate(self.cols.keys()):
                if key == 'item':
                    self._set_minimum_column_width(i, 350)
                else:
                    self._set_minimum_column_width(i, 100)
        else:
            for c,w in enumerate(column_widths):
                self.setColumnWidth(c, w)

    def populate_table(self, items_payload):
        self.items_payload = items_payload
        self.setRowCount(0)
        self.setRowCount(len(items_payload))
        for row, item_payload in enumerate(items_payload):
            self.populate_table_row(row, item_payload)

    def _set_minimum_column_width(self, col, minimum):
        if self.columnWidth(col) < minimum:
            self.setColumnWidth(col, minimum)

    def populate_table_row(self, row, item_payload):
        for i, key in enumerate(self.cols.keys()):
            data = item_payload[key]
            self.setItem(row, i, ReadOnlyTableWidgetItem(data))

class ResolveCommon(object):

    ID = 2

    def init_controls(self):
        self.block_events = True
        l = self.l = QVBoxLayout(self)
        self.setLayout(l)
        self.splitter = QSplitter(self)
        self.splitter.setOrientation(Qt.Vertical)
        self.splitter.setChildrenCollapsible(False)

        self.list_gb = QGroupBox('', self)
        self.list_gb.setStyleSheet('QGroupBox { font-weight: bold; }')
        self.splitter.addWidget(self.list_gb)
        gbl = QVBoxLayout()
        self.list_gb.setLayout(gbl)

        fl = QHBoxLayout()
        self.filter_all = QRadioButton(_("Show All"), self)
        self.filter_all.setChecked(True)
        self.filter_matched = QRadioButton(_("Single Match"), self)
        self.filter_multiple_match = QRadioButton(_("Multiple Matches"), self)
        self.filter_unmatched = QRadioButton(_("Unmatched"), self)
        for btn in [self.filter_all, self.filter_matched, self.filter_multiple_match, self.filter_unmatched]:
            btn.clicked.connect(self._refresh_filter)
        fl.addWidget(self.filter_all)
        fl.addWidget(self.filter_matched)
        fl.addWidget(self.filter_multiple_match)
        fl.addWidget(self.filter_unmatched)

        self.item_type_filter = ItemTypeFilterWidget(self, _('Filter by item type'), slot=self._refresh_filter)

        bll = QGridLayout()
        gbl.addLayout(bll)
        hl1 = QHBoxLayout()
        self.list_imported_items_label = QLabel(_('Imported Items:'), self)
        self.list_imported_items_label.setStyleSheet('QLabel { font-weight: bold; }')
        self.matched_items_label = QLabel(_('Matches in library:'), self)
        self.matched_items_label.setStyleSheet('QLabel { font-weight: bold; }')
        bll.addWidget(self.list_imported_items_label, 0, 0, 1, 1)
        bll.addWidget(self.item_type_filter, 1, 0, 1, 1)
        bll.addWidget(self.matched_items_label, 0, 1, 1, 1)
        bll.addLayout(fl, 1, 1, 1, 1)
        self.list_item_view = ListItemView(self)
        self.matched_item_view = MatchedItemView(self)
        bll.addWidget(self.list_item_view, 2, 0, 1, 1)
        bll.addWidget(self.matched_item_view, 2, 1, 1, 2)
        bll.setColumnStretch(0, 2)
        bll.setColumnStretch(1, 1)
        bll.setColumnStretch(2, 1)

        btnl = QVBoxLayout()
        bll.addLayout(btnl, 2, 3, 1, 1)
        self.clear_match_button = QToolButton(self)
        self.clear_match_button.setIcon(get_icon('list_remove.png'))
        self.clear_match_button.setToolTip(_('Clear the match associated with the selected items in the list'))
        self.clear_match_button.clicked.connect(self._clear_match)
        self.remove_item_button = QToolButton(self)
        self.remove_item_button.setIcon(get_icon('minus.png'))
        self.remove_item_button.setToolTip(_('Remove the selected items from the list'))
        self.remove_item_button.clicked.connect(self._remove_item)
        self.empty_book_button = QToolButton(self)
        self.empty_book_button.setIcon(get_icon('add_book.png'))
        self.empty_book_button.setToolTip(_('Create an empty book for the selected items in the list. '
                                            'This is only possible for items belonging to columns with multiple values. '
                                            'The items will be added to book titled: *** User Categories ***'))
        self.empty_book_button.clicked.connect(self._add_items_to_empty_book)
        btnl.addWidget(self.clear_match_button)
        btnl.addWidget(self.empty_book_button)
        btnl.addWidget(self.remove_item_button)
        btnl.addStretch(1)

        gb = QGroupBox(_('Possible matches for selected item:'), self)
        gb.setStyleSheet('QGroupBox { font-weight: bold; }')
        self.splitter.addWidget(gb)
        gl = QVBoxLayout()
        gb.setLayout(gl)

        sl = QHBoxLayout()
        gl.addLayout(sl)
        search_label = QLabel(_('Search:'), self)
        self.search_ledit = QLineEdit(self)
        self.go_button = QPushButton(_('&Go!'), self)
        self.go_button.clicked.connect(partial(self._on_search_click))
        self.item_type_filter_2 = ItemTypeFilterWidget(self, _('Filter by item type'))
        self.clear_button = QToolButton(self)
        self.clear_button.setIcon(get_icon('clear_left.png'))
        self.clear_button.clicked.connect(partial(self._on_clear_search_text))
        sl.addWidget(search_label)
        sl.addWidget(self.search_ledit, 1)
        sl.addWidget(self.item_type_filter_2)
        sl.addWidget(self.go_button)
        sl.addWidget(self.clear_button)

        bll2 = QHBoxLayout()
        gl.addLayout(bll2)
        self.search_matches_table = SearchMatchesTableWidget(self)
        self.search_matches_table.itemSelectionChanged.connect(self._update_match_buttons_state)
        bll2.addWidget(self.search_matches_table, 1)

        btn2 = QVBoxLayout()
        bll2.addLayout(btn2)
        btn2.addStretch(1)
        self.select_item_button = QToolButton(self)
        self.select_item_button.setIcon(get_icon('ok.png'))
        self.select_item_button.setToolTip(_('Select this item as the match for this list title'))
        self.select_item_button.clicked.connect(self._on_search_matches_select)
        btn2.addWidget(self.select_item_button)

        self.add_item_button = QToolButton(self)
        self.add_item_button.setIcon(get_icon('plus.png'))
        self.add_item_button.setToolTip(_('Add this item to the list of matches'))
        self.add_item_button.clicked.connect(self._on_search_matches_add)
        btn2.addWidget(self.add_item_button)

        self.remove_item_button = QToolButton(self)
        self.remove_item_button.setIcon(get_icon('minus.png'))
        self.remove_item_button.setToolTip(_('Remove this item from the list of matches'))
        self.remove_item_button.clicked.connect(self._on_search_matches_remove)
        btn2.addWidget(self.remove_item_button)

        self.splitter.setStretchFactor(0, 3)
        self.splitter.setStretchFactor(1, 2)
        l.addWidget(self.splitter, 1)

        self.list_item_view.doubleClicked.connect(self._on_item_list_double_clicked)
        self.search_matches_table.doubleClicked.connect(self._on_search_matches_double_clicked)
        self.list_item_view.verticalScrollBar().valueChanged[int].connect(self._sync_to_list_scrollbar)
        self.matched_item_view.verticalScrollBar().valueChanged[int].connect(self._sync_to_matched_scrollbar)
        self.is_scrolling = False

        self.block_events = False

        self._create_context_menu_actions()
        self._create_context_menus(self.list_item_view)
        self._create_context_menus(self.matched_item_view)
        self._create_search_matches_context_menus()

    def initializePage(self):
        self.library_matches_for_imported_items_map = self.info['library_matches_for_imported_items_map']
        self.resolve_model = ResolveModel(self.gui, self.library_matches_for_imported_items_map)
        self.proxy_model = ResolveSortFilterModel(self)
        self.proxy_model.setSourceModel(self.resolve_model)
        self.list_item_view.set_model(self.proxy_model)
        self.matched_item_view.set_model(self.proxy_model)
        self.list_item_view.selectRow(0)
        self.matched_item_view.setSelectionModel(self.list_item_view.selectionModel())
        self.list_item_view.selectionModel().currentChanged.connect(self._on_item_list_current_changed)
        self.list_item_view.selectionModel().selectionChanged.connect(self._on_item_list_selection_changed)

        column_widths = self.info.get('state', {}).get('resolve_search_column_widths', None)
        self.search_matches_table.initialise(column_widths, {'item': 'Item', 'item_type': 'Item Type'})

        self._update_book_counts()
        # If our first page needs a search, fire it off now
        if len(str(self.search_ledit.text())):
            self._on_search_click()

        splitter_state = self.info.get('state', {}).get('resolve_splitter_state', None)
        if splitter_state is not None:
            self.splitter.restoreState(splitter_state)
        # Make sure the buttons are set correctly for the opening state
        self._update_item_list_buttons_state()
        self._update_match_buttons_state()
        self._on_item_list_current_changed(self.list_item_view.model().index(0,0,QModelIndex()), None)

        self.item_type_filter.update()
        self.item_type_filter_2.update()

        self._refresh_filter()

    def _create_context_menu_actions(self):
        self.clear_match_action = QAction(get_icon('list_remove.png'), _('&Clear match'), self)
        self.clear_match_action.setToolTip(self.clear_match_button.toolTip())
        self.clear_match_action.triggered.connect(self._clear_match)
        self.remove_item_action = QAction(get_icon('minus.png'), _('&Remove item'), self)
        self.remove_item_action.setToolTip(self.remove_item_button.toolTip())
        self.remove_item_action.triggered.connect(self._remove_item)
        self.sep1 = QAction(self)
        self.sep1.setSeparator(True)
        self.empty_book_action = QAction(get_icon('add_book.png'), _('Add to &empty book'), self)
        self.empty_book_action.setToolTip(self.empty_book_button.toolTip())
        self.empty_book_action.triggered.connect(self._add_items_to_empty_book)
        self.sep2 = QAction(self)
        self.sep2.setSeparator(True)
        self.search_item_action = QAction(get_icon('search.png'), _('&Search for item'), self)
        self.search_item_action.setToolTip(_('&Search your library for item'))
        self.search_item_action.triggered.connect(partial(self._force_search_item))

    def _create_context_menus(self, table):
        table.setContextMenuPolicy(Qt.ActionsContextMenu)
        table.addAction(self.search_item_action)
        table.addAction(self.sep1)
        table.addAction(self.clear_match_action)
        table.addAction(self.empty_book_action)
        table.addAction(self.sep2)
        table.addAction(self.remove_item_action)

    def _create_search_matches_context_menus(self):
        table = self.search_matches_table
        table.setContextMenuPolicy(Qt.ActionsContextMenu)
        self.select_item_action = QAction(get_icon('ok.png'), _('&Select Item'), table)
        self.select_item_action.setToolTip(self.select_item_button.toolTip())
        self.select_item_action.triggered.connect(self._on_search_matches_select)
        sep5 = QAction(table)
        sep5.setSeparator(True)

        table.addAction(self.select_item_action)
        table.addAction(sep5)

        self.add_item_action = QAction(get_icon('plus.png'), _('&Add Item'), table)
        self.add_item_action.setToolTip(self.select_item_button.toolTip())
        self.add_item_action.triggered.connect(self._on_search_matches_add)
        sep7 = QAction(table)
        sep7.setSeparator(True)

        table.addAction(self.add_item_action)
        table.addAction(sep7)

        self.remove_item_action = QAction(get_icon('minus.png'), _('&Remove Item'), table)
        self.remove_item_action.setToolTip(self.select_item_button.toolTip())
        self.remove_item_action.triggered.connect(self._on_search_matches_remove)
        sep6 = QAction(table)
        sep6.setSeparator(True)

        table.addAction(self.remove_item_action)
        table.addAction(sep6)

    def _sync_to_list_scrollbar(self, value):
        if self.is_scrolling:
            return
        self.is_scrolling = True
        self.matched_item_view.verticalScrollBar().setValue(value)
        self.is_scrolling = False

    def _sync_to_matched_scrollbar(self, value):
        if self.is_scrolling:
            return
        self.is_scrolling = True
        self.list_item_view.verticalScrollBar().setValue(value)
        self.is_scrolling = False

    def _update_item_list_buttons_state(self):

        typs = self.library_matches_for_imported_items_map.keys()
        multiple_typs = [typ for typ in typs if self.db.field_metadata.all_metadata()[typ]['is_multiple']] + ['authors']

        is_row_selected = self.list_item_view.currentIndex().isValid()

        can_clear = can_add_empty = is_row_selected
        row = None
        if is_row_selected:
            rows = self.list_item_view.selectionModel().selectedRows()
            for selrow in rows:
                actual_idx = self.proxy_model.mapToSource(selrow)
                item_data = self.resolve_model.all_items_data[actual_idx.row()]
                if item_data['status'] in ['unmatched','empty']:
                    can_clear = False
                else:
                    can_add_empty = False
                if item_data['item_type'] not in multiple_typs:
                    can_add_empty = False

        self.clear_match_button.setEnabled(is_row_selected and can_clear)
        self.clear_match_action.setEnabled(self.clear_match_button.isEnabled())
        self.remove_item_button.setEnabled(is_row_selected and len(self.resolve_model.all_items_data) > 0)
        self.remove_item_action.setEnabled(self.remove_item_button.isEnabled())
        self.empty_book_button.setEnabled(is_row_selected and can_add_empty)
        self.empty_book_action.setEnabled(self.empty_book_button.isEnabled())
        self.search_item_action.setEnabled(is_row_selected)

    def _update_match_buttons_state(self):
        have_items = len(self.search_matches_table.items_payload) > 0
        is_row_selected = self.search_matches_table.currentRow() != -1
        #
        same_item_type = False
        if is_row_selected:
            match_item = self.search_matches_table.items_payload[self.search_matches_table.currentRow()]
            selected_idx = self.list_item_view.currentIndex()
            actual_idx = self.proxy_model.mapToSource(selected_idx)
            item_data = self.resolve_model.all_items_data[actual_idx.row()]
            same_item_type = item_data['item_type'] == match_item['item_type']
        #
        in_matches = False
        if is_row_selected:
            match_item = self.search_matches_table.items_payload[self.search_matches_table.currentRow()]
            selected_idx = self.list_item_view.currentIndex()
            actual_idx = self.proxy_model.mapToSource(selected_idx)
            item_data = self.resolve_model.all_items_data[actual_idx.row()]
            in_matches = match_item['item'] in item_data['matches']
        #        
        self.select_item_button.setEnabled(have_items and same_item_type)
        self.select_item_action.setEnabled(is_row_selected and have_items and same_item_type)
        self.add_item_action.setEnabled(is_row_selected and same_item_type and not in_matches)
        self.add_item_button.setEnabled(is_row_selected and same_item_type and not in_matches)
        self.remove_item_action.setEnabled(is_row_selected and in_matches)
        self.remove_item_button.setEnabled(is_row_selected and in_matches)

    def _clear_match(self):
        QApplication.setOverrideCursor(Qt.WaitCursor)
        try:
            rows = self.list_item_view.selectionModel().selectedRows()
            for selrow in rows:
                actual_idx = self.proxy_model.mapToSource(selrow)
                item_data = self.resolve_model.all_items_data[actual_idx.row()]
                item_data['item_alt'] = '*** No Matches ***'
                item_data['matches'] = set()
                item_data['status'] = 'unmatched'
                idx = self.resolve_model.index(actual_idx.row(), actual_idx.row())
                self.resolve_model.dataChanged.emit(idx, idx)
            self._update_book_counts()
            self._update_item_list_buttons_state()
        finally:
            QApplication.restoreOverrideCursor()

    def _remove_item(self):
        message = _('<p>Are you sure you want to remove the selected items from the list?</p>')
        if not confirm(message,'category_tags_delete_from_list', self):
            return
        QApplication.setOverrideCursor(Qt.WaitCursor)
        try:
            rows = sorted(self.list_item_view.selectionModel().selectedRows())
            actual_idx = self.proxy_model.mapToSource(rows[0])
            sel_row = actual_idx.row()
            for selrow in reversed(rows):
                actual_idx = self.proxy_model.mapToSource(selrow)
                self.resolve_model.removeRow(actual_idx.row())

            cnt = len(self.resolve_model.all_items_data)
            if sel_row == cnt:
                sel_row = cnt - 1
            if cnt == 0:
                sel_row = -1
            else:
                self.list_item_view.selectRow(sel_row)
                self._refresh_selected_item(sel_row)
            self._update_book_counts()
            self._update_item_list_buttons_state()
        finally:
            QApplication.restoreOverrideCursor()

    def _add_items_to_empty_book(self):
        QApplication.setOverrideCursor(Qt.WaitCursor)
        try:
            rows = self.list_item_view.selectionModel().selectedRows()
            update = defaultdict(set)
            book_id = self.get_empty_book_id()
            for selrow in reversed(rows):
                actual_idx = self.proxy_model.mapToSource(selrow)
                # TODO: add items to empty book
                #idx = self.resolve_model.index(actual_idx.row(), actual_idx.row())
                #self.resolve_model.dataChanged.emit(idx, idx)
                item_data = self.resolve_model.all_items_data[actual_idx.row()]
                update[item_data['item_type']].update(set([item_data['item']]))
                item_data['item_alt'] = item_data['item']
                item_data['matches'] = set(item_data['item'])
                item_data['status'] = 'empty'
            for item_type, items in update.items():
                all_items = set(self.db.new_api.field_for(item_type, book_id))
                all_items.update(items)
                self.db.new_api.set_field(item_type, {book_id: list(all_items)})
            self._update_book_counts()
            self._update_item_list_buttons_state()
        finally:
            QApplication.restoreOverrideCursor()

    def _force_search_item(self):
        actual_idx = self.proxy_model.mapToSource(self.list_item_view.selectionModel().selectedRows()[0])
        item_data = self.resolve_model.all_items_data[actual_idx.row()]
        self._on_clear_search_text()
        self._prepare_search_text(item_data)
        self._on_search_click()

    def _on_item_list_current_changed(self, row, old_row):
        if self.block_events:
            return
        actual_idx = self.proxy_model.mapToSource(row)
        self._refresh_selected_item(actual_idx.row())

    def _refresh_selected_item(self, row):
        self.search_ledit.setText('')
        self._clear_match_list()
        if row < 0:
            return
        item_data = self.resolve_model.all_items_data[row]
        if item_data['status'] in ['multiple']:
            self.search_ledit.setPlaceholderText(_('Displaying all similar matches for this item'))
            items_payload = [{'item': match, 'item_type': item_data['item_type']} for match in item_data['matches']]
            self._display_multiple_matches(items_payload)
        else:
            self._on_clear_search_text()
            self._prepare_search_text(item_data)

    def _on_item_list_selection_changed(self, sel, desel):
        if self.block_events:
            return
        self._update_item_list_buttons_state()

    def _on_item_list_double_clicked(self, row):
        actual_idx = self.proxy_model.mapToSource(row)
        item_data = self.resolve_model.all_items_data[actual_idx.row()]
        if item_data['item_alt'] != '*** Multiple Matches ***':
            self._on_search_click()

    def _on_clear_search_text(self):
        self.search_ledit.setPlaceholderText(_('Search for an item in your library'))
        self.search_ledit.clear()

    def _prepare_search_text(self, item_data):
        query = item_data['item']
        self.search_ledit.setText(query.strip())
        self.go_button.setAutoDefault(True)
        self.go_button.setDefault(True)

    def _on_search_click(self):
        query = str(self.search_ledit.text())
        QApplication.setOverrideCursor(Qt.WaitCursor)
        item_type = self.item_type_filter_2.currentText()
        matches = self.search_items(query, item_type)
        QApplication.restoreOverrideCursor()
        self._display_multiple_matches(matches)

    def _on_search_matches_select(self):
        self._on_search_matches_double_clicked(self.search_matches_table.currentIndex())

    def _on_search_matches_double_clicked(self, row):
        match_item = self.search_matches_table.items_payload[row.row()]
        actual_idx = self.proxy_model.mapToSource(self.list_item_view.selectionModel().selectedRows()[0])
        list_row = actual_idx.row()
        item_data = self.resolve_model.all_items_data[list_row]
        if item_data['status'] in ['unmatched', 'multiple']:
            item_data['item_alt'] = match_item['item']
            item_data['status'] = 'matched'
            item_data['matches'] = set([match_item['item']])
            idx = self.resolve_model.index(list_row, list_row)
            self.resolve_model.dataChanged.emit(idx, idx)
            self._update_book_counts()
            self._clear_match_list()
            self._update_match_buttons_state()
            self._update_item_list_buttons_state()

    def _on_search_matches_add(self):
        current_idx = self.search_matches_table.currentIndex()
        match_item = self.search_matches_table.items_payload[current_idx.row()]
        actual_idx = self.proxy_model.mapToSource(self.list_item_view.selectionModel().selectedRows()[0])
        list_row = actual_idx.row()
        item_data = self.resolve_model.all_items_data[list_row]
        item_data['matches'].add(match_item['item'])
        if len(item_data['matches']) == 1:
            item_data['status'] = 'matched'
            item_data['item_alt'] = match_item['item']
        elif len(item_data['matches']) > 1:
            item_data['status'] = 'multiple'
            item_data['item_alt'] = '*** Multiple Matches ***'
        idx = self.resolve_model.index(list_row, list_row)
        self.resolve_model.dataChanged.emit(idx, idx)
        self._update_book_counts()
        self._update_match_buttons_state()
        self._update_item_list_buttons_state()

    def _on_search_matches_remove(self):
        current_idx = self.search_matches_table.currentIndex()
        match_item = self.search_matches_table.items_payload[current_idx.row()]
        actual_idx = self.proxy_model.mapToSource(self.list_item_view.selectionModel().selectedRows()[0])
        list_row = actual_idx.row()
        item_data = self.resolve_model.all_items_data[list_row]
        if item_data['status'] in ['multiple']:
            item_data['matches'].discard(match_item['item'])
            if len(item_data['matches']) == 1:
                item_data['status'] = 'matched'
                item_data['item_alt'] = match_item['item']
            elif len(item_data['matches']) == 0:
                item_data['status'] = 'unmatched'
                item_data['item_alt'] = '*** No Matches ***'
            idx = self.resolve_model.index(list_row, list_row)
            self.resolve_model.dataChanged.emit(idx, idx)
            self._update_book_counts()
            self._clear_match_list()
            self._refresh_selected_item(list_row)
            self._update_match_buttons_state()
            self._update_item_list_buttons_state()

    def _display_multiple_matches(self, items_payload):
        # Multisort items_payload
        items_payload.sort(key=lambda x: x['item'])
        items_payload.sort(key=lambda x: x['item_type'])
        self.search_matches_table.populate_table(items_payload)
        if items_payload:
            self.search_matches_table.selectRow(0)
        self._update_match_buttons_state()

    def _clear_match_list(self):
        self.search_matches_table.populate_table([])
        self._update_match_buttons_state()

    def _refresh_filter(self):
        if self.filter_all.isChecked():
            filter_criteria = 'all'
        elif self.filter_matched.isChecked():
            filter_criteria = 'matched'
        elif self.filter_multiple_match.isChecked():
            filter_criteria = 'multiple'
        elif self.filter_unmatched.isChecked():
            filter_criteria = 'unmatched'
        item_type = self.item_type_filter.currentText()
        self.block_events = True
        self.proxy_model.set_filter_criteria(filter_criteria, item_type)
        self.block_events = False
        actual_idx = self.proxy_model.mapToSource(self.list_item_view.model().index(0,0,QModelIndex()))
        self._refresh_selected_item(actual_idx.row())

    def _update_book_counts(self):
        matches_cnt = 0
        total = len(self.resolve_model.all_items_data)
        for item_data in self.resolve_model.all_items_data:
            if item_data['status'] not in ['unmatched', 'multiple']:
                matches_cnt += 1
        if total == 0:
            self.matched_items_label.setText(_('Matches in library:'))
        elif total == 1 and matches_cnt == 1:
            self.matched_items_label.setText(_('Matches in library:(1 match)'))
        else:
            self.matched_items_label.setText(_('Matches in library: (%(count)d of %(total)d matches)')%{'count': matches_cnt, 'total': total})
        # wizard
        if isinstance(self, QWizardPage):
            self.completeChanged.emit()

    def get_search_matches_table_column_widths(self):
        table_column_widths = []
        for c in range(0, self.search_matches_table.columnCount()):
            table_column_widths.append(self.search_matches_table.columnWidth(c))
        return table_column_widths

    def search_items(self, query, item_type=None):
        res = []
        if item_type:
            typs = [item_type]
        else:
            typs = self.library_matches_for_imported_items_map.keys()
        for typ in typs:
            for item in self.db.new_api.all_field_names(typ):
                if contains(query, item):
                    res.append({'item': item, 'item_type': typ})
        return res

    def get_empty_book_id(self):
        book_ids = self.db.data.search_getting_ids('title:"=*** User Categories ***"', None)
        if book_ids:
            return sorted(book_ids)[0]
        else:
            mi = Metadata(_('*** User Categories ***'))
            mi.set_all_user_metadata(self.db.field_metadata.custom_field_metadata())
            book_id = self.db.import_book(mi, [])
            return book_id

class ResolvePage(QWizardPage, ResolveCommon):

    def __init__(self, parent, gui, action):
        QWizardPage.__init__(self, parent)
        self.gui = gui
        self.db = gui.current_db
        self.info = parent.info
        self.action = action
        self.library_config = parent.library_config
        self.init_controls()

    def isComplete(self):
        return True

    def validatePage(self):
        db = self.gui.library_view.model().db
        
        self.action.library_matches_for_imported_items_map = self.resolve_model.get_library_matches_for_imported_items_map()
        
        add_multiple_matches = self.info['add_multiple_matches']
        match_rules_for_categories = self.info.get('match_rules_for_categories', {})
        updated_categories = dict.copy(db.prefs.get('user_categories', {}))
        case_insensitive_keys = {x.lower() for x in updated_categories.keys()}
        link_map = defaultdict()
        #
        for item_type, record in self.info['pivot_format'].items():
            default_match_rules = [ { "algos": [ { "name": "Identical Match", "settings": {} } ], "field": item_type } ]
            match_rules = match_rules_for_categories.get(item_type, default_match_rules)
            #for item, categories_for_item in record.items():
            for item, item_dict in record.items():
                categories_for_item = item_dict.get('categories', '')
                if not categories_for_item: categories_for_item = set()
                note = item_dict.get('note')
                link = item_dict.get('link')
                added = self.action.add_categories_for_item(
                                item,
                                categories_for_item,
                                item_type,
                                updated_categories,
                                case_insensitive_keys,
                                match_rules,
                                link,
                                link_map,
                                note,
                                add_multiple_matches)

        # New Top level nodes must be added to calibre before updating with the new categories
        self.action.create_top_level_nodes(updated_categories)            

        final_categories = dict.copy(db.prefs.get('user_categories', {}))
        final_categories.update(updated_categories)

        self.action.refresh(final_categories)

        for item_type, type_link_map in link_map.items():
            self.db.new_api.set_link_map(item_type, type_link_map)

        cfg.set_library_config(self.db, self.library_config)
        return True
