#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__   = 'GPL v3'
__copyright__ = '2021 additions by Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'


from qt.core import (Qt, QApplication, QWidget, QVBoxLayout, QHBoxLayout, QGridLayout,
                     QTextBrowser, QToolButton, QRadioButton, QCheckBox, QIcon, QGroupBox,
                     QSizePolicy, QSpacerItem, QScrollArea, QTabWidget, QWizardPage, QDialog)

import os
import copy
from functools import partial
from collections import defaultdict

from calibre import prints
from calibre.constants import iswindows, DEBUG
from calibre.gui2 import choose_files, gprefs, error_dialog

from calibre_plugins.category_tags.advanced_matching import SettingsWidgetDialog
from calibre_plugins.category_tags.common_utils import get_icon, DragDropComboBox
from calibre_plugins.category_tags.user_categories.views import MatchTable
from calibre_plugins.category_tags.user_categories.models import MatchModel, UP, DOWN
import calibre_plugins.category_tags.config as cfg

try:
    load_translations()
except NameError:
    prints("Category Tags::user_categories/page_import.py - exception when loading translations")

help_text = _('''
<ul>
<li>You can add match rules per category type (e.g authors, tags ... etc) using the table below.</li>
<li>For category types that have no configured match rules, imported items will be matched using a case-insensitive identical match.</li>
</ul>
''')

class MatchWidget(QWidget):
    def __init__(self, parent, gui, match_rules_for_categories):
        self.gui = gui
        self.db = self.gui.current_db
        self.match_rules_for_categories = copy.deepcopy(match_rules_for_categories)
        self.KEY_TABLE_STATE = 'MatchingTableState'
        QWidget.__init__(self)
        self.setup_ui()

    def setup_ui(self):
        self.setWindowTitle(_('Add match rules'))
        l = QVBoxLayout()
        self.setLayout(l)

        tb = QTextBrowser()
        tb.setHtml(help_text)
        l.addWidget(tb)

        settings_l = QGridLayout()
        l.addLayout(settings_l)

        table_groupbox = QGroupBox(_('Rules for categories'))
        table_layout = QHBoxLayout()
        table_groupbox.setLayout(table_layout)
        l.addWidget(table_groupbox)
        
        self._table = MatchTable(self)
        table_layout.addWidget(self._table)
        
        match_model = MatchModel(self.gui, self.match_rules_for_categories)
        match_model.validate()
        self._table.set_model(match_model)
        self._table.selectionModel().selectionChanged.connect(self._on_table_selection_change)
        
        # Restore table state
        state = gprefs.get(self.KEY_TABLE_STATE)
        if state:
            self._table.apply_state(state)

        # Add a vertical layout containing the the buttons to move up/down etc.
        button_layout = QVBoxLayout()
        table_layout.addLayout(button_layout)
        
        move_up_button = self.move_up_button = QToolButton(self)
        move_up_button.setToolTip(_('Move row up'))
        move_up_button.setIcon(QIcon(I('arrow-up.png')))
        button_layout.addWidget(move_up_button)
        spacerItem1 = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem1)

        add_button = self.add_button = QToolButton(self)
        add_button.setToolTip(_('Add row'))
        add_button.setIcon(QIcon(I('plus.png')))
        button_layout.addWidget(add_button)
        spacerItem2 = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem2)

        delete_button = self.delete_button = QToolButton(self)
        delete_button.setToolTip(_('Delete row'))
        delete_button.setIcon(QIcon(I('minus.png')))
        button_layout.addWidget(delete_button)
        spacerItem4 = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem4)

        move_down_button = self.move_down_button = QToolButton(self)
        move_down_button.setToolTip(_('Move row down'))
        move_down_button.setIcon(QIcon(I('arrow-down.png')))
        button_layout.addWidget(move_down_button)

        move_up_button.clicked.connect(partial(self._table.move_rows,UP))
        move_down_button.clicked.connect(partial(self._table.move_rows,DOWN))
        add_button.clicked.connect(self._table.add_row)
        delete_button.clicked.connect(self._table.delete_rows)

        self.add_multiple_matches_chk = QCheckBox(_('Add all matches'))
        self.add_multiple_matches_chk.setToolTip(_('For items with multiple matches, apply categories for all the matches'))
        self.add_multiple_matches_chk.setChecked(True)
        l.addWidget(self.add_multiple_matches_chk)
        
        self._on_table_selection_change()

    def _on_table_selection_change(self):
        sm = self._table.selectionModel()
        selection_count = len(sm.selectedRows())
        self.delete_button.setEnabled(selection_count > 0)
        self.move_up_button.setEnabled(selection_count > 0)
        self.move_down_button.setEnabled(selection_count > 0)

    def save_table_state(self):
        # Save table state
        gprefs[self.KEY_TABLE_STATE] = self._table.get_state()

class OptionsWidget(QWidget):

    def __init__(self, gui):
        QWidget.__init__(self)
        self.gui = gui
        self.plugin_action = gui.iactions['Category Tags']
        self.db = self.gui.current_db
        self.action = self.plugin_action.actions['Import User Catetories']
        self._init_controls()

    def _init_controls(self):

        l = QVBoxLayout()
        self.setLayout(l)

        format_groupbox = QGroupBox(_('Choose format type'))
        format_groupbox_l = QGridLayout()
        format_groupbox.setLayout(format_groupbox_l)

        scroll = QScrollArea()
        scroll.setVerticalScrollBarPolicy(Qt.ScrollBarAsNeeded)
        scroll.setHorizontalScrollBarPolicy(Qt.ScrollBarAsNeeded)
        scroll.setWidgetResizable(True)
        scroll.setSizePolicy(QSizePolicy.Expanding, QSizePolicy.Preferred)
        scroll.setWidget(format_groupbox)
        scroll.setMinimumHeight(80)
        l.addWidget(scroll)

        self.format_opts = {}
        self.format_settings = defaultdict(dict)
        idx = 0
        for name, fmt in self.action.fmts.items():
            description = fmt.description
            opt_name = name + '_opt'
            setattr(self, opt_name, QRadioButton(description))
            opt = getattr(self, opt_name)
            format_groupbox_l.addWidget(opt, idx, 0, 1, 1)
            if fmt.config_widget():
                button = QToolButton()
                button.setIcon(get_icon('gear.png'))
                button.clicked.connect(partial(self._on_format_settings_button_clicked, name))            
                format_groupbox_l.addWidget(button, idx, 1, 1, 1)
            self.format_opts[name] = opt
            if idx == 0:
                opt.setChecked(True)
            idx += 1

        file_groupbox = QGroupBox(_('File path options'))
        file_groupbox_l = QVBoxLayout()
        file_groupbox.setLayout(file_groupbox_l)
        l.addWidget(file_groupbox)

        self.file_box = QGroupBox(_('&Choose file path:'))
        file_groupbox_l.addWidget(self.file_box)
        csv_layout = QVBoxLayout()
        self.file_box.setLayout(csv_layout)
        self.file_combo = DragDropComboBox(self, drop_mode='file')
        csv_layout.addWidget(self.file_combo)
        hl1 = QHBoxLayout()
        csv_layout.addLayout(hl1)
        hl1.addWidget(self.file_combo, 1)
        self.choose_file_button = QToolButton(self)
        self.choose_file_button.setToolTip(_('Choose file'))
        self.choose_file_button.setIcon(get_icon('document_open.png'))
        self.choose_file_button.clicked.connect(self._choose_file)
        hl1.addWidget(self.choose_file_button)
        
        l.addStretch(1)

        self.setMinimumSize(400,500)

    def _choose_file(self):

        files = choose_files(None, _('Choose categories file dialog'), _('Choose file'),
                             all_files=True, select_only_single_file=True)
        if not files:
            return
        file_path = files[0]
        if iswindows:
            file_path = os.path.normpath(file_path)
        
        if not file_path:
            return

        self.block_events = True
        existing_index = self.file_combo.findText(file_path, Qt.MatchExactly)
        if existing_index >= 0:
            self.file_combo.setCurrentIndex(existing_index)
        else:
            self.file_combo.insertItem(0, file_path)
            self.file_combo.setCurrentIndex(0)
        self.block_events = False

#    def _file_opt_changed(self):
#        self.file_box.setEnabled(self.predefined_opt.isChecked())

    def _on_format_settings_button_clicked(self, fmt_name):
        fmt = self.action.fmts[fmt_name]
        fmt_settings = self.format_settings[fmt_name]

        if not fmt_settings:
            fmt_settings = fmt.default_settings()
        config_widget = fmt.config_widget()
        if config_widget:
            name = '{}_settings'.format(fmt_name)
            title = fmt.description
            if issubclass(config_widget, QDialog):
                # config_widget is a dialog
                d = config_widget(self, self.gui, fmt, name, title)
            else:
                # config_widget is a qwidget
                d = SettingsWidgetDialog(name, self, self.gui, config_widget, fmt, title=title)

            if fmt_settings:
                d.load_settings(fmt_settings)
            if d.exec_() == d.Accepted:
                self.format_settings[fmt_name] = d.settings

class ImportCategoriesConfigWidget(QWidget):
    def __init__(self, gui):
        QWidget.__init__(self)
        self.gui = gui
        self.db = self.gui.current_db
        self._init_controls()

    def _init_controls(self):

        l = QVBoxLayout()
        self.setLayout(l)

        self.options_tab = OptionsWidget(self.gui)
        self.match_tab = MatchWidget(self, self.gui, {})
        self.tw = QTabWidget(self)
        self.tw.addTab(self.options_tab, 'Import Settings')
        self.tw.addTab(self.match_tab, 'Match Settings')
        l.addWidget(self.tw)
        
        l.addStretch(1)

        self.setMinimumSize(400,500)

    def load_settings(self, settings):
        if settings:            
            opt = self.options_tab.format_opts.get(settings['format_opt'])
            if opt:
                opt.setChecked(True)
                fmt_name = settings['format_opt']
                self.options_tab.format_settings[fmt_name] = settings['format_settings']

            self.options_tab.file_combo.populate_items(settings.get('file_list', []), settings['path_to_file'])
            #self.options_tab.file_combo.setCurrentText(settings['path_to_file'])
            self.match_tab._table.model().update_match_rules_for_categories(settings['match_rules_for_categories'])
            self.match_tab.add_multiple_matches_chk.setChecked(settings.get('add_multiple_matches', True))

    def save_settings(self):
        settings = {}
        for name, opt in self.options_tab.format_opts.items():
            if opt.isChecked():
                settings['format_opt'] = name
                settings['format_settings'] = self.options_tab.format_settings.get(name, {})
                break

        settings['path_to_file'] = self.options_tab.file_combo.currentText()
        settings['file_list'] = self.options_tab.file_combo.get_items_list()
        settings['match_rules_for_categories'] = self.match_tab._table.model().get_match_rules_for_categories()
        settings['add_multiple_matches'] = self.match_tab.add_multiple_matches_chk.isChecked()
        self.match_tab.save_table_state()
        return settings

class ImportPage(QWizardPage):

    ID = 1

    def __init__(self, parent, gui, action):
        QWizardPage.__init__(self, parent)
        self.gui = gui
        self.db = gui.current_db
        self.info = parent.info
        self.action = action
        self.library_config = parent.library_config
        self.init_controls()

    def init_controls(self):
        l = QVBoxLayout()
        self.setLayout(l)
        self.config_widget = ImportCategoriesConfigWidget(self.gui)
        l.addWidget(self.config_widget)

    def initializePage(self):
        splitter_state = self.info['state'].get('import_splitter_state', None)
        if splitter_state is not None:
            self.splitter.restoreState(splitter_state)
        self.config_widget.load_settings(self.library_config[cfg.KEY_IMPORT_LAST_SETTINGS])

    def isComplete(self):
        return True

    def validatePage(self):
        settings = self.config_widget.save_settings()
        is_valid = self.action.validate(settings)
        if is_valid is True:
            self.info.update(settings)
        else:
            msg, details = is_valid
            error_dialog(self, msg, details, show=True)
            return False
        self.pivot_format = self.action.get_pivot_format(self.gui, self.info)
        if not self.pivot_format:
            return False
        db = self.gui.library_view.model().db
        self.action.create_category_caches()
        #
        match_rules_for_categories = self.info.get('match_rules_for_categories', {})
        self.action.create_library_matches_for_imported_items_map(self.pivot_format, match_rules_for_categories)
        self.info['library_matches_for_imported_items_map'] = self.action.library_matches_for_imported_items_map
        self.info['pivot_format'] = self.pivot_format
        self.library_config[cfg.KEY_IMPORT_LAST_SETTINGS] = settings
        return True
