#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

from collections import defaultdict
import copy

from qt.core import (Qt, QApplication, QModelIndex, QAbstractTableModel,
                     QSortFilterProxyModel, QBrush, pyqtSignal)

from calibre import prints
from calibre.constants import DEBUG

from calibre_plugins.category_tags.advanced_matching.models import UP, DOWN
from calibre_plugins.category_tags.user_categories import validate_category_match_rules

try:
    load_translations()
except NameError:
    prints("Category Tags::user_categories/models.py - exception when loading translations")

class ResolveModel(QAbstractTableModel):

    error = pyqtSignal(str, str)

    def __init__(self, gui, library_matches_for_imported_items_map):
        QAbstractTableModel.__init__(self)
        self.gui = gui
        self.db = self.gui.current_db
        self.library_matches_for_imported_items_map = library_matches_for_imported_items_map
        self.all_items_data = []
        for item_type, item_match_dict in library_matches_for_imported_items_map.items():
            for item, matches in item_match_dict.items():
                if len(matches) == 0:
                    item_alt = '*** No Matches ***'
                    status = 'unmatched'
                elif len(matches) == 1:
                    item_alt = list(matches)[0]
                    status = 'matched'
                elif len(matches) > 1:
                    item_alt = '*** Multiple Matches ***'
                    status = 'multiple'
                d = {'item': item, 'item_type': item_type, 'matches': matches, 'item_alt': item_alt, 'status': status}
                self.all_items_data.append(d)
        self.col_map = ['item','item_alt','item_type']
        self.editable_columns = ['item','item_alt','item_type']
        self.hidden_cols_1 = ['item_alt']
        self.hidden_cols_2 = ['item']
        self.col_min_width = {
            'item': 300,
            'item_alt': 300,
            'item_type': 100
        }
        all_headers = [('Item'),_('Item'),_('Item Type')]
        self.headers = all_headers

    def rowCount(self, parent):
        if parent and parent.isValid():
            return 0
        return len(self.all_items_data)

    def columnCount(self, parent):
        if parent and parent.isValid():
            return 0
        return len(self.headers)

    def headerData(self, section, orientation, role):
        if role == Qt.ItemDataRole.DisplayRole and orientation == Qt.Orientation.Horizontal:
            return self.headers[section]
        elif role == Qt.ItemDataRole.DisplayRole and orientation == Qt.Orientation.Vertical:
            return section + 1
        return None

    def data(self, index, role):
        if not index.isValid():
            return None;
        row, col = index.row(), index.column()
        if row < 0 or row >= len(self.all_items_data):
            return None
        item_data = self.all_items_data[row]
        col_name = self.col_map[col]
        value = item_data.get(col_name, '')
        status = item_data.get('status')
        
        if role in [Qt.ItemDataRole.DisplayRole, Qt.ItemDataRole.UserRole, Qt.ItemDataRole.EditRole]:
            return value              

        elif role == Qt.ItemDataRole.ToolTipRole:
            tooltip = _('A matching book was found in your calibre library')
            if status == 'multiple':
                tooltip = _('Multiple matches found for this title/author.\n' \
                          'Resolve this by selecting your match below.')
            elif status == 'unmatched':
                tooltip = _('No matching book found in your library.\n' \
                          'Add an empty book or search for a match below.')
            elif status == 'empty':
                tooltip = _('An empty book will be added if you save this list')
            return tooltip

        elif role == Qt.ItemDataRole.ForegroundRole:
            color = None
            if status == 'multiple':
                color = Qt.GlobalColor.magenta
            elif status == 'unmatched':
                color = Qt.GlobalColor.red
            elif status == 'empty':
                color = Qt.GlobalColor.blue
            if color is not None:
                return QBrush(color)

        return None

    def setData(self, index, value, role):
        done = False

        row, col = index.row(), index.column()
        item_data = self.all_items_data[row]
        val = str(value).strip()
        col_name = self.col_map[col]
        
        if role == Qt.ItemDataRole.EditRole:
            item_data[col_name] = val
            done = True
            
        return done

    def flags(self, index):
        flags = QAbstractTableModel.flags(self, index)
        if index.isValid():
            item_data = self.all_items_data[index.row()]
            col_name = self.col_map[index.column()]
            if col_name in self.editable_columns:
                flags |= Qt.ItemFlag.ItemIsEditable
        return flags

    def insertRows(self, row, count, idx):
        self.beginInsertRows(QModelIndex(), row, row + count - 1)
        for i in range(0, count):
            item_data = {}
            item_data['item'] = ''
            item_data['item_alt'] = ''
            item_data['matches'] = set()
            self.all_items_data.insert(row + i, item_data)
        self.endInsertRows()
        return True

    def removeRows(self, row, count, idx):
        self.beginRemoveRows(QModelIndex(), row, row + count - 1)
        for i in range(0, count):
            self.all_items_data.pop(row + i)
        self.endRemoveRows()
        return True

    def get_library_matches_for_imported_items_map(self):
        res = defaultdict(dict)
        for item_data in self.all_items_data:
            res[item_data['item_type']][item_data['item']] = item_data['matches']
        return res

class ResolveSortFilterModel(QSortFilterProxyModel):

    def __init__(self, parent):
        QSortFilterProxyModel.__init__(self, parent)
        self.setSortRole(Qt.ItemDataRole.UserRole)
        self.setSortCaseSensitivity(Qt.CaseInsensitive)
        self.filter_criteria = 'all'
        self.item_type = None

    def filterAcceptsRow(self, sourceRow, sourceParent):
        index = self.sourceModel().index(sourceRow, 0, sourceParent)
        item_data = self.sourceModel().all_items_data[index.row()]
        if self.item_type:
            is_item_type = item_data['item_type'] == self.item_type
        else:
            is_item_type = True
        if self.filter_criteria == 'all':
            return is_item_type
        if self.filter_criteria == 'matched':
            return is_item_type and (item_data['status'] == 'matched')
        if self.filter_criteria == 'multiple':
            return is_item_type and (item_data['status'] == 'multiple')
        if self.filter_criteria == 'unmatched':
            return is_item_type and (item_data['status'] == 'unmatched')
        return False

    def set_filter_criteria(self, filter_value, item_type):
        self.filter_criteria = filter_value
        self.item_type = item_type
        self.invalidateFilter()

class MatchModel(QAbstractTableModel):

    error = pyqtSignal(str, str)

    def __init__(self, gui, match_rules_for_categories={}):
        QAbstractTableModel.__init__(self)
        self.update_match_rules_for_categories(match_rules_for_categories)
        self.gui = gui
        self.db = self.gui.current_db
        self.col_map = ['name','settings','errors']
        self.editable_columns = ['name']
        #self.hidden_cols = ['errors']
        self.hidden_cols = []
        self.col_min_width = {
            'name': 300,
            'settings': 50,
            'errors': 250
        }
        all_headers = [('Category'),_('Settings'),_('Errors')]
        self.headers = all_headers

    def rowCount(self, parent):
        if parent and parent.isValid():
            return 0
        return len(self.categories_match_rules)

    def columnCount(self, parent):
        if parent and parent.isValid():
            return 0
        return len(self.headers)

    def headerData(self, section, orientation, role):
        if role == Qt.ItemDataRole.DisplayRole and orientation == Qt.Orientation.Horizontal:
            return self.headers[section]
        elif role == Qt.ItemDataRole.DisplayRole and orientation == Qt.Orientation.Vertical:
            return section + 1
        return None

    def data(self, index, role):
        if not index.isValid():
            return None;
        row, col = index.row(), index.column()
        if row < 0 or row >= len(self.categories_match_rules):
            return None
        category_match_rules = self.categories_match_rules[row]
        col_name = self.col_map[col]
        value = category_match_rules.get(col_name, '')
        error = category_match_rules.get('errors', '')
        
        if role in [Qt.ItemDataRole.DisplayRole, Qt.ItemDataRole.UserRole, Qt.ItemDataRole.EditRole]:
            if col_name in ['settings']:
                pass
            elif col_name == 'errors':
                if error:
                    return _('Validation Error. Double click for details')
            else:
                return value

        elif role == Qt.ItemDataRole.DecorationRole:
            if col_name == 'errors':
                if error:
                    return QIcon(get_icon('dialog_error.png'))
                
        elif role == Qt.ItemDataRole.ToolTipRole:
            if col_name == 'errors':
                if error:
                    return error

        elif role == Qt.ItemDataRole.ForegroundRole:
            color = None
            if error:
                color = Qt.GlobalColor.red
            if color is not None:
                return QBrush(color)

        return None

    def setData(self, index, value, role):
        done = False

        row, col = index.row(), index.column()
        category_match_rules = self.categories_match_rules[row]
        val = str(value).strip()
        col_name = self.col_map[col]
        
        if role == Qt.ItemDataRole.EditRole:
            if col_name == 'settings':
                pass
            elif col_name == 'name':
                old_name = self.data(index, Qt.ItemDataRole.DisplayRole)
                names = self.get_names()
                if old_name in names:
                    names.remove(old_name)
                if not val:
                    msg = _('Invalid value')
                    details = _('Name cannot be empty'.format(val))
                    self.error.emit(msg, details)
                    return False
                if val in names:
                    msg = _('Name must be unique')
                    details = _('Name ({}) is already used by another menu'.format(val))
                    self.error.emit(msg, details)
                    return False
                else:
                    category_match_rules[col_name] = val
            else:
                category_match_rules[col_name] = val
            done = True
            
        return done

    def flags(self, index):
        flags = QAbstractTableModel.flags(self, index)
        if index.isValid():
            category_match_rules = self.categories_match_rules[index.row()]
            col_name = self.col_map[index.column()]
            if col_name in self.editable_columns:
                flags |= Qt.ItemFlag.ItemIsEditable
        return flags

    def button_state(self, index):
        visible = False
        enabled = False
        row, col = index.row(), index.column()
        category_match_rules = self.categories_match_rules[row]
        name = category_match_rules.get('name')
        if name:
            visible = True
            enabled = True
        return visible, enabled

    def insertRows(self, row, count, idx):
        self.beginInsertRows(QModelIndex(), row, row + count - 1)
        all_names = self.get_names()
        for i in range(0, count):
            category_match_rules = {}
            new_name = ''
            category_match_rules['name'] = new_name
            category_match_rules['settings'] = {}
            self.categories_match_rules.insert(row + i, category_match_rules)
        self.endInsertRows()
        return True

    def removeRows(self, row, count, idx):
        self.beginRemoveRows(QModelIndex(), row, row + count - 1)
        for i in range(0, count):
            self.categories_match_rules.pop(row + i)
        self.endRemoveRows()
        return True

    def move_rows(self, rows, direction=DOWN):
        srows = sorted(rows, reverse=direction == DOWN)
        for row in srows:
            pop = self.categories_match_rules.pop(row)
            self.categories_match_rules.insert(row+direction, pop)
        self.layoutChanged.emit()

    def get_names(self):
        names = []
        col = self.col_map.index('name')
        for row in range(self.rowCount(QModelIndex())):
            index = self.index(row, col, QModelIndex())
            name = self.data(index, Qt.ItemDataRole.DisplayRole)
            # empty name belong to separators, dont include
            if name:
                names.append(name)
        return names

    def update_match_rules_for_categories(self, match_rules_for_categories):
        self.categories_match_rules = []
        for category, match_rules in match_rules_for_categories.items():
            self.categories_match_rules.append({'name': category, 'settings': copy.deepcopy(match_rules), 'error': ''})
        self.layoutChanged.emit()

    def get_match_rules_for_categories(self):
        match_rules_for_categories = {}
        for category_match_rules in self.categories_match_rules:
            category = category_match_rules['name']
            match_rules = category_match_rules['settings']
            match_rules_for_categories[category] = match_rules
        return match_rules_for_categories
            

    def validate(self):
        error_col = self.col_map.index('errors')
        all_categories_match_rules_valid  = True
        for row, category_match_rules in enumerate(self.categories_match_rules):
            is_category_match_rules_valid = validate_category_match_rules(self.gui, category_match_rules, get_cols(self.db))
            if is_category_match_rules_valid is not True:
                all_categories_match_rules_valid = False
                msg, details = is_category_match_rules_valid
                category_match_rules['errors'] = details
                index = self.index(row, error_col, QModelIndex())
                self.dataChanged.emit(index, index)
        return all_categories_match_rules_valid
