#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__   = 'GPL v3'
__copyright__ = '2021 additions by Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import csv
from collections import defaultdict

from qt.core import (Qt, QApplication, QWidget, QVBoxLayout, QGridLayout, QGroupBox,
                     QLabel, QLineEdit)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog

from calibre_plugins.category_tags.user_categories.formats.base import UserCategoriesFormat
from calibre_plugins.category_tags.advanced_matching import column_metadata

try:
    load_translations()
except NameError:
    prints("Category Tags::user_categories/formats/items_for_categories.py - exception when loading translations")

class ItemsForCategoryConfigWidget(QWidget):

    def __init__(self, gui):
        QWidget.__init__(self)
        self.gui = gui
        self.db = self.gui.current_db
        self._init_controls()

    def _init_controls(self):

        l = QVBoxLayout()
        self.setLayout(l)
        
        options_groupbox = QGroupBox(_('Options'))
        options_groupbox_l = QGridLayout()
        options_groupbox.setLayout(options_groupbox_l)
        l.addWidget(options_groupbox)
        
        self.authors_sep_label = QLabel(_('Separator for author like fields'))
        self.authors_sep_edit = QLineEdit()
        options_groupbox_l.addWidget(self.authors_sep_label, 0, 0, 1, 1)
        options_groupbox_l.addWidget(self.authors_sep_edit, 0, 1, 1, 1)

        self.tags_sep_label = QLabel(_('Separator for tag like fields'))
        self.tags_sep_edit = QLineEdit()
        options_groupbox_l.addWidget(self.tags_sep_label, 1, 0, 1, 1)
        options_groupbox_l.addWidget(self.tags_sep_edit, 1, 1, 1, 1)

        self.text_sep_label = QLabel(_('Separator for text and series fields'))
        self.text_sep_edit = QLineEdit()
        options_groupbox_l.addWidget(self.text_sep_label, 2, 0, 1, 1)
        options_groupbox_l.addWidget(self.text_sep_edit, 2, 1, 1, 1)

        csv_groupbox = QGroupBox(_('CSV Options'))
        csv_groupbox_l = QGridLayout()
        csv_groupbox.setLayout(csv_groupbox_l)
        l.addWidget(csv_groupbox)

        self.csv_delimiter_label = QLabel(_('CSV Delimiter'))
        self.csv_delimiter_edit = QLineEdit()
        csv_groupbox_l.addWidget(self.csv_delimiter_label, 0, 0, 1, 1)
        csv_groupbox_l.addWidget(self.csv_delimiter_edit, 0, 1, 1, 1)

    def load_settings(self, settings):
        if settings:
            self.authors_sep_edit.setText(settings['authors_sep'])
            self.tags_sep_edit.setText(settings['tags_sep'])
            self.text_sep_edit.setText(settings['text_sep'])
            self.csv_delimiter_edit.setText(settings['csv_delimiter'])

    def save_settings(self):
        settings = {}
        settings['authors_sep'] = self.authors_sep_edit.text()
        settings['tags_sep'] = self.tags_sep_edit.text()
        settings['text_sep'] = self.text_sep_edit.text()
        settings['csv_delimiter'] = self.csv_delimiter_edit.text()
        return settings


class CSVItemsForCategory(UserCategoriesFormat):

    name = 'csv_items_for_category'
    description = 'CSV: Items For Category'
    filters = [(_('Categories'), ['csv'])]
    _is_builtin = True

    def get_item_type_delimiter(self, item_type, format_settings={}):
        delegate = column_metadata(self.db, item_type)['delegate']
        if delegate == 'authors':
            return format_settings.get('authors_sep', ' & ')
        elif delegate in ['tags']:
            return format_settings.get('tags_sep', ', ')
        else:
            return format_settings.get('text_sep', ' ::: ')

    def to_pivot(self, file_path, settings):
        format_settings = settings.get('format_settings', {})
        csv_delimiter = format_settings.get('csv_delimiter', ',')
        pivot_format = defaultdict(dict)
        with open(file_path, 'r', encoding="utf-8-sig") as f:
            reader = csv.reader(f, delimiter=csv_delimiter, quoting=csv.QUOTE_MINIMAL)
            for idx, row in enumerate(reader):
                if idx == 0:
                    headers = row
                    if not headers == ['category','item_type','items']:
                        error_dialog(self.gui,
                                    _('Invalid CSV Headers'),
                                    _('CSV headers for this format should be: category, item_type, items'),
                                    show=True)
                        return {}
                else:
                    imported_category, item_type, items = row
                    if item_type not in self.all_item_types:
                        error_dialog(self.gui,
                                    _('Invalid item type'),
                                    _('Item type ({}) is not valid. Valid item types are: {}'.fomrat(item_type, self.all_item_types)),
                                    show=True)
                        return {}  
                    items = items.split(self.get_item_type_delimiter(item_type, format_settings))
                    items = [x.strip() for x in items]
                    for item in items:
                        item_categories = pivot_format.get(item_type, {}).get(item, {}).get('categories', set())
                        item_categories.add(imported_category)
                        pivot_format[item_type][item] = {'categories': item_categories}
        return pivot_format

    def from_user_categories(self, user_categories, file_path, settings):
        format_settings = settings.get('format_settings', {})
        csv_delimiter = format_settings.get('csv_delimiter', ',')
        d = defaultdict(set)
        for category, category_items in user_categories.items():
            for category_item in category_items:
                item, item_type, _ign = category_item
                category_hash = category+'|'+item_type
                items_for_category = d[category_hash]
                items_for_category.add(item)
                d[category_hash] = items_for_category

        with open(file_path, 'w', encoding='utf-8') as f:
            writer = csv.writer(f, delimiter=csv_delimiter, dialect='excel', quoting=csv.QUOTE_NONNUMERIC)
            writer.writerow(['category','item_type','items'])
            for category_hash, items in d.items():
                category, item_type = category_hash.split('|')
                item_type_delimiter = self.get_item_type_delimiter(item_type, format_settings)
                items = item_type_delimiter.join(list(items))
                writer.writerow([category, item_type, items])

    def validate(self, settings):
        if settings:
            for key in ['authors_sep','tags_sep','text_sep','csv_delimiter']:
                if not settings.get(key):
                    return _('Settings Error'), _('Some fields contain empty values.')
        return True

    def default_settings(self):
        return {
            'authors_sep': ' & ',
            'tags_sep': ', ',
            'text_sep': ':::',
            'csv_delimiter': ','
        }

    def config_widget(self):
        return ItemsForCategoryConfigWidget
