#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__   = 'GPL v3'
__copyright__ = '2021 additions by Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import csv
from collections import defaultdict, OrderedDict

from qt.core import (Qt, QApplication, QWidget, QVBoxLayout, QGridLayout, QGroupBox,
                     QLabel, QLineEdit)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog

from calibre_plugins.category_tags.user_categories.formats.base import UserCategoriesFormat

try:
    load_translations()
except NameError:
    prints("Category Tags::user_categories/formats/categories_for_item.py - exception when loading translations")

class CategoriesForItemConfigWidget(QWidget):

    def __init__(self, gui):
        QWidget.__init__(self)
        self.gui = gui
        self.db = self.gui.current_db
        self._init_controls()

    def _init_controls(self):

        l = QVBoxLayout()
        self.setLayout(l)

        csv_groupbox = QGroupBox(_('CSV Options'))
        csv_groupbox_l = QGridLayout()
        csv_groupbox.setLayout(csv_groupbox_l)
        l.addWidget(csv_groupbox)

        self.csv_delimiter_label = QLabel(_('CSV Delimiter'))
        self.csv_delimiter_edit = QLineEdit()
        csv_groupbox_l.addWidget(self.csv_delimiter_label, 0, 0, 1, 1)
        csv_groupbox_l.addWidget(self.csv_delimiter_edit, 0, 1, 1, 1)

    def load_settings(self, settings):
        if settings:
            self.csv_delimiter_edit.setText(settings['csv_delimiter'])

    def save_settings(self):
        settings = {}
        settings['csv_delimiter'] = self.csv_delimiter_edit.text()
        return settings

class CSVCategoriesForItem(UserCategoriesFormat):

    name = 'csv_categories_for_item'
    description = 'CSV: Categories For Item'
    filters = [(_('Categories'), ['csv'])]
    _is_builtin = True

    def to_pivot(self, file_path, settings):
        format_settings = settings.get('format_settings', {})
        csv_delimiter = format_settings.get('csv_delimiter', ',')
        pivot_format = defaultdict(dict)
        with open(file_path, 'r', encoding="utf-8-sig") as f:
            reader = csv.reader(f, delimiter=csv_delimiter, quoting=csv.QUOTE_MINIMAL)
            for idx, row in enumerate(reader):
                if idx == 0:
                    headers = row
                    if not headers in ( ['item','item_type','categories'], ['item','item_type','categories','link','note'] ):
                        error_dialog(self.gui,
                                    _('Invalid CSV Headers'),
                                    _('CSV headers for this format should be: item, item_type, categories'),
                                    show=True)
                        return {}
                else:
                    try:
                        item, item_type, imported_categories, link, note = row
                    except:
                        item, item_type, imported_categories = row
                        note = None
                        link = None
                    if item_type not in self.all_item_types:
                        error_dialog(self.gui,
                                    _('Invalid item type'),
                                    _('Item type ({}) is not valid. Valid item types are: {}'.fomrat(item_type, self.all_item_types)),
                                    show=True)
                        return {}                    
                    imported_categories = imported_categories.split(',')
                    imported_categories = [x.strip() for x in imported_categories if x.strip() != '']
                    item_categories = pivot_format.get(item_type, {}).get(item, {}).get('categories', set())
                    item_categories = item_categories.union(imported_categories)
                    pivot_format[item_type][item] = {'categories': item_categories, 'link': link, 'note': note}
        return pivot_format

    def from_user_categories(self, user_categories, file_path, settings):
        format_settings = settings.get('format_settings', {})
        csv_delimiter = format_settings.get('csv_delimiter', ',')
        link_map = {}
        all_fields = [ k for k,v in self.db.field_metadata.all_metadata().items() if v['datatype'] not in [None,'composite'] ]
        for item_type in all_fields:
            if self.db.new_api.has_link_map(item_type):
                link_map[item_type] = self.db.new_api.get_link_map(item_type).copy()
        d = {}
        for category, category_items in user_categories.items():
            for category_item in category_items:
                item, item_type, _ign = category_item
                item_hash = item_type+'|'+item
                item_categories = d.get(item_hash, set())
                item_categories.add(category)
                d[item_hash] = item_categories
        # Get all notes
        all_notes = OrderedDict()
        for item_type, all_item_ids in self.db.new_api.get_all_items_that_have_notes(field_name=None).items():
            for item_id in all_item_ids:
                item = self.db.new_api.get_item_name(item_type, item_id)
                item_hash = item_type+'|'+item
                try:
                    note = self.get_note(item_type, item)
                except:
                    note = None
                if note:
                    all_notes[item_hash] = self.get_note(item_type, item)

        with open(file_path, 'w', encoding='utf-8') as f:
            writer = csv.writer(f, delimiter=csv_delimiter, dialect='excel', quoting=csv.QUOTE_NONNUMERIC)
            writer.writerow(['item','item_type','categories','link','note'])
            for item_hash, item_categories in d.items():
                item_type, item = item_hash.split('|')
                categories = ','.join(list(item_categories))
                note = all_notes.get(item_hash, '')
                try: del all_notes[item_hash]
                except: pass
                link = link_map.get(item_type).get(item, '')
                try: del link_map[item_type][item]
                except: pass
                writer.writerow([item, item_type, categories, link, note])
            # Add items that have notes but not associated with any user categories
            for item_hash, note in all_notes.items():
                item_type, item = item_hash.split('|')
                link = link_map.get(item_type, {}).get(item, '')
                try: del link_map[item_type][item]
                except: pass
                writer.writerow([item, item_type, '', link, note])
            # Add items that have links but not associated with any user categories
            for item_type, type_link_map in link_map.items():
                for item, link in type_link_map.items():
                    writer.writerow([item, item_type, '', link, ''])


    def validate(self, settings):
        if settings:
            for key in ['csv_delimiter']:
                if not settings.get(key):
                    return _('Settings Error'), _('Some fields contain empty values.')
        return True

    def default_settings(self):
        return {'csv_delimiter': ','}

    def config_widget(self):
        return CategoriesForItemConfigWidget
