#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__   = 'GPL v3'
__copyright__ = '2021 additions by Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from collections import OrderedDict

from calibre import prints
from calibre.constants import DEBUG

from calibre_plugins.category_tags.user_categories.formats.base import UserCategoriesFormat
from calibre_plugins.category_tags.user_categories.formats.categories_for_item import CSVCategoriesForItem
from calibre_plugins.category_tags.user_categories.formats.items_for_category import CSVItemsForCategory
from calibre_plugins.category_tags.user_categories.formats.json import JSONFormat

try:
    load_translations()
except NameError:
    prints("Category Tags::user_categories/formats/__init__.py - exception when loading translations")


BUILTIN_FMTS = [
    CSVItemsForCategory,
    CSVCategoriesForItem,
    JSONFormat
]


def get_user_fmts(user_modules):
    user_fmts = {}
    for cls in user_modules.get_classes(class_filters=[UserCategoriesFormat]):
        name = cls.name
        # must define a name attribute
        if name in ['', 'no name provided']:
            continue
        user_fmts[name] = cls            
    return user_fmts

def get_fmts(gui, user_modules=None):
    all_fmts = OrderedDict()

    _builtin_fmts = OrderedDict()
    for cls in BUILTIN_FMTS:
        _builtin_fmts[cls.name] = cls

    if user_modules:
        user_fmts = get_user_fmts(user_modules)
    else:
        user_fmts = {}

    for fmt_name, fmt_cls in _builtin_fmts.items():
        fmt = fmt_cls(gui)
        all_fmts[fmt_name] = fmt

    for fmt_name, fmt_cls in user_fmts.items():
        try:
            if fmt_name in all_fmts.keys():
                if DEBUG:
                    prints('Category Tags: User Categories: another format with the name ({}) already exists'.format(fmt_name))
                continue
            fmt = fmt_cls(gui)
            all_fmts[fmt_name] = fmt
        except Exception as e:
            import traceback
            if DEBUG:
                prints('Category Tags: User Categories: Error intializing user fmt: {}\n{}'.format(fmt_name, traceback.format_exc()))

    return all_fmts
