#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__   = 'GPL v3'
__copyright__ = '2021 additions by Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from qt.core import (Qt, QApplication, QVBoxLayout, QDialogButtonBox)

from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog, choose_save_file

from calibre_plugins.category_tags.user_categories.page_import import OptionsWidget
from calibre_plugins.category_tags.common_utils import get_file_path, SizePersistedDialog
import calibre_plugins.category_tags.config as cfg

try:
    load_translations()
except NameError:
    prints("Category Tags::user_categories/dialog_export.py - exception when loading translations")

class ExportCategoriesConfigWidget(OptionsWidget):

    def _choose_file(self):

        file_path = choose_save_file(self.gui, 'export-user-categories', _('Choose file'), filters=[
            (_('Export User Categories'), ['csv'])], all_files=False)
        
        if not file_path:
            return

        self.block_events = True
        existing_index = self.file_combo.findText(file_path, Qt.MatchExactly)
        if existing_index >= 0:
            self.file_combo.setCurrentIndex(existing_index)
        else:
            self.file_combo.insertItem(0, file_path)
            self.file_combo.setCurrentIndex(0)
        self.block_events = False

    def load_settings(self, settings):
        if settings:            
            opt = self.format_opts.get(settings['format_opt'])
            if opt:
                opt.setChecked(True)
                fmt_name = settings['format_opt']
                self.format_settings[fmt_name] = settings['format_settings']

            self.file_combo.populate_items(settings.get('file_list', []), settings['path_to_file'])
            #self.file_combo.setCurrentText(settings['path_to_file'])

    def save_settings(self):
        settings = {}
        for name, opt in self.format_opts.items():
            if opt.isChecked():
                settings['format_opt'] = name
                settings['format_settings'] = self.format_settings.get(name, {})
                break

        settings['path_to_file'] = self.file_combo.currentText()
        settings['file_list'] = self.file_combo.get_items_list()
        return settings

class ExportCategoriesDialog(SizePersistedDialog):

    def __init__(self, name, gui, action, title=_('Export Categories')):
        self.gui = gui
        self.db = gui.current_db
        self.action = action
        self.widget_cls = ExportCategoriesConfigWidget
        SizePersistedDialog.__init__(self, gui, name)
        self.library_config = cfg.get_library_config(self.db)
        self.setup_ui()
        self.setWindowTitle(title)
        self.resize_dialog()

    def setup_ui(self):
        self.widget = self.widget_cls(self.gui)
        l = QVBoxLayout()
        self.setLayout(l)
        l.addWidget(self.widget)

        self.button_box = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        self.button_box.accepted.connect(self.accept)
        self.button_box.rejected.connect(self.reject)
        l.addWidget(self.button_box)
        self.load_settings(self.library_config[cfg.KEY_EXPORT_LAST_SETTINGS])

    def load_settings(self, settings):
        self.widget.load_settings(settings)

    def save_settings(self):
        return self.widget.save_settings()

    def validate(self, settings):
        if hasattr(self.widget, 'validate'):
            return self.widget.validate(settings)
        else:
            return self.action.validate(settings)
    
    def accept(self):
        self.settings = self.save_settings()
        # Validate settings
        #is_valid = self.action.validate(self.settings)
        is_valid = self.validate(self.settings)
        if is_valid is not True:
            msg, details = is_valid
            error_dialog(self, msg, details, show=True)
            return
        self.library_config[cfg.KEY_EXPORT_LAST_SETTINGS] = self.settings
        cfg.set_library_config(self.db, self.library_config)
        SizePersistedDialog.accept(self)

