#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__   = 'GPL v3'
__copyright__ = '2021 additions by Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from qt.core import (Qt, QApplication)

import copy
from operator import itemgetter
from collections import defaultdict

from calibre import prints
from calibre.constants import iswindows, DEBUG
from calibre.gui2 import choose_save_file

from calibre_plugins.category_tags.common_utils import get_file_path, validate_writable_file_path
from calibre_plugins.category_tags.advanced_matching import HashBuilder
from calibre_plugins.category_tags.user_categories import get_cols, validate_all_categories_match_rules
from calibre_plugins.category_tags.user_categories.formats import get_fmts

try:
    load_translations()
except NameError:
    prints("Category Tags::user_categories/actions.py - exception when loading translations")

class ImportAction(object):

    name = 'Import User Catetories'
    display_tree = ['User Categories']
    
    def __init__(self, gui):
        self.gui = gui
        self.db = self.gui.current_db
        self.hash_maps = {}
        self.all_item_types = list(get_cols(self.db))

    def all_field_names(self, item_type):
        return list(self.gui.current_db.new_api.all_field_names(item_type))

    def clear_caches(self):
        self.cat_sets = {}
        self.hash_maps = {}
        self.library_matches_for_imported_items_map = {}
        self.hash_builders = {}

    def create_category_caches(self):
        # Create a dictionary of sets for performant lookup of matchship
        self.cat_sets = {}
        user_categories = self.gui.library_view.model().db.prefs.get('user_categories', {})
        for category_name, category_value in user_categories.items():
            self.cat_sets[category_name] = set([item + '|' + item_type for (item, item_type, _ign) in category_value])

    def create_item_type_hash_maps(self, item_type, match_rules):
        hmap = self.hash_maps.get(item_type)
        if not hmap:
            algorithms = self.gui.iactions['Category Tags'].algorithms
            hash_builder = HashBuilder(self.gui, algorithms, match_rules)
            # cache hash builder to use later for imported items, this save performance
            self.hash_builders[item_type] = hash_builder
            self.hash_maps[item_type] = defaultdict(set)
            hmap = self.hash_maps[item_type]
            items = self.all_field_names(item_type)
            for item in items:
                hashes = hash_builder.from_category_item(item)
                for hash_ in hashes:
                    if hash_:
                        hmap[hash_].add(item)
        return hmap

    def create_library_matches_for_imported_items_map(self, pivot_format, match_rules_for_categories):
        self.library_matches_for_imported_items_map = defaultdict(dict)
        for item_type, record in pivot_format.items():
            default_match_rules = [ { "algos": [ { "name": "Identical Match", "settings": {} } ], "field": item_type } ]
            match_rules = match_rules_for_categories.get(item_type, default_match_rules)
            hmap = self.create_item_type_hash_maps(item_type, match_rules)
            for item, categories_for_item in record.items():
                matches = set()
                hash_builder = self.hash_builders[item_type]
                item_hashes = hash_builder.from_category_item(item)
                for hash_ in item_hashes:
                    matched_items_for_hash = hmap[hash_]
                    matches |= matched_items_for_hash
                self.library_matches_for_imported_items_map[item_type][item] = matches

    def add_categories_for_item(
            self,
            item,
            categories_for_item,
            item_type,
            user_categories,
            case_insensitive_keys,
            match_rules,
            link,
            link_map,
            note,
            add_multiple_matches=True):
        hmap = self.create_item_type_hash_maps(item_type, match_rules)
        matches = self.library_matches_for_imported_items_map.get(item_type, {}).get(item, set())
        if matches:
            if note:
                for match in matches:
                    if self.db.new_api.field_supports_notes(item_type):
                        item_id = self.db.new_api.get_item_id(item_type, match, case_sensitive=True)
                        self.db.new_api.set_notes_for(item_type,item_id,note)

            if link:
                for match in matches:
                    if self.db.new_api.has_link_map(item_type):
                        if not link_map.get(item_type):
                            link_map[item_type] = {}
                        link_map[item_type][match] = link

            for cat_name in categories_for_item:
                #if not user_categories.get(cat_name):
                if not cat_name in case_insensitive_keys:
                    user_categories[cat_name] = []
                    case_insensitive_keys.add(cat_name.lower())

                if not add_multiple_matches:
                    if len(matches) > 1:
                        continue

                for match in matches:
                    lookup_value = match + '|' + item_type
                    cat_set = self.cat_sets.get(cat_name, set())
                    if lookup_value not in cat_set:
                        #
                        user_categories[cat_name].append([ match, item_type, 0])
                        # Add to cat_sets
                        cat_set.add(lookup_value)
                        self.cat_sets[cat_name] = cat_set
                        #
                        # Re-sort the collection
                        l = []
                        for n in sorted(user_categories[cat_name], key=itemgetter(0)):
                            l.append(n)
                        user_categories[cat_name] = l
            return True
        else:
            return False

    def refresh(self, user_categories):
        db = self.gui.library_view.model().db
        
        # Order is important. The categories must be removed before setting
        # the preference because setting the pref recomputes the dynamic categories
        db.field_metadata.remove_user_categories()
        db.new_api.set_pref('user_categories', user_categories)
        db.new_api.refresh_search_locations()
        self.gui.tags_view.recount()
        db.new_api.clear_search_caches()
        self.gui.library_view.model().refresh()

    def create_top_level_nodes(self, updated_categories):
        db = self.gui.library_view.model().db
        user_categories = dict.copy(db.prefs.get('user_categories', {}))

        nodes_before = [node for node in user_categories.keys()]
        top_level_before = set([node.partition('.')[0] for node in nodes_before])

        nodes_after = [node for node in updated_categories.keys()]
        new_nodes = set(nodes_after).difference(set(nodes_before))
        new_top_level_nodes = set([node.partition('.')[0] for node in new_nodes if not node.partition('.')[0] in top_level_before])

        for node in new_top_level_nodes:
            user_categories[node] = []
        self.refresh(user_categories)

    def get_pivot_format(self, gui, settings):
        fmt_name = settings['format_opt']

        file_path = settings['path_to_file']

        if not file_path:
            return

        # Clear any caches from previous runs
        self.clear_caches()

        pivot_format = self.fmts[fmt_name].to_pivot(file_path, settings)
        
        return pivot_format

    def validate(self, settings):
        if settings.get('file_location_opt') == 'predefined':
            filename = settings.get('path_to_file')
            if not filename:
                return (_('No filename specified'), _('You must specify a valid filename'.format(file_path)))

        categories_match_rules = []        
        match_rules_for_categories = settings.get('match_rules_for_categories', {})
        for category, match_rules in match_rules_for_categories.items():
            categories_match_rules.append({'name': category, 'settings': copy.deepcopy(match_rules), 'error': ''})
        is_valid = validate_all_categories_match_rules(self.gui, get_cols(self.gui.current_db), categories_match_rules)
        if is_valid is not True:
            msg, details = _('Invalid match rules'), _('Some match rules are not valid. Double click on error cell for more details.')
            return (msg, details)
        return True

    def config_widget(self):
        return ImportCategoriesConfigWidget

    def on_modules_update(self, user_modules):
        self.fmts = {}
        if DEBUG:
            prints('Category Tags: User Categories: {}: running on_modules_update()'.format(self.name))
        self.fmts = get_fmts(self.gui, user_modules)

class ExportAction(ImportAction):

    name = 'Export User Catetories'
    display_tree = ['User Categories']

    def run(self, gui, settings):
        db = gui.library_view.model().db
        user_categories = dict.copy(db.prefs.get('user_categories', {}))

        fmt_name = settings['format_opt']

        filters = self.fmts[fmt_name].filters

        file_path = settings['path_to_file']
        if not file_path:
            return

        self.fmts[fmt_name].from_user_categories(user_categories, file_path, settings)

    def validate(self, settings):
        filename = settings.get('path_to_file')
        if filename:
            file_path = get_file_path(filename)
            if not validate_writable_file_path(file_path):
                return (_('Invalid filename'), _('Filename: {} is not a valid filename'.format(file_path)))
        else:
            return (_('No filename specified'), _('You must specify a valid filename'))
        return True

    def config_widget(self):
        return ExportCategoriesConfigWidget
