#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__   = 'GPL v3'
__copyright__ = '2021, by Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import copy

from qt.core import (QWidget, QVBoxLayout, QPushButton)

from calibre.gui2 import dynamic, info_dialog

from calibre_plugins.category_tags.common_utils import (KeyboardConfigDialog,
                                        PrefsViewerDialog, debug_print)

try:
    load_translations()
except NameError:
    debug_print("CategoryTags::config.py - exception when loading translations")
    pass

KEY_SCHEMA_VERSION = 'schemaVersion'
DEFAULT_SCHEMA_VERSION = 1.0

# Per library settings are persisted in the calibre library database.
# For performance reasons I am just using a single parent key of 'Settings'
# so as to minimise the number of writes to the database. The reading of
# settings is done at the time the database is opened so reads are cached.
PREFS_NAMESPACE = 'CategoryTagsPlugin'

PREFS_KEY_SETTINGS = 'settings'

KEY_IMPORT_LAST_SETTINGS = 'importLastSettings'
KEY_EXPORT_LAST_SETTINGS = 'exportLastSettings'
KEY_ALGORITHMS_TABLE_STATE = 'algorithmsTableState'

DEFAULT_LIBRARY_VALUES = {
                          KEY_IMPORT_LAST_SETTINGS: {},
                          KEY_EXPORT_LAST_SETTINGS: {},
                          KEY_SCHEMA_VERSION: DEFAULT_SCHEMA_VERSION
                         }

def migrate_library_config_if_required(db, library_config):
    schema_version = library_config.get(KEY_SCHEMA_VERSION, 0)
    if schema_version == DEFAULT_SCHEMA_VERSION:
        return
    # We have changes to be made - mark schema as updated
    library_config[KEY_SCHEMA_VERSION] = DEFAULT_SCHEMA_VERSION

    # Any migration code in future will exist in here.
    if schema_version < 1.0:
        pass

    get_missing_values_from_defaults(DEFAULT_LIBRARY_VALUES, library_config)

def get_missing_values_from_defaults(default_settings, settings):
    '''add keys present in default_settings and absent in setting'''
    for k, default_value in default_settings.items():
        try:
            setting_value = settings[k]
            if isinstance(default_value, dict):
                get_missing_values_from_defaults(default_value, setting_value)
        except KeyError:
            settings[k] = copy.deepcopy(default_value)

def get_library_config(db):
    library_config = db.prefs.get_namespaced(PREFS_NAMESPACE, PREFS_KEY_SETTINGS, copy.deepcopy(DEFAULT_LIBRARY_VALUES))
    migrate_library_config_if_required(db, library_config)
    return library_config

def set_library_config(db, library_config):
    db.prefs.set_namespaced(PREFS_NAMESPACE, PREFS_KEY_SETTINGS, library_config)


class ConfigWidget(QWidget):

    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.plugin_action = plugin_action
        self.column_keys = self.plugin_action.gui.current_db.field_metadata.displayable_field_keys()
        self._initialise_controls()

        self.library_config = get_library_config(plugin_action.gui.current_db)

    def _initialise_controls(self):
        layout = QVBoxLayout(self)
        self.setLayout(layout)

        keyboard_shortcuts_button = QPushButton(_('Keyboard shortcuts...'), self)
        keyboard_shortcuts_button.setToolTip(_(
                    'Edit the keyboard shortcuts associated with this plugin'))
        keyboard_shortcuts_button.clicked.connect(self._edit_shortcuts)
        layout.addWidget(keyboard_shortcuts_button)

        reset_confirmation_button = QPushButton(_('Reset disabled &confirmation dialogs'), self)
        reset_confirmation_button.setToolTip(_(
                    'Reset all show me again dialogs for the category plugin'))
        reset_confirmation_button.clicked.connect(self._reset_dialogs)
        layout.addWidget(reset_confirmation_button)

        view_prefs_button = QPushButton(_('&View library preferences...'), self)
        view_prefs_button.setToolTip(_(
                    'View data stored in the library database for this plugin'))
        view_prefs_button.clicked.connect(self._view_prefs)
        layout.addWidget(view_prefs_button)

    def _reset_dialogs(self):
        for key in dynamic.keys():
            if key.startswith('category_tags_') and key.endswith('_again') \
                                              and dynamic[key] is False:
                dynamic[key] = True
        info_dialog(self, _('Done'),
                _('Confirmation dialogs have all been reset'), show=True)

    def _edit_shortcuts(self):
        d = KeyboardConfigDialog(self.plugin_action.gui, self.plugin_action.action_spec[0])
        d.exec_()

    def _view_prefs(self):
        d = PrefsViewerDialog(self.plugin_action.gui, PREFS_NAMESPACE)
        d.exec_()

    def save_settings(self):
        pass
