#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__   = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

import copy
from functools import partial

from qt.core import (Qt, QApplication, QVBoxLayout, QHBoxLayout, QGridLayout,
                     QDialogButtonBox, QModelIndex, QIcon, QToolButton, QSpacerItem,
                     QSizePolicy, QSize, QAbstractTableModel, QTextBrowser, QGroupBox)


from calibre import prints
from calibre.constants import DEBUG
from calibre.gui2 import error_dialog, gprefs

from calibre_plugins.category_tags.common_utils import SizePersistedDialog, ImageTitleLayout
from calibre_plugins.category_tags.advanced_matching.models import AdvancedMatchModel, UP, DOWN
from calibre_plugins.category_tags.advanced_matching.views import AdvancedMatchTable

from calibre_plugins.category_tags.advanced_matching.rules_widget import RulesWidget

try:
    load_translations()
except NameError:
    prints("Category Tags::advanced_matching/dialogs.py - exception when loading translations")

help_text = _('''
<ul>
<li>You can match books using whatever combination of columns you want. You can choose from predefined match algorithms like: Identical Match, Similar Match, Soundex Match, Fuzzy Match. You can also use the Template Match to make your own custom algorithms, using calibre's template language</li>
<li>Each row represent a Match Tier where you can specify criteria for matching imported books to books in the library.</li>
<li>The uppermost Match Tiers should have the least aggressive match criteria (e.g. identical match).</li>
<li>Matching will start with the uppermost Match Tiers, and as soon as it finds matches, it will stop and will not go down to lower Match Tiers.</li>
<li>Note that a single Match Tier can yeild more than one match.</li>
''')

class AdvancedBookDuplicatesDialog(SizePersistedDialog):
    def __init__(self, parent, gui, selected_cols):
        SizePersistedDialog.__init__(self, parent, 'Advanced Book Duplicate Dialog')
        self.gui = gui
        self.db = gui.current_db
        self.selected_cols = selected_cols
        self._init_controls()
        self.resize_dialog()

    def _init_controls(self):
        self.setWindowTitle(_('Match books'))
        l = QVBoxLayout()
        self.setLayout(l)
        
        algorithms = self.gui.iactions['Category Tags'].algorithms
        self.rules_widget = RulesWidget(self, self.gui, algorithms, self.selected_cols, has_sort=False)
        l.addWidget(self.rules_widget, 1)
        
        self.button_box = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        self.button_box.accepted.connect(self._accept_clicked)
        self.button_box.rejected.connect(self.reject)
        l.addWidget(self.button_box)
        
        self.resize(600, 700)
        self.setSizePolicy(QSizePolicy.Preferred, QSizePolicy.Preferred)

    def load_settings(self, match_rules):
        if match_rules:
            if not self.rules_widget.restore_match_rules(match_rules):
                self.rules_widget.reset()

    def _accept_clicked(self):
        if not self.rules_widget.isComplete():
            error_dialog(self,
                        _('Match rule(s) has empty values'),
                        _('Match rule(s) missing field or algorithm. '
                          'To proceed, complete the missing data, or delete the match rule(s).'),
                        show=True)
            return
        self.match_rules = copy.deepcopy(self.rules_widget.get_match_rules())
        self.accept()

class AdvancedMatchDialog(SizePersistedDialog):
    def __init__(self, parent_tab, selected_cols, advanced_match_config):
        self.parent_tab = parent_tab
        self.gui = parent_tab.parent_page.gui
        self.db = self.gui.current_db
        self.selected_cols = selected_cols
        self.advanced_match_config = copy.deepcopy(advanced_match_config)
        self.KEY_TABLE_STATE = 'advancedMatchingTableState'
        SizePersistedDialog.__init__(self, parent_tab, 'import-list-advanced-match-dialog')
        self.setWindowTitle(_('Advanced Match Dialog'))
        self.setup_ui()
        self.resize_dialog()

    def setup_ui(self):
        self.setWindowTitle(_('Add tiers'))
        l = QVBoxLayout()
        self.setLayout(l)

        settings_l = QGridLayout()
        l.addLayout(settings_l)

        tb = QTextBrowser()
        tb.setHtml(help_text)
        l.addWidget(tb)

        table_groupbox = QGroupBox(_('Tiers'))
        table_layout = QHBoxLayout()
        table_groupbox.setLayout(table_layout)
        l.addWidget(table_groupbox)
        
        self._table = AdvancedMatchTable(self, self.selected_cols)
        table_layout.addWidget(self._table)
        
        advanced_match_model = AdvancedMatchModel(self.gui, self.advanced_match_config, self.selected_cols)
        advanced_match_model.validate()
        self._table.set_model(advanced_match_model)
        self._table.selectionModel().selectionChanged.connect(self._on_table_selection_change)
        
        # Restore table state
        state = gprefs.get(self.KEY_TABLE_STATE)
        if state:
            self._table.apply_state(state)

        # Add a vertical layout containing the the buttons to move up/down etc.
        button_layout = QVBoxLayout()
        table_layout.addLayout(button_layout)
        
        move_up_button = self.move_up_button = QToolButton(self)
        move_up_button.setToolTip(_('Move row up'))
        move_up_button.setIcon(QIcon(I('arrow-up.png')))
        button_layout.addWidget(move_up_button)
        spacerItem1 = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem1)

        add_button = self.add_button = QToolButton(self)
        add_button.setToolTip(_('Add row'))
        add_button.setIcon(QIcon(I('plus.png')))
        button_layout.addWidget(add_button)
        spacerItem2 = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem2)

        copy_button = self.copy_button = QToolButton(self)
        copy_button.setToolTip(_('Duplicate tier'))
        copy_button.setIcon(QIcon(I('edit-copy.png')))
        button_layout.addWidget(copy_button)
        spacerItem3 = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem3)

        delete_button = self.delete_button = QToolButton(self)
        delete_button.setToolTip(_('Delete row'))
        delete_button.setIcon(QIcon(I('minus.png')))
        button_layout.addWidget(delete_button)
        spacerItem4 = QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding)
        button_layout.addItem(spacerItem4)

        move_down_button = self.move_down_button = QToolButton(self)
        move_down_button.setToolTip(_('Move row down'))
        move_down_button.setIcon(QIcon(I('arrow-down.png')))
        button_layout.addWidget(move_down_button)

        move_up_button.clicked.connect(partial(self._table.move_rows,UP))
        move_down_button.clicked.connect(partial(self._table.move_rows,DOWN))
        add_button.clicked.connect(self._table.add_row)
        delete_button.clicked.connect(self._table.delete_rows)
        copy_button.clicked.connect(self._table.copy_row)
        
        self.button_box = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        self.button_box.accepted.connect(self.accept)
        self.button_box.rejected.connect(self.reject)
        l.addWidget(self.button_box)
        
        self._on_table_selection_change()

        self.setMinimumSize(600,300)

    def _on_table_selection_change(self):
        sm = self._table.selectionModel()
        selection_count = len(sm.selectedRows())
        self.delete_button.setEnabled(selection_count > 0)
        self.copy_button.setEnabled(selection_count == 1)
        self.move_up_button.setEnabled(selection_count > 0)
        self.move_down_button.setEnabled(selection_count > 0)

    def save_table_state(self):
        # Save table state
        gprefs[self.KEY_TABLE_STATE] = self._table.get_state()

    def reject(self):
        self.save_table_state()
        SizePersistedDialog.reject(self)

    def accept(self):
        self.save_table_state()
        
        if self._table.model().rowCount(QModelIndex()) < 1:
            return error_dialog(
                self,
                _('No match tiers selected'),
                _('You must select at least one match tier.'),
                show=True
            )
        is_valid = self._table.model().validate()
        if is_valid is not True:
            msg, details = _('Invalid match tiers'), _('Some match tiers are not valid. Double click on error cell for more details.')
            return error_dialog(
                self,
                msg,
                details,
                show=True
            )

        advanced_match_config = self._table.model().advanced_match_config
        # Remove error keys from advanced_match_config
        for tier_config in advanced_match_config:
            try:
                del tier_config['errors']
            except:
                pass
        self.advanced_match_config = advanced_match_config
        SizePersistedDialog.accept(self)
