#!/usr/bin/env python
# ~*~ coding: utf-8 ~*~

__license__   = 'GPL v3'
__copyright__ = '2021, Ahmed Zaki <azaki00.dev@gmail.com>'
__docformat__ = 'restructuredtext en'

from functools import partial
from operator import itemgetter

from qt.core import QApplication, Qt, QToolButton, QMenu, QAction, QUrl

from calibre.gui2 import open_url
from calibre.gui2.dialogs.confirm_delete import confirm
from calibre.gui2.actions import InterfaceAction
from calibre.gui2.tag_browser.model import TagTreeItem

from calibre_plugins.category_tags.common_utils import (set_plugin_icon_resources, get_icon, create_menu_action_unique)
from calibre_plugins.category_tags.user_categories.wizards import ImportCategoriesWizard
from calibre_plugins.category_tags.user_categories.dialog_export import ExportCategoriesDialog
from calibre_plugins.category_tags.advanced_matching.matching import get_all_algorithms, MetadataMatch
from calibre_plugins.category_tags.user_categories.actions import ImportAction, ExportAction
from calibre_plugins.category_tags.user_categories.user_cat_editor import UserCatEditor

try:
    load_translations()
except NameError:
    debug_print("CategoryTags::action.py - exception when loading translations")
    pass

PLUGIN_ICONS = ['images/category_tags.png']

class CategoryTags(InterfaceAction):

    name = 'Category Tags'
    # Create our top-level menu/toolbar action (text, icon_path, tooltip, keyboard shortcut)
    action_spec = ('Category Tags', None, 'Tag items using user categories. Also import/export user categories.', None)
    popup_type = QToolButton.InstantPopup
    action_add_menu = True
    action_type = 'current'

    current_instance = None
    reading_list_plugin = None

    def genesis(self):
        
        # Read the plugin icons and store for potential sharing with the config widget
        icon_resources = self.load_resources(PLUGIN_ICONS)
        set_plugin_icon_resources(self.name, icon_resources)

        # Assign our menu to this action and an icon
        self.qaction.setIcon(get_icon(PLUGIN_ICONS[0]))

    def update_algorithms(self):
        self.algorithms, self.builtin_algorithms, self.user_algorithms = get_all_algorithms(self.gui, self.user_algorithm_classes)

    def initialization_complete(self):
        # Delay instantiating our finder as we require access to the library view
        self.import_action = ImportAction(self.gui)
        self.export_action = ExportAction(self.gui)
        self.actions = {
            'Import User Catetories': self.import_action,
            'Export User Catetories': self.export_action
        }
        self.import_action.on_modules_update(None)
        self.export_action.on_modules_update(None)
        self.user_algorithm_classes = {}
        self.update_algorithms()
        self.build_menus()

    def library_changed(self, db):
        if self.current_instance and not self.current_instance.is_closed:
            self.current_instance.close()
            self.current_instance = None
        self.update_algorithms()

    def build_menus(self):
        m = self.menu = QMenu(self.gui)
        m.clear()

        create_menu_action_unique(self, m, _('Export User Categories')+'...', 'arrow-up.png',
                                  triggered=self.export_user_categories)

        create_menu_action_unique(self, m, _('Import User Categories')+'...', 'arrow-down.png',
                                  triggered=self.import_user_categories)

        m.addSeparator()

        create_menu_action_unique(self, m, _('Delete User Categories')+'...', 'minus.png',
                                  triggered=self.delete_user_categories)

        m.addSeparator()

        create_menu_action_unique(self, m, _('&Customize plugin')+'...', 'config.png',
                                  triggered=self.show_configuration)

        m.addSeparator()

        self.advanced_help_action = create_menu_action_unique(self, m, _('&Help'), 'help.png',
                                  triggered=self.show_help)

        self.gui.keyboard.finalize()

        # Assign our menu to this action and an icon, also add dropdown menu
        self.qaction.setMenu(self.menu)

    def show_configuration(self):
        self.interface_action_base_plugin.do_user_config(self.gui)

    def show_help(self):
        url = 'https://www.mobileread.com/forums/showpost.php?p=4142902&postcount=1'
        open_url(QUrl(url))

    def import_user_categories(self):
        self.import_wizard = ImportCategoriesWizard(self)
        self.import_wizard.show()

    def export_user_categories(self):
        d = ExportCategoriesDialog('category-tags-export-categories-dialog', self.gui, self.export_action)
        if d.exec_() == d.Accepted:
            settings = d.settings
            self.export_action.run(self.gui, settings)

    def show_user_cat_editor(self, item, item_type):
        db = self.gui.library_view.model().db
        d = UserCatEditor(self.gui, item, item_type)
        if d.exec_() == d.Accepted:
            QApplication.setOverrideCursor(Qt.WaitCursor)
            try:
                # Delete unapplied tags
                if d.remove_tags:
                    for user_cat in d.remove_tags:
                        self.gui.tags_view.model().delete_item_from_user_category(user_cat,
                                                              item, item_type)

                    self.gui.tags_view.recount()
                    db.new_api.clear_search_caches()
                    self.gui.library_view.model().refresh()
                
                # Add new tags
                new_tags = d.new_tags
                if new_tags:
                    updated_categories = dict.copy(db.prefs.get('user_categories', {}))
                    for user_cat in new_tags:
                        if not updated_categories.get(user_cat):
                            updated_categories[user_cat] = []

                    for user_cat in new_tags:
                        updated_categories[user_cat].append([ item, item_type, 0])
                        # Re-sort the collection
                        l = []
                        for n in sorted(updated_categories[user_cat], key=itemgetter(0)):
                            l.append(n)
                        updated_categories[user_cat] = l

                    # New Top level nodes must be added to calibre before updating with the new categories
                    new_top_level_nodes = set([node.partition('.')[0] for node in new_tags])
                    user_categories = dict.copy(db.prefs.get('user_categories', {}))
                    for node in new_top_level_nodes:
                        if not user_categories.get(node):
                            user_categories[node] = []
                    self.refresh_user_categories(user_categories)

                    final_categories = dict.copy(db.prefs.get('user_categories', {}))
                    final_categories.update(updated_categories)
                    self.refresh_user_categories(final_categories)
            finally:
                QApplication.restoreOverrideCursor()

    def refresh_user_categories(self, user_categories):
        db = self.gui.library_view.model().db
        
        # Order is important. The categories must be removed before setting
        # the preference because setting the pref recomputes the dynamic categories
        db.field_metadata.remove_user_categories()
        db.new_api.set_pref('user_categories', user_categories)
        db.new_api.refresh_search_locations()
        self.gui.tags_view.recount()
        db.new_api.clear_search_caches()
        self.gui.library_view.model().refresh()

    def delete_user_categories(self):
        if not confirm(
            _('All user categories will be deleted. Are you sure you want to proceed?'),
            'category_tags_delete_all_categories'):
            return
        self.refresh_user_categories({})

    def tag_browser_context_action(self, index):
        context_actions = []
        #named_path = self.gui.tags_view.model().named_path_for_index(index)
        item = index.data(Qt.ItemDataRole.UserRole)
        if item.type == TagTreeItem.TAG:
            item_type = item.tag.category
            item_text = item.tag.name
            ac = QAction(get_icon(PLUGIN_ICONS[0]), _('Add/Remove User Categories ...'), self)
            ac.triggered.connect(partial(self.show_user_cat_editor, item_text, item_type))
            context_actions.append(ac)
        return context_actions
