import sys
from PySide6.QtCore import QRegularExpression
from PySide6.QtGui import QSyntaxHighlighter
from PySide6 import QtCore, QtGui, QtWidgets

def format(color, style=''):
    """Return a QTextCharFormat with the given attributes.
    """
    _color = QtGui.QColor.fromString(color)

    _format = QtGui.QTextCharFormat()
    _format.setForeground(_color)
    if 'bold' in style:
        _format.setFontWeight(QtGui.QFont.Bold)
    if 'italic' in style:
        _format.setFontItalic(True)

    return _format


# Syntax styles that can be shared by all languages
STYLES = {}

# Dark Theme Syntax styles that can be shared by all languages
STYLES_DARK = {
    'keyword': format('lightBlue', 'bold'),
    'operator': format('magenta'),
    'brace': format('orangered'),
    'defclass': format('lightGreen', 'bold'),
    'string': format('lightCyan'),
    'string2': format('lightCyan'),
    'comment': format('lightYellow', 'italic'),
    'self': format('blue', 'italic'),
    'numbers': format('lightBrown'),
}

# Light Theme Syntax styles that can be shared by all languages
STYLES_LIGHT = {
    'keyword': format('blue', 'bold'),
    'operator': format('darkMagenta'),
    'brace': format('darkRed'),
    'defclass': format('green', 'bold'),
    'string': format('darkCyan'),
    'string2': format('darkCyan'),
    'comment': format('darkYellow', 'italic'),
    'self': format('darkBlue', 'italic'),
    'numbers': format('brown'),
}

STYLES_ORIG = {
    'keyword': format('navy', 'bold'),
    'operator': format('red'),
    'brace': format('blue'),
    'defclass': format('green', 'bold'),
    'string': format('magenta'),
    'string2': format('darkMagenta'),
    'comment': format('darkGreen', 'italic'),
    'self': format('black', 'italic'),
    'numbers': format('brown'),
}


class PythonSyntaxHighlighter(QSyntaxHighlighter):
    """Syntax highlighter for the Python language.
    """
    # Python keywords
    keywords = [
        'and', 'assert', 'break', 'class', 'continue', 'def',
        'del', 'elif', 'else', 'except', 'exec', 'finally',
        'for', 'from', 'global', 'if', 'import', 'in',
        'is', 'lambda', 'not', 'or', 'pass', 'print',
        'raise', 'return', 'try', 'while', 'yield',
        'None', 'True', 'False', 'as',
    ]

    # Python operators
    operators = [
        '=',
        # Comparison
        '==', '!=', '<', '<=', '>', '>=',
        # Arithmetic
        '\+', '-', '\*', '/', '//', '\%', '\*\*',
        # In-place
        '\+=', '-=', '\*=', '/=', '\%=',
        # Bitwise
        '\^', '\|', '\&', '\~', '>>', '<<',
    ]

    # Python braces
    braces = [
        '\{', '\}', '\(', '\)', '\[', '\]',
    ]

    def __init__(self, parent: QtGui.QTextDocument) -> None:
        super().__init__(parent)
        self.isDark = qApp.palette().color(QtGui.QPalette.Active,QtGui.QPalette.WindowText).lightness() > 128;
        self.setUpStyleRules()


    def setUpStyleRules(self):
        if self.isDark:
            STYLES= STYLES_DARK
        else:
            STYLES= STYLES_LIGHT

        # Multi-line strings (expression, flag, style)
        self.tri_single = (QtCore.QRegularExpression("'''"), 1, STYLES['string2'])
        self.tri_double = (QtCore.QRegularExpression('"""'), 2, STYLES['string2'])

        rules = []

        # Keyword, operator, and brace rules
        rules += [(r'\b%s\b' % w, 0, STYLES['keyword'])
            for w in PythonSyntaxHighlighter.keywords]
        rules += [(r'%s' % o, 0, STYLES['operator'])
            for o in PythonSyntaxHighlighter.operators]
        rules += [(r'%s' % b, 0, STYLES['brace'])
            for b in PythonSyntaxHighlighter.braces]

        # All other rules
        rules += [
            # 'self'
            (r'\bself\b', 0, STYLES['self']),

            # 'def' followed by an identifier
            (r'\bdef\b\s*(\w+)', 1, STYLES['defclass']),
            # 'class' followed by an identifier
            (r'\bclass\b\s*(\w+)', 1, STYLES['defclass']),

            # Numeric literals
            (r'\b[+-]?[0-9]+[lL]?\b', 0, STYLES['numbers']),
            (r'\b[+-]?0[xX][0-9A-Fa-f]+[lL]?\b', 0, STYLES['numbers']),
            (r'\b[+-]?[0-9]+(?:\.[0-9]+)?(?:[eE][+-]?[0-9]+)?\b', 0, STYLES['numbers']),

            # Double-quoted string, possibly containing escape sequences
            (r'"[^"\\]*(\\.[^"\\]*)*"', 0, STYLES['string']),
            # Single-quoted string, possibly containing escape sequences
            (r"'[^'\\]*(\\.[^'\\]*)*'", 0, STYLES['string']),

            # From '#' until a newline
            (r'#[^\n]*', 0, STYLES['comment']),
        ]

        # Build a QRegularExpression for each pattern
        self.rules = [(QtCore.QRegularExpression(pat), index, fmt)
            for (pat, index, fmt) in rules]

        
    def highlightBlock(self, text):
        """Apply syntax highlighting to the given block of text.
        """
        # Do other syntax formatting
        for expression, nth, aformat in self.rules:
            match = expression.match(text)
            index = match.capturedStart()

            while index >= 0:
                # We actually want the index of the nth match
                length = match.capturedLength()
                self.setFormat(index, length, aformat)
                match = expression.match(text, index + length)
                index = match.capturedStart()

        self.setCurrentBlockState(0)
                
        # Do multi-line strings
        in_multiline = self.match_multiline(text, *self.tri_single)
        if not in_multiline:
            in_multiline = self.match_multiline(text, *self.tri_double)

    def do_rehighlight(self):
        isDark = qApp.palette().color(QtGui.QPalette.Active,QtGui.QPalette.WindowText).lightness() > 128;
        if isDark != self.isDark:
            # theme has changed
            self.isDark = isDark
            self.setUpStyleRules()
        self.rehighlight()

    def match_multiline(self, text, delimiter, in_state, style):
        """Do highlighting of multi-line strings. ``delimiter`` should be a
        ``QRegularExpression`` for triple-single-quotes or triple-double-quotes, and
        ``in_state`` should be a unique integer to represent the corresponding
        state changes when inside those strings. Returns True if we're still
        inside a multi-line string when this function is finished.
        """

        # If inside triple-single quotes, start at 0
        if self.previousBlockState() == in_state:
            start = 0
            add = 0
        # Otherwise, look for the delimiter on this line
        else:
            match = delimiter.match(text)
            start = match.capturedStart()
            add = match.capturedLength()

        # As long as there's a delimiter match on this line...
        while start >= 0:
            # Look for the ending delimiter
            match = delimiter.match(text, start + add)
            end = match.capturedStart()
            # Ending delimiter on this line?
            if end >= add:
                length = end - start + add + match.capturedLength()
                self.setCurrentBlockState(0)
            # No; multi-line string
            else:
                self.setCurrentBlockState(in_state)
                length = len(text) - start + add
            # Apply formatting
            self.setFormat(start, length, style)
            # Look for the next match
            match = delimiter.match(text, start + length)
            start = match.capturedStart()

        # Return True if still inside a multi-line string, False otherwise
        if self.currentBlockState() == in_state:
            return True
        else:
            return False
