import io
import subprocess
import os

from PIL import Image
from calibre.ebooks.metadata.book.base import Metadata
from calibre.utils.logging import default_log as log
from calibre_plugins.audiobook_metadata.tinytag import TinyTag

# IMPORTANT: Use Calibre's datetime with UTC to avoid "Bad binding argument type"
from calibre.utils.date import now, datetime as cdatetime, utc_tz


def get_duration_ffprobe(filepath):
    try:
        cmd = [
            'ffprobe', '-v', 'quiet',
            '-show_entries', 'format=duration',
            '-of', 'default=noprint_wrappers=1:nokey=1',
            filepath
        ]
        result = subprocess.run(cmd, capture_output=True, text=True)
        if result.stdout.strip():
            return float(result.stdout)
    except Exception as e:
        log.debug(f"FFprobe error: {e}")
    return None


def get_metadata(path) -> Metadata:
    """
    Read metadata from an audio file at `path` using TinyTag, then fallback to ffprobe
    if TinyTag has no duration. Return a calibre Metadata object. 
    Avoid passing any None fields into calibre (e.g. in title, authors).
    """
    log.debug(f"Processing file: {path}")
    tag = TinyTag.get(filename=path, image=True, duration=True)

    # 1) Title with fallback
    raw_title = tag.album or tag.title
    if raw_title:
        raw_title = raw_title.strip()
        if raw_title.endswith(" (Unabridged)"):
            raw_title = raw_title[:-12]  # remove exactly " (Unabridged)"
    title = raw_title or "Unknown"

    # 2) Authors, removing None
    raw_authors = [tag.albumartist, tag.artist, tag.composer]
    authors = [a for a in raw_authors if a]
    if not authors:
        authors = ["Unknown"]

    meta = Metadata(title, authors)

    # 3) Year => pubdate as a properly-constructed UTC datetime
    if tag.year is not None:
        try:
            y = int(tag.year)
            # If it looks like a valid year, store as 1/1/year UTC
            if 1 <= y <= 9999:
                meta.pubdate = cdatetime(y, 1, 1, tzinfo=utc_tz)
        except ValueError:
            pass

    # 4) Cover image
    image_bytes = tag.get_image()
    if image_bytes is not None:
        image = Image.open(io.BytesIO(image_bytes))
        if image.format:
            format_type = image.format.lower()
            meta.cover_data = (format_type, image_bytes)

    # 5) Copyright
    if tag.extra and "copyright" in tag.extra:
        meta.rights = tag.extra["copyright"]

    # 6) Genre
    if tag.genre:
        meta.tags = tuple(tag.genre.split(", "))

    # 7) Comments & Performer => fallback to ""
    meta.comments = tag.comment or ""
    meta.performer = tag.composer or ""

    # 8) Duration => from TinyTag, else ffprobe
    duration_seconds = None
    if hasattr(tag, 'duration') and tag.duration is not None:
        duration_seconds = tag.duration
        log.debug(f"Got duration from TinyTag: {duration_seconds}")
    else:
        duration_seconds = get_duration_ffprobe(path)
        log.debug(f"Got duration from FFprobe: {duration_seconds}")

    if duration_seconds is not None:
        total_seconds = int(duration_seconds)
        hours = total_seconds // 3600
        minutes = (total_seconds % 3600) // 60
        seconds = total_seconds % 60
        duration_str = f"{hours:02d}:{minutes:02d}:{seconds:02d}"
        log.debug(f"Formatted duration: {duration_str}")

        meta.set_user_metadata('#duration', {
            '#value#': duration_str,
            '#extra#': None,
            'datatype': 'text',
            'is_multiple': None,
            'name': 'Duration'
        })
        log.debug("Successfully set duration metadata")

    # Return the finished, safe-to-use metadata object
    return meta