from __future__ import unicode_literals, division, absolute_import, print_function

__license__   = 'GPL v3'
__copyright__ = '2024'

from functools import partial
try:
    from qt.core import (QMenu, QToolButton, QPixmap, Qt, QIcon,
                        QProgressDialog, QTimer)
except ImportError:
    from PyQt5.Qt import (QMenu, QToolButton, QPixmap, Qt, QIcon,
                         QProgressDialog, QTimer)

from calibre.gui2 import error_dialog, info_dialog
from calibre.gui2.actions import InterfaceAction
from calibre_plugins.audiobook_metadata.audio_metadata import get_metadata
from PyQt5.Qt import (QDialog, QVBoxLayout, QHBoxLayout, QCheckBox, 
                     QPushButton, QLabel)
import os
from collections import defaultdict

class FetchMetaDialog(QDialog):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle('Select metadata fields to update')
        layout = QVBoxLayout()
        self.setLayout(layout)
        
        # Metadata fields selection
        layout.addWidget(QLabel('Select metadata fields to update:'))
        
        self.checkboxes = {}
        fields = {
            'title': 'Title',
            'authors': 'Authors',
            'duration': 'Duration',
            'year': 'Year',
            'genre': 'Genre',
            'comments': 'Comments',
            'cover': 'Cover Image'
        }
        
        for field_id, field_name in fields.items():
            cb = QCheckBox(field_name)
            self.checkboxes[field_id] = cb
            layout.addWidget(cb)
        
        # Buttons
        button_layout = QHBoxLayout()
        select_all = QPushButton('Select All')
        select_all.clicked.connect(self.select_all_clicked)
        button_layout.addWidget(select_all)
        
        select_none = QPushButton('Select None')
        select_none.clicked.connect(self.select_none_clicked)
        button_layout.addWidget(select_none)
        
        layout.addLayout(button_layout)
        
        buttons = QHBoxLayout()
        ok_button = QPushButton('OK')
        ok_button.clicked.connect(self.accept)
        cancel_button = QPushButton('Cancel')
        cancel_button.clicked.connect(self.reject)
        buttons.addWidget(ok_button)
        buttons.addWidget(cancel_button)
        layout.addLayout(buttons)
    
    def select_all_clicked(self):
        for cb in self.checkboxes.values():
            cb.setChecked(True)
    
    def select_none_clicked(self):
        for cb in self.checkboxes.values():
            cb.setChecked(False)
    
    def get_selected_fields(self):
        return [k for k, v in self.checkboxes.items() if v.isChecked()]

class FetchAudioMetaAction(InterfaceAction):
    name = 'Fetch Audio Meta'
    action_spec = ('Fetch Audio Meta', None, None, None)
    
    def genesis(self):
        icon = get_icons('images/icon.png')
        self.qaction.setIcon(icon)
        self.qaction.triggered.connect(self.fetch_metadata)
    
    def fetch_metadata(self):
        book_ids = self.gui.library_view.get_selected_ids()
        if not book_ids:
            return error_dialog(self.gui, 'No books selected', 
                              'Please select some books first', show=True)
        
        d = FetchMetaDialog(self.gui)
        if d.exec_() != QDialog.Accepted:
            return
        
        selected_fields = d.get_selected_fields()
        if not selected_fields:
            return error_dialog(self.gui, 'No fields selected', 
                              'Please select at least one field to update', show=True)
        
        self.process_books(book_ids, selected_fields)
    
    def process_books(self, book_ids, selected_fields):
        db = self.gui.current_db.new_api
        total_books = len(book_ids)
        processed = 0
        errors = defaultdict(list)
        
        # Create progress dialog with percentage
        progress = QProgressDialog(self.gui)
        progress.setWindowTitle('Updating Metadata')
        progress.setWindowModality(Qt.WindowModal)
        progress.setMinimum(0)
        progress.setMaximum(total_books)
        progress.setCancelButton(None)  # Remove cancel button
        progress.setMinimumWidth(400)  # Make dialog wider
        
        def process_next_book():
            nonlocal processed
            
            if processed >= total_books:
                progress.close()
                self.show_results(errors)
                self.gui.library_view.model().refresh_ids(book_ids)
                return
            
            book_id = book_ids[processed]
            # Update progress text
            percent = int((processed / total_books) * 100)
            progress.setLabelText(
                f'Processing audiobook {processed + 1} of {total_books}\n'
                f'Progress: {percent}%'
            )
            progress.setValue(processed)
            
            try:
                formats = db.formats(book_id)
                if not formats:
                    errors['No formats'].append(self.get_book_info(db, book_id))
                    processed += 1
                    QTimer.singleShot(100, process_next_book)
                    return
                
                audio_formats = [f for f in formats if f.lower() in ['m4b', 'm4a', 'mp3', 'ogg', 'opus', 'flac', 'wma', 'mp4', 'aiff']]
                if not audio_formats:
                    errors['No audio formats'].append(self.get_book_info(db, book_id))
                    processed += 1
                    QTimer.singleShot(100, process_next_book)
                    return
                
                path = db.format_abspath(book_id, audio_formats[0])
                if not path or not os.path.exists(path):
                    errors['File not found'].append(self.get_book_info(db, book_id))
                    processed += 1
                    QTimer.singleShot(100, process_next_book)
                    return
                
                try:
                    meta = get_metadata(path)
                    if not meta:
                        errors['Metadata read failed'].append(
                            f"{self.get_book_info(db, book_id)} - The file may be damaged or missing metadata tags"
                        )
                        processed += 1
                        QTimer.singleShot(100, process_next_book)
                        return
                except Exception as e:
                    errors['Metadata read failed'].append(
                        f"{self.get_book_info(db, book_id)} - Could not process file (Damaged or unsupported format)"
                    )
                    processed += 1
                    QTimer.singleShot(100, process_next_book)
                    return
                
                # Track which fields couldn't be read
                missing_fields = []
                
                # Handle duration first if selected
                if 'duration' in selected_fields:
                    duration_metadata = meta.get_user_metadata('#duration', None)
                    if duration_metadata and duration_metadata.get('#value#'):
                        db.set_field('#duration', {book_id: duration_metadata.get('#value#')})
                    else:
                        missing_fields.append('duration')

                # Get current metadata to update only selected fields
                mi = db.get_metadata(book_id)
                
                # Check and update each selected field
                if 'title' in selected_fields:
                    if meta.title:
                        mi.title = meta.title
                    else:
                        missing_fields.append('title')
                        
                if 'authors' in selected_fields:
                    if meta.authors:
                        mi.authors = meta.authors
                    else:
                        missing_fields.append('authors')
                        
                if 'year' in selected_fields:
                    if meta.pubdate:
                        mi.pubdate = meta.pubdate
                    else:
                        missing_fields.append('year')
                        
                if 'genre' in selected_fields:
                    if meta.tags:
                        mi.tags = meta.tags
                    else:
                        missing_fields.append('genre')
                        
                if 'comments' in selected_fields:
                    if meta.comments:
                        mi.comments = meta.comments
                    else:
                        missing_fields.append('comments')
                        
                if 'cover' in selected_fields:
                    if meta.cover_data:
                        mi.cover_data = meta.cover_data
                    else:
                        missing_fields.append('cover')
                
                # Update metadata and report missing fields
                db.set_metadata(book_id, mi)
                if missing_fields:
                    errors['Missing metadata'].append(
                        f"{self.get_book_info(db, book_id)} - Could not read selected metadata ({', '.join(missing_fields)}) - The file may be missing these tags"
                    )
                
            except Exception as e:
                errors['Processing error'].append(
                    f"{self.get_book_info(db, book_id)}: {str(e)}"
                )
            
            processed += 1
            QTimer.singleShot(100, process_next_book)
        
        # Start processing
        process_next_book()
        progress.exec_()
    
    def get_book_info(self, db, book_id):
        """Get book title and authors for error reporting"""
        mi = db.get_metadata(book_id)
        return f"{mi.title} by {' & '.join(mi.authors)}"
    
    def show_results(self, errors):
        """Show summary of processing results and any errors"""
        if not errors:
            info_dialog(self.gui, 'Success',
                       'Updated metadata for all selected books successfully',
                       show=True)
            return
        
        error_msg = ['The following errors occurred:']
        for error_type, books in errors.items():
            error_msg.append(f'\n{error_type}:')
            for book in books:
                error_msg.append(f'  • {book}')
        
        error_dialog(self.gui, 'Processing Complete with Errors',
                    '\n'.join(error_msg),
                    show=True)