from datetime import date
import io
import subprocess
import os
from collections import defaultdict

from calibre.customize import MetadataReaderPlugin
from calibre.ebooks.metadata.book.base import Metadata
from calibre.utils.logging import default_log as log
from PIL import Image
from calibre_plugins.audiobook_metadata_import.tinytag_import.tinytag import TinyTag


class AudioBookPluginImport(MetadataReaderPlugin):
    file_types = {"m4b", "m4a", "mp3", "ogg", "opus", "flac", "wma", "mp4", "aiff"}
    author = "Artur Kupiec"

    name = "Import Audiobooks Metadata"
    description = "Import metadata from various audio formats including m4b, m4a, mp3, ogg, opus, flac, wma, mp4, and aiff"
    version = (0, 2, 7)
    minimum_calibre_version = (7, 0, 0)
    can_be_disabled = False

    def get_metadata(self, stream, type) -> Metadata:
        """Extract metadata from audio files using TinyTag with comprehensive error handling."""
        errors = defaultdict(list)
        filepath = stream.name
        
        try:
            # Basic file validation
            if not os.path.exists(filepath):
                log.error(f"File not found: {filepath}")
                return self.create_empty_metadata(f"File not found: {filepath}")

            # Attempt to read tags
            try:
                tag = TinyTag.get(filename=filepath, file_obj=stream, image=True, duration=True)
            except Exception as e:
                log.error(f"Failed to read tags from {filepath}: {str(e)}")
                return self.create_empty_metadata(f"Could not read audio file (possibly damaged or unsupported format): {str(e)}")

            # Validate metadata completeness
            missing_fields = self.validate_metadata(tag, filepath)
            if missing_fields:
                log.warning(f"Missing metadata fields in {filepath}: {', '.join(missing_fields)}")
                # Continue processing but log the warning

            # Process metadata
            title = get_title_from_tag(tag)
            if not title:
                log.warning(f"No title found in {filepath}")
                title = os.path.splitext(os.path.basename(filepath))[0]

            authors = get_author_list(tag)
            meta = Metadata(title, authors)

            # Publication date
            if tag.year is not None:
                try:
                    meta.pubdate = date(int(tag.year), 1, 1)
                except ValueError as e:
                    log.warning(f"Invalid year value in {filepath}: {tag.year}")

            # Cover image
            image_bytes = tag.get_image()
            if image_bytes is not None:
                try:
                    image = Image.open(io.BytesIO(image_bytes))
                    if image.format is not None:
                        format_type = image.format.lower()
                        meta.cover_data = (format_type, image_bytes)
                except Exception as e:
                    log.warning(f"Failed to process cover image in {filepath}: {str(e)}")

            # Additional metadata
            if tag.extra is not None:
                if "copyright" in tag.extra:
                    meta.rights = tag.extra["copyright"]
                # Process additional custom fields
                for key, value in tag.extra.items():
                    if value and key not in ['copyright']:
                        try:
                            meta.set_user_metadata(f'#{key.lower()}', {
                                '#value#': value,
                                '#extra#': None,
                                'datatype': 'text',
                                'is_multiple': None,
                                'name': key.title()
                            })
                        except Exception as e:
                            log.warning(f"Failed to set custom metadata {key} in {filepath}: {str(e)}")

            # Genre
            if tag.genre is not None:
                meta.tags = tuple(tag.genre.split(", "))

            # Comments and performer
            meta.comments = tag.comment
            meta.performer = tag.composer

            # Duration
            self.add_duration_metadata(meta, tag, filepath)

            # Log missing fields in comments if any
            if missing_fields:
                warning_msg = f"Warning: Missing metadata fields: {', '.join(missing_fields)}"
                if meta.comments:
                    meta.comments = f"{meta.comments}\n\n{warning_msg}"
                else:
                    meta.comments = warning_msg

            return meta

        except Exception as e:
            log.error(f"Unexpected error processing {filepath}: {str(e)}")
            return self.create_empty_metadata(f"Unexpected error: {str(e)}")

    def get_duration_ffprobe(self, filepath):
        """Get audio duration using ffprobe as a fallback method."""
        try:
            cmd = ['ffprobe', '-v', 'quiet', '-show_entries', 'format=duration', 
                  '-of', 'default=noprint_wrappers=1:nokey=1', filepath]
            result = subprocess.run(cmd, capture_output=True, text=True)
            if result.stdout.strip():
                return float(result.stdout)
        except Exception as e:
            log.debug(f"FFprobe error: {e}")
            return None

    def validate_metadata(self, tag, filepath):
        """Validate the metadata and return a list of missing or invalid fields."""
        missing_fields = []
        
        # Check essential metadata fields
        if not (tag.title or tag.album):
            missing_fields.append('title')
        if not any([tag.albumartist, tag.artist, tag.composer]):
            missing_fields.append('artist')
        if not tag.duration and not self.get_duration_ffprobe(filepath):
            missing_fields.append('duration')
            
        return missing_fields

    def create_empty_metadata(self, error_message):
        """Create empty metadata with error message in comments."""
        meta = Metadata(None, [])
        meta.comments = f"Error reading metadata: {error_message}"
        return meta

    def add_duration_metadata(self, meta, tag, filepath):
        """Add duration metadata to the book with error handling."""
        duration_seconds = None
        
        # Try TinyTag duration first
        if hasattr(tag, 'duration') and tag.duration is not None:
            duration_seconds = tag.duration
            log.debug(f"Got duration from TinyTag: {duration_seconds}")
        else:
            # Fallback to FFprobe
            duration_seconds = self.get_duration_ffprobe(filepath)
            log.debug(f"Got duration from FFprobe: {duration_seconds}")

        if duration_seconds is not None:
            try:
                total_seconds = int(duration_seconds)
                hours = total_seconds // 3600
                minutes = (total_seconds % 3600) // 60
                seconds = total_seconds % 60
                duration_str = f"{hours:02d}:{minutes:02d}:{seconds:02d}"
                log.debug(f"Formatted duration: {duration_str}")

                meta.set_user_metadata('#duration', {
                    '#value#': duration_str,
                    '#extra#': None,
                    'datatype': 'text',
                    'is_multiple': None,
                    'name': 'Duration'
                })
                log.debug("Successfully set duration metadata")
            except Exception as e:
                log.error(f"Error setting duration metadata: {e}")
                # Fallback to comments
                prefix = "Duration: "
                if meta.comments:
                    if prefix not in meta.comments:
                        meta.comments = f"{meta.comments}\n{prefix}{duration_str}"
                else:
                    meta.comments = f"{prefix}{duration_str}"
        else:
            log.warning(f"Could not determine duration for {filepath}")


def get_author_list(tag):
    """Extract and process author information from the tag."""
    authors = [tag.albumartist, tag.artist, tag.composer]
    # Remove duplicates while preserving order
    seen = set()
    unique_authors = []
    for author in authors:
        if author and author not in seen:
            unique_authors.append(author)
            seen.add(author)
    return unique_authors if unique_authors else ["Unknown"]


def get_title_from_tag(tag):
    """Extract and clean up the title from the tag."""
    title = tag.album or tag.title
    if title is None:
        return None
        
    title = title.strip()
    # Remove common audiobook suffixes
    suffixes = [
        " (Unabridged)",
        " [Unabridged]",
        " (Audiobook)",
        " [Audiobook]",
        " (Audio Book)",
        " [Audio Book]"
    ]
    for suffix in suffixes:
        if title.endswith(suffix):
            return title[:-len(suffix)]
    return title