__license__   = 'GPL v3'
__copyright__ = '2024, Anareaty <reatymain@gmail.com>'
__docformat__ = 'restructuredtext en'


from calibre.gui2.actions import InterfaceAction
from calibre_plugins.pocketbook_collections.__init__ import PLUGIN_NAME
from calibre_plugins.pocketbook_collections.config import prefs
from calibre.devices.usbms.driver import debug_print
from calibre.gui2 import error_dialog
import os

try:
    from qt.core import QMenu
except:
    try:
        from PyQt5.Qt import QMenu
    except:
        from PyQt4.Qt import QMenu



PLUGIN_ICONS = ['images/pocketbook.png', 
                'images/forward.png', 
                'images/back.png', 
                'images/config.png', 
                'images/catalog.png', 
                'images/ok.png', 
                'images/donate.png',
                'images/auto-reload.png'
                ]




class InterfacePlugin(InterfaceAction):
    name = PLUGIN_NAME
    action_spec = (PLUGIN_NAME, None, PLUGIN_NAME, None)
    action_type = 'current'
    dont_add_to = frozenset(['toolbar'])


    def genesis(self):
        main_menu = QMenu(self.gui)
        self.menu = main_menu
        self.qaction.setMenu(main_menu)


    def initialization_complete(self):
        self.rebuild_menus()

        
    # Set menus
        
    def rebuild_menus(self):
        fm = self.gui.current_db.field_metadata.custom_field_metadata()
        self.has_read_column = prefs["read_lookup_name"] in fm and fm[prefs["read_lookup_name"]]['datatype'] == 'bool'
        self.has_fav_column = prefs["fav_lookup_name"] in fm and fm[prefs["fav_lookup_name"]]['datatype'] == 'bool'
        self.has_shelf_column = prefs["shelf_lookup_name"] in fm and fm[prefs["shelf_lookup_name"]]['datatype'] == 'text'
        self.has_pb_position_column = prefs["pb_position_lookup_name"] in fm and fm[prefs["pb_position_lookup_name"]]['datatype'] in ["text", "comments"]
        self.has_cr3_position_column = prefs["cr3_position_lookup_name"] in fm and fm[prefs["cr3_position_lookup_name"]]['datatype'] in ["text", "comments"]

        main_menu = self.menu
        main_menu.clear()
        
        self.qaction.setIcon(self.get_icon('pocketbook'))

        main_menu.addAction(self.get_icon('forward'), ('Send all metadata to Pocketbook'), self.send_all_command)
        sendMenu = main_menu.addMenu(('Send selected'))
        sendMenu.addAction(self.get_icon('catalog'), ('Send collections'), self.send_collections_command)
        sendMenu.addAction(self.get_icon('ok'), ('Send read statuses'), self.send_read_command)
        sendMenu.addAction(self.get_icon('donate'), ('Send favorite statuses'), self.send_favorite_command)

        main_menu.addAction(self.get_icon('back'), ('Load all metadata from Pocketbook'), self.load_all_command)
        loadMenu = main_menu.addMenu(('Load selected'))
        loadMenu.addAction(self.get_icon('catalog'), ('Load collections'), self.load_collections_command)
        loadMenu.addAction(self.get_icon('ok'), ('Load read statuses'), self.load_read_command)
        loadMenu.addAction(self.get_icon('donate'), ('Load favorite statuses'), self.load_favorite_command)

        main_menu.addAction(self.get_icon('auto-reload'), ('Sync reading positions'), self.sync_position)

        main_menu.addAction(self.get_icon('config'), ('Configure plugin'), self.open_settings)
        
    
    # Menu commands functions

    def send_all_command(self):
        print("send all")
        if self.has_shelf_column or self.has_fav_column or self.has_read_column:
            self.run_sync_job("send all", "Sending all metadata to Pocketbook", "All statuses and collections have been sent to Pocketbook")
        else:
            if prefs["shelf_lookup_name"] == "" and prefs["read_lookup_name"] == "" and prefs["fav_lookup_name"] == "":
                error_dialog(self.gui, "Can not send metadata", "No columns set up for sending metadata", show=True)
            if prefs["shelf_lookup_name"] != "" and self.has_shelf_column == False:
                error_dialog(self.gui, "Can not send metadata", "Column " + prefs["shelf_lookup_name"] + " does not exist or is of wrong type.", show=True)
            if prefs["read_lookup_name"] != "" and self.has_read_column == False:
                error_dialog(self.gui, "Can not send metadata", "Column " + prefs["read_lookup_name"] + " does not exist or is of wrong type.", show=True)
            if prefs["fav_lookup_name"] != "" and self.has_fav_column == False:
                error_dialog(self.gui, "Can not send metadata", "Column " + prefs["fav_lookup_name"] + " does not exist or is of wrong type.", show=True)



    def send_collections_command(self):
        if self.has_shelf_column:
            self.run_sync_job("send collections", "Sending collections to Pocketbook", "All collections have been sent to Pocketbook")
        else:
            if prefs["shelf_lookup_name"] == "":
                error_dialog(self.gui, "Can not send collections", "No column set up for syncing collections.", show=True)
            else:
                error_dialog(self.gui, "Can not send collections", "Column " + prefs["shelf_lookup_name"] + " does not exist or is of wrong type.", show=True)
        


    def send_read_command(self):
        if self.has_read_column:
            self.run_sync_job("send read", "Sending read statuses to Pocketbook", "All read statuses have been sent to Pocketbook")
        else:
            if prefs["read_lookup_name"] == "":
                error_dialog(self.gui, "Can not send read statuses", "No column set up for syncing read statuses.", show=True)
            else:
                error_dialog(self.gui, "Can not send read statuses", "Column " + prefs["read_lookup_name"] + " does not exist or is of wrong type.", show=True)



    def send_favorite_command(self):
        if self.has_fav_column:
            self.run_sync_job("send favorite", "Sending favorite statuses to Pocketbook", "All favorite statuses have been sent to Pocketbook")
        else:
            if prefs["fav_lookup_name"] == "":
                error_dialog(self.gui, "Can not send favorite statuses", "No column set up for syncing favorite statuses.", show=True)
            else:
                error_dialog(self.gui, "Can not send favorite statuses", "Column " + prefs["fav_lookup_name"] + " does not exist or is of wrong type.", show=True)



    def load_all_command(self):
        if self.has_shelf_column or self.has_fav_column or self.has_read_column:
            self.run_sync_job("load all", "Loading all metadata from Pocketbook", "All statuses and collections have been loaded from Pocketbook")
        else:
            if prefs["shelf_lookup_name"] == "" and prefs["read_lookup_name"] == "" and prefs["fav_lookup_name"] == "":
                error_dialog(self.gui, "Can not load metadata", "No columns set up for loading metadata", show=True)
            if prefs["shelf_lookup_name"] != "" and self.has_shelf_column == False:
                error_dialog(self.gui, "Can not load metadata", "Column " + prefs["shelf_lookup_name"] + " does not exist or is of wrong type.", show=True)
            if prefs["read_lookup_name"] != "" and self.has_read_column == False:
                error_dialog(self.gui, "Can not load metadata", "Column " + prefs["read_lookup_name"] + " does not exist or is of wrong type.", show=True)
            if prefs["fav_lookup_name"] != "" and self.has_fav_column == False:
                error_dialog(self.gui, "Can not load metadata", "Column " + prefs["fav_lookup_name"] + " does not exist or is of wrong type.", show=True)




    def load_collections_command(self):
        if self.has_shelf_column:
            self.run_sync_job("load collections", "Loading collections from Pocketbook", "All collections have been loaded from Pocketbook")
        else:
            if prefs["shelf_lookup_name"] == "":
                error_dialog(self.gui, "Can not load collections", "No column set up for syncing collections.", show=True)
            else:
                error_dialog(self.gui, "Can not load collections", "Column " + prefs["shelf_lookup_name"] + " does not exist or is of wrong type.", show=True)
        



    def load_read_command(self):
        if self.has_read_column:
            self.run_sync_job("load read", "Loading read statuses from Pocketbook", "All read statuses have been loaded from Pocketbook")
        else:
            if prefs["read_lookup_name"] == "":
                error_dialog(self.gui, "Can not load read statuses", "No column set up for syncing read statuses.", show=True)
            else:
                error_dialog(self.gui, "Can not load read statuses", "Column " + prefs["read_lookup_name"] + " does not exist or is of wrong type.", show=True)




    def load_favorite_command(self):
        if self.has_fav_column:
            self.run_sync_job("load favorite", "Loading favorite statuses from Pocketbook", "All favorite statuses have been loaded from Pocketbook")
        else:
            if prefs["fav_lookup_name"] == "":
                error_dialog(self.gui, "Can not load favorite statuses", "No column set up for syncing favorite statuses.", show=True)
            else:
                error_dialog(self.gui, "Can not load favorite statuses", "Column " + prefs["fav_lookup_name"] + " does not exist or is of wrong type.", show=True)
    



    def sync_position(self):
        if self.has_pb_position_column or self.has_cr3_position_column:
            self.run_sync_job("sync_position", "Syncing with reading positions", "Reading positions synced")
        else:
            if prefs["pb_position_lookup_name"] == "" and prefs["cr3_position_lookup_name"] == "":
                error_dialog(self.gui, "Can not sync reading positions", "No columns set up for syncing reading positions.", show=True)
            if prefs["pb_position_lookup_name"] != "" and self.has_pb_position_column == False:
                error_dialog(self.gui, "Can not sync reading positions", "Column " + prefs["pb_position_lookup_name"] + " does not exist or is of wrong type.", show=True)
            if prefs["cr3_position_lookup_name"] != "" and self.has_cr3_position_column == False:
                error_dialog(self.gui, "Can not sync reading positions", "Column " + prefs["cr3_position_lookup_name"] + " does not exist or is of wrong type.", show=True)
        

    
    # Run job to sync any metadata
        
    def run_sync_job(self, command, desc, done_msg):
        if self.get_device_DB_path():
            print("PB-COLLECTIONS: Start syncing metadata")

            # data in a dict where all additianal variables used by a job are stored
            data = {}        
            data["dbpath"] = self.gui.current_db.library_path
            data["device"] = self.gui.library_view.model().device_connected
            data["device_DB_path"] = self.get_device_DB_path()
            data["device_storages"] = self.get_device_storages()
            data["has_read_column"] = self.has_read_column
            data["has_fav_column"] = self.has_fav_column
            data["has_shelf_column"] = self.has_shelf_column
            data["has_cr3_position_column"] = self.has_cr3_position_column
            data["has_pb_position_column"] = self.has_pb_position_column

            args = ['calibre_plugins.pocketbook_collections.main','sync_metadata', (data, command, done_msg)]
            self.gui.job_manager.run_job(self.Dispatcher(self.sync_done), "arbitrary", args=args, description=desc)

        else:
            error_dialog(self.gui, "Database not found", "No device collected or current device is not supported.", show=True)


    # After the job finished we must update downloaded metadata in calibre. It should not be updated from inside the job, because then GUI would not be refreshed.

    def sync_done(self, job):

        result = job.result

        try:
            to_load, done_msg = result

            if to_load == "error":
                error_dialog(self.gui, "Error", done_msg, show=True)

            else:

                if len(to_load["read"]) > 0:
                    self.gui.current_db.new_api.set_field(prefs["read_lookup_name"], to_load["read"])

                if len(to_load["fav"]) > 0:
                    self.gui.current_db.new_api.set_field(prefs["fav_lookup_name"], to_load["fav"])

                if len(to_load["shelf"]) > 0:
                    self.gui.current_db.new_api.set_field(prefs["shelf_lookup_name"], to_load["shelf"])

                if len(to_load["cr3_position"]) > 0:
                    self.gui.current_db.new_api.set_field(prefs["cr3_position_lookup_name"], to_load["cr3_position"])

                if len(to_load["pb_position"]) > 0:
                    self.gui.current_db.new_api.set_field(prefs["pb_position_lookup_name"], to_load["pb_position"])


                self.gui.iactions['Edit Metadata'].refresh_gui(to_load["books_to_refresh"], covers_changed=False)

                print(done_msg)
                print("PB-COLLECTIONS: End syncing metadata")

        except:
            error_dialog(self.gui, "Error", "Unknown error.", show=True)



            




    def open_settings(self):
        self.interface_action_base_plugin.do_user_config(self.gui)

        

    def get_icon(self, icon_name):
        icons = get_icons(PLUGIN_ICONS)
        icon_path = "images/" + icon_name + ".png"
        return icons[icon_path]



    def get_device_DB_path(self):
        device = self.gui.library_view.model().device_connected
        if device:
            devicePath = self.gui.device_manager.connected_device._main_prefix

            for version in (3, 2):
                explorer = 'explorer-%i' % version
                dbPath = os.path.join(devicePath, 'system', explorer, explorer + '.db')
                if os.path.exists(dbPath):
                    return dbPath
        return False
    


    def get_device_storages(self):
        storages = {}
        device = self.gui.library_view.model().device_connected
        if device:
            main_storage = self.gui.device_manager.connected_device._main_prefix
            storages["main"] = main_storage

            card_a = self.gui.device_manager.connected_device._card_a_prefix
            card_b = self.gui.device_manager.connected_device._card_b_prefix

            if(card_a):
                storages["card"] = card_a
            elif(card_b):
                storages["card"] = card_b

        return storages









    
