__license__   = 'GPL v3'
__copyright__ = '2024, Anareaty <reatymain@gmail.com>'
__docformat__ = 'restructuredtext en'


import json, time, re, os, errno
import sqlite3 as sqlite
from contextlib import closing
from calibre.library import db as calibre_db
from calibre_plugins.pocketbook_collections.config import prefs
import xml.etree.ElementTree as ET



# The main function to start syncing any metadata. It is run as a job.

def sync_metadata(data, command, done_msg):
    
    # Create the dict to return after syncing
    to_load = {
        "read": {},
        "fav": {},
        "shelf": {},
        "cr3_position": {},
        "pb_position": {},
        "books_to_refresh": []
    }

    
    # Check if columns are set up and exist

    current_db = calibre_db(data["dbpath"])
    calibreAPI = current_db.new_api

    



    # Check if device is connected
    device_DB_path = data["device_DB_path"]

    if device_DB_path == False:
        return "error", "No Pocketbook device connected or current device is not supported"
    
    else:

        # Find calibre metadata in device storages

        device_main_storage = data["device_storages"]["main"]
        device_metadata_path_main = device_main_storage + "metadata.calibre"

        with open(device_metadata_path_main, "r") as file_main:
            try:
                data_main = json.load(file_main)
            except:
                return "error", "Can not reach device metadata, it is probably still updating. Try again later."

      
        device_card = None
        if data["device_storages"].get("card"):
            device_card = data["device_storages"]["card"]
            device_metadata_path_card = device_card + "metadata.calibre"

            with open(device_metadata_path_card, "r") as file_card:
                try:
                    data_card = json.load(file_card)
                except:
                    return "error", "Can not reach device metadata, it is probably still updating. Try again later."
        else:
            print("No card in the device")


        



        
        

        # Connect to the Pocketbook database

        with closing(sqlite.connect(device_DB_path)) as db:
            db.row_factory = lambda cursor, row: {col[0]: row[i] for i, col in enumerate(cursor.description)}
            cursor = db.cursor()

            


            # Clean database

            cursor.execute("DELETE FROM books_settings WHERE bookid IN (SELECT id FROM books_impl WHERE id NOT IN (SELECT book_id FROM files))")
            cursor.execute("DELETE FROM booktogenre WHERE bookid IN (SELECT id FROM books_impl WHERE id NOT IN (SELECT book_id FROM files))")
            cursor.execute("DELETE FROM social WHERE bookid IN (SELECT id FROM books_impl WHERE id NOT IN (SELECT book_id FROM files))")
            cursor.execute("DELETE FROM books_impl WHERE id NOT IN (SELECT book_id FROM files)")
            cursor.execute("DELETE FROM books_fast_hashes WHERE book_id NOT IN (SELECT book_id FROM files)")
            db.commit()

            



            # Find prexixes of the reader storages

            storage_id_main = cursor.execute("SELECT id FROM storages WHERE name = 'InternalStorage'").fetchone()["id"]
            storage_prefix_main_split = cursor.execute("SELECT name FROM folders WHERE storageid = " + str(storage_id_main)).fetchone()["name"].split("/")
            storage_prefix_main = "/" + storage_prefix_main_split[1] + "/" + storage_prefix_main_split[2]

            storage_prefix_card = None
            if data["device_storages"].get("card"):
                storage_id_card = cursor.execute("SELECT id FROM storages WHERE name LIKE 'SDCard%'").fetchone()["id"]
                storage_prefix_card_split = cursor.execute("SELECT name FROM folders WHERE storageid = " + str(storage_id_card)).fetchone()["name"].split("/")
                storage_prefix_card = "/" + storage_prefix_card_split[1] + "/" + storage_prefix_card_split[2]

            

            # Find current profile

            profile_ID = str(get_current_profile_ID(device_main_storage, device_DB_path))

            


            # Get CoolReader history path

            if data["has_cr3_position_column"] and command == "sync_position":
                try:
                    cr3hist_path = get_cr3hist_path(device_main_storage, device_card, storage_prefix_main, storage_prefix_card)
                except:
                    print("PB-COLLECTIONS: Can not find Coolreader history path")



            # Get all books with missing authors

            books_missing_authors = cursor.execute("SELECT id FROM books_impl WHERE author = ''").fetchall()
            books_missing_authors_IDs = [book["id"] for book in books_missing_authors]




            # Get IDs of all books in Calibre
                
            calibre_book_IDs = calibreAPI.all_book_ids()

            # Try to check and sync every book

            for calibre_book_ID in calibre_book_IDs:

                # Find the book in Calibre metadata objects on device. We must check both the main storage and the card, it it is exist.

                bookData = next((bookData for bookData in data_main if bookData["application_id"] == calibre_book_ID), None)
                storage_prefix = storage_prefix_main                

                if bookData == None and data["device_storages"].get("card"):
                    bookData = next((bookData for bookData in data_card if bookData["application_id"] == calibre_book_ID), None)
                    storage_prefix = storage_prefix_card
                
                
                # If bookData is found, it means it is on device and indexed by Calibre. Proceed to check if it is indexed by Pocketbook.
                    
                book_row = None
                
                if bookData:

                    # Find the path to the book on the device

                    bookPath = bookData["lpath"]
                    book_filesize = bookData["size"]

                    # Get the filename an folder of the book

                    bookPath_split = bookPath.split("/")
                    bookPath_file = bookPath_split[-1].replace("'", r"''")
                    bookPath_split.pop()
                    bookPath_folder = storage_prefix + "/" + "/".join(bookPath_split)
                    bookPath_folder = bookPath_folder.replace("'", r"''")


                    # Find the folder in Pocketbook database

                    book_folder_row = cursor.execute("SELECT id FROM folders WHERE name = '" + bookPath_folder + "'").fetchone()

                    if book_folder_row:
                        book_folder_id = str(book_folder_row["id"])

                        # Find the book in Pocketbook database

                        book_row = cursor.execute("SELECT book_id as id, filename FROM files WHERE folder_id = " + book_folder_id + " AND filename = '" + bookPath_file + "'").fetchone()
                    

                # If the book exist and indexed by Pocketbook proceed to sync metadata between the reader and Calibre

                if book_row:

                    reader_book_ID = str(book_row["id"])

                    # Find the timestamp of the last modification in Calibre
                    lastMod = calibreAPI.field_for("last_modified", calibre_book_ID)
                    try:
                        lastModTS = int(lastMod.timestamp())
                    except:
                        lastModTS = int(time.mktime(lastMod.timetuple()))







                    # Fix missing author

                    if book_row["id"] in books_missing_authors_IDs:
                        title = calibreAPI.field_for("title", calibre_book_ID)
                        authors = calibreAPI.field_for("author_sort", calibre_book_ID, default_value=[])
                        authors_string = authors.replace(",", "").replace(" & ", ", ")
                        author_first = authors.split(" & ")[0].split(",")[0]
                        author_first_letter = author_first[0]
                        book_format = bookPath_file.rsplit('.', 1)[1] 

                        if book_format == "pdf":
                            title = calibreAPI.field_for("title", calibre_book_ID)


                            print("title")
                            print(title)
                            print("authors_string")
                            print(authors_string)
                            print("author_first")
                            print(author_first)
                            print("author_first_letter")
                            print(author_first_letter)
                            print("reader_book_ID")
                            print(reader_book_ID)

                            # cursor.execute("UPDATE books_impl SET title = '" + title + "', author = '" + authors_string + "', firstauthor = '" + author_first + "', first_author_letter = '" + author_first_letter + "' WHERE id = " + reader_book_ID)
                        else:
                            print("not pdf")
                            cursor.execute("UPDATE books_impl SET author = '" + authors_string + "', firstauthor = '" + author_first + "', first_author_letter = '" + author_first_letter + "' WHERE id = " + reader_book_ID)
                        db.commit()




                    
                    
                    
                    # Run functions to send or load selected metadata based on specified command

                    refresh_book = False
                    
                    if command == "send all":
                        send_statuses(data, db, calibreAPI, calibre_book_ID, reader_book_ID, lastModTS, profile_ID)
                        send_collections(data, db, calibreAPI, calibre_book_ID, reader_book_ID, lastModTS)

                    if command == "send collections":
                        send_collections(data, db, calibreAPI, calibre_book_ID, reader_book_ID, lastModTS)

                    elif command == "send read":
                        send_read_status(data, db, calibreAPI, calibre_book_ID, reader_book_ID, lastModTS, profile_ID)


                    elif command == "send favorite":
                        send_favorite_status(data, db, calibreAPI, calibre_book_ID, reader_book_ID, lastModTS, profile_ID)

                    if command == "load all":
                        to_load_statuses = load_statuses(data, db, calibreAPI, calibre_book_ID, reader_book_ID, profile_ID)
                        if to_load_statuses["read"]:
                            to_load["read"][calibre_book_ID] = to_load_statuses["read"]["status"]
                            refresh_book = True
                        if to_load_statuses["fav"]:
                            to_load["fav"][calibre_book_ID] = to_load_statuses["fav"]["status"]
                            refresh_book = True
                        to_load_shelf = load_collections(data, db, calibreAPI, calibre_book_ID, reader_book_ID)
                        if to_load_shelf:
                            to_load["shelf"][calibre_book_ID] = to_load_shelf["status"]
                            refresh_book = True


                    elif command == "load collections":  
                        to_load_shelf = load_collections(data, db, calibreAPI, calibre_book_ID, reader_book_ID)
                        if to_load_shelf:
                            to_load["shelf"][calibre_book_ID] = to_load_shelf["status"]
                            refresh_book = True

                    elif command == "load read":
                        to_load_read = load_read_status(data, db, calibreAPI, calibre_book_ID, reader_book_ID, profile_ID)
                        if to_load_read:
                            to_load["read"][calibre_book_ID] = to_load_read["status"]
                            refresh_book = True

                    elif command == "load favorite":
                        to_load_fav = load_favorite_status(data, db, calibreAPI, calibre_book_ID, reader_book_ID, profile_ID)
                        if to_load_fav:
                            to_load["fav"][calibre_book_ID] = to_load_fav["status"]
                            refresh_book = True


                    elif command == "sync_position":
                        if data["has_cr3_position_column"]:
                            to_load_position_cr3 = cr3_sync_position(cr3hist_path, bookPath_folder, bookPath_file, book_filesize, calibre_book_ID, calibreAPI)
                            if to_load_position_cr3:
                                to_load["cr3_position"][calibre_book_ID] = to_load_position_cr3["status"]
                        if data["has_pb_position_column"]:
                            to_load_position_pb = pb_sync_position(db, calibreAPI, calibre_book_ID, reader_book_ID, profile_ID)
                            if to_load_position_pb:
                                to_load["pb_position"][calibre_book_ID] = to_load_position_pb["status"]

                        
                    if refresh_book:
                        to_load["books_to_refresh"].append(calibre_book_ID)

                    


    # We must return the dict containing all the loaded data that should be changed in Calibre
    return to_load, done_msg




# Function to send collections from Calibre to the reader

def send_collections(data, db, calibreAPI, calibre_book_ID, reader_book_ID, lastModTS):
    # Sync must only work if custom column for collections is exist

    if data["has_shelf_column"]:
        cursor = db.cursor()

        # For it to be easier to compare collections between the reader and Calibre we must create the dict of relations between the collections names and their IDs in both locations. 
        shelf_dicts = create_shelfs_dicts(data, db, calibreAPI)

        # Find the list of existing collections names for the book in Calibre
        calibreBookShelfNames = calibreAPI.field_for(prefs["shelf_lookup_name"], calibre_book_ID, default_value=[])
    
        # Find the list of existing collections objects for the book on reader
        readerBookShelfs = cursor.execute("SELECT bookshelfid, is_deleted, ts FROM bookshelfs_books WHERE bookid = " + reader_book_ID).fetchall()


        # Check every collection in Calibre

        for calibreBookShelfName in calibreBookShelfNames:

            # For each collection in Calibre find corresponding collection on reader

            readerBookShelfID = shelf_dicts["shelfsNameToReader"][calibreBookShelfName]

            # If the collection in reader does not exist we must create it
            if readerBookShelfID == None:
                # First create the collection itself
                cursor.execute("INSERT INTO bookshelfs(name, is_deleted, ts) VALUES('" + calibreBookShelfName + "', 0, " + str(lastModTS) + ")")
                db.commit()
                readerBookShelfID = cursor.execute("SELECT id FROM bookshelfs WHERE name = '" + calibreBookShelfName +"'").fetchone()["id"]

                # The add book to collection
                cursor.execute("INSERT INTO bookshelfs_books(bookshelfid, bookid, ts, is_deleted) VALUES(" + str(readerBookShelfID) + ", " + reader_book_ID + ", " + str(lastModTS) + ", 0)")
                db.commit()
                

            # If the collection in reader does exist, we must check if it has the book
            else:

                # Find the collection in the list of reader collections for the book 
                readerBookShelf = None
                for readerShelfRow in readerBookShelfs:
                    if readerShelfRow["bookshelfid"] == readerBookShelfID:
                        readerBookShelf = readerShelfRow


                # If the book is in collection check if it is marked as deleted
                if readerBookShelf:
                    if readerBookShelf["is_deleted"]:
                        # If deleted mark as not deleted
                        cursor.execute("UPDATE bookshelfs_books SET is_deleted = 0, ts = " + str(lastModTS) + " WHERE bookid = " + reader_book_ID + " AND bookshelfid = " + str(readerBookShelfID))
                        db.commit()

                # If the book is not in collection, add it to collection
                else:
                    cursor.execute("INSERT INTO bookshelfs_books(bookshelfid, bookid, ts, is_deleted) VALUES(" + str(readerBookShelfID) + ", " + reader_book_ID + ", " + str(lastModTS) + ", 0)")
                    db.commit()


        # Check every collection in reader

        for readerBookShelfRow in readerBookShelfs:

            # Get collection name and id in reader
            readerBookShelfID = readerBookShelfRow["bookshelfid"]
            shelfName = shelf_dicts["shelfsReaderToCalibre"][readerBookShelfID]["name"]

            # Look for the corresponding collection in Calibre
            calibreShelf = None
            for calibreBookShelfName in calibreBookShelfNames:
                if calibreBookShelfName == shelfName:
                    calibreShelf = calibreBookShelfName

            # If collection is not in Calibre and not marked as deleted in reader, we must mark it as deleted
            if calibreShelf == None and readerBookShelfRow["is_deleted"] != 1:
                cursor.execute("UPDATE bookshelfs_books SET is_deleted = 1, ts = " + str(lastModTS) + " WHERE bookid = " + reader_book_ID + " AND bookshelfid = " + str(readerBookShelfID))
                db.commit()

            # If collection is in Calibre, it is already processed, so do nothing.




# Function to send read statuses from Calibre to the reader

def send_read_status(data, db, calibreAPI, calibre_book_ID, reader_book_ID, lastModTS, profile_ID):

    if data["has_read_column"]:
        cursor = db.cursor()

        # Find read status in Calibre

        read = calibreAPI.field_for(prefs["read_lookup_name"], calibre_book_ID)
        completed = "0"
        if read: 
            completed = "1"

        # Find read status in reader

        statusRow = cursor.execute("SELECT bookid, completed FROM books_settings WHERE bookid = " + reader_book_ID + " AND profileid = " + profile_ID).fetchone()

        # If status in reader exist and different from Calibre, update status in reader
        if statusRow:
            if completed != str(statusRow["completed"]):
                cursor.execute("UPDATE books_settings SET completed = " + str(completed) + ", completed_ts = " + str(lastModTS) + " WHERE bookid = " + reader_book_ID + " AND profileid = " + profile_ID)
                db.commit()
        # If status in reader do not exist, insert new status row
        else:
            cursor.execute("INSERT INTO books_settings(bookid, profileid, completed, completed_ts) VALUES(" + reader_book_ID + ", " + profile_ID + ", " + str(completed) + ", " + str(lastModTS) + ")")
            db.commit()




# Function to send favorite statuses from Calibre to the reader

def send_favorite_status(data, db, calibreAPI, calibre_book_ID, reader_book_ID, lastModTS, profile_ID):

    if data["has_fav_column"]:
        cursor = db.cursor()

        # Find favorite status in Calibre

        fav = calibreAPI.field_for(prefs["fav_lookup_name"], calibre_book_ID)
        favorite = "0"
        if fav: 
            favorite = "1"

        # Find favorite status in reader

        statusRow = cursor.execute("SELECT bookid, favorite FROM books_settings WHERE bookid = " + reader_book_ID + " AND profileid = " + profile_ID).fetchone()

        # If status in reader exist and different from Calibre, update status in reader
        if statusRow:
            if favorite != str(statusRow["favorite"]):
                cursor.execute("UPDATE books_settings SET favorite = " + str(favorite) + ", favorite_ts = " + str(lastModTS) + " WHERE bookid = " + reader_book_ID + " AND profileid = " + profile_ID)
                db.commit()
        # If status in reader do not exist, insert new status row
        else:
            cursor.execute("INSERT INTO books_settings(bookid, profileid, favorite, favorite_ts) VALUES(" + reader_book_ID + ", " + profile_ID + ", " + str(favorite) + ", " + str(lastModTS) + ")")
            db.commit()




# Function to send both read and favorite statuses from Calibre to the reader

def send_statuses(data, db, calibreAPI, calibre_book_ID, reader_book_ID, lastModTS, profile_ID):
    
    if data["has_read_column"] or data["has_fav_column"]:
        cursor = db.cursor()

        # Find statuses in Calibre
        completed = "0"
        favorite = "0"

        if data["has_read_column"]:
            read = calibreAPI.field_for(prefs["read_lookup_name"], calibre_book_ID)
            if read: 
                completed = "1"

        if data["has_fav_column"]:
            fav = calibreAPI.field_for(prefs["fav_lookup_name"], calibre_book_ID)
            if fav: 
                favorite = "1"
        
        # Find statuses in reader

        statusRow = cursor.execute("SELECT bookid, completed, favorite FROM books_settings WHERE bookid = " + reader_book_ID + " AND profileid = " + profile_ID).fetchone()

        # If any status in reader exist and different from Calibre, update statuses in reader
        if statusRow:
            if (data["has_read_column"] and completed != str(statusRow["completed"])) or (data["has_fav_column"] and favorite != str(statusRow["favorite"])):
                cursor.execute("UPDATE books_settings SET completed = " + str(completed) + ", completed_ts = " + str(lastModTS) + ", favorite = " + str(favorite) + ", favorite_ts = " + str(lastModTS) + " WHERE bookid = " + reader_book_ID + " AND profileid = " + profile_ID)
                db.commit()
        # If statuses in reader do not exist, insert new status row
        else:
            cursor.execute("INSERT INTO books_settings(bookid, profileid, completed, completed_ts, favorite, favorite_ts) VALUES(" + reader_book_ID + ", " + profile_ID + ", " + str(completed) + ", " + str(lastModTS) + ", " + str(favorite) + ", " + str(lastModTS) + ")")
            db.commit()




# Function to load collections from reader to Calibre
# We don't update Calibre metadata directly from this module, because it would not refresh the GUI. 
# Instead we collect all the data that need to be updated and return it to later update in sync_done function in ui module.

def load_collections(data, db, calibreAPI, calibre_book_ID, reader_book_ID):

    # Create the list of collections that must be loaded for this book
    to_load_shelf = None

    # Sync must only work if custom column for collections is exist

    if data["has_shelf_column"]:
        cursor = db.cursor()

        # For it to be easier to compare collections between the reader and Calibre we must create the dict of relations between the collections names and their IDs in both locations.
        shelf_dicts = create_shelfs_dicts(data, db, calibreAPI)

        # Variable to show if collections must be updated
        update_shelfs_in_Calibre = False

        # Find the list of existing collections names for the book in Calibre
        calibreBookShelfNames = calibreAPI.field_for(prefs["shelf_lookup_name"], calibre_book_ID, default_value=[])

        # Create the editable copy of the collections in Calibre
        calibreBookShelfNamesList = list(calibreBookShelfNames)

        # Find the list of existing collections objects for the book on reader
        readerBookShelfs = cursor.execute("SELECT bookshelfid, is_deleted, ts FROM bookshelfs_books WHERE bookid = " + reader_book_ID).fetchall()


        # Check every collection in Calibre

        for calibreBookShelfName in calibreBookShelfNames:

            # For each collection in Calibre find corresponding collection in reader
            readerBookShelfID = shelf_dicts["shelfsNameToReader"][calibreBookShelfName]

            # If the collection in reader does not exist, delete collection in Calibre
            if readerBookShelfID == None:
                update_shelfs_in_Calibre = True
                calibreBookShelfNamesList.remove(calibreBookShelfName)
                
            # If the collection in reader does exist, we must check if it has the book
            else:
                # Find the collection in the list of reader collections for the book 
                readerBookShelf = None
                for readerShelfRow in readerBookShelfs:
                    if readerShelfRow["bookshelfid"] == readerBookShelfID:
                        readerBookShelf = readerShelfRow

                # If the book is in collection check if it is marked as deleted
                if readerBookShelf:
                    # If deleted, delete collection from Calibre collections list
                    if readerBookShelf["is_deleted"]:
                        update_shelfs_in_Calibre = True
                        calibreBookShelfNamesList.remove(calibreBookShelfName)

                # If the book is not in collection, delete collection from Calibre collections list
                else:
                    update_shelfs_in_Calibre = True
                    calibreBookShelfNamesList.remove(calibreBookShelfName)


        # Check every collection in reader
                    
        for readerBookShelfRow in readerBookShelfs:

            # Get collection name and id in reader
            readerBookShelfID = readerBookShelfRow["bookshelfid"]
            shelfName = shelf_dicts["shelfsReaderToCalibre"][readerBookShelfID]["name"]

            # Look for the corresponding collection in Calibre
            calibreShelf = None
            for calibreBookShelfName in calibreBookShelfNames:
                if calibreBookShelfName == shelfName:
                    calibreShelf = calibreBookShelfName

            # If collection is not in Calibre and not marked as deleted, add collection to Calibre collections list
            if calibreShelf == None and readerBookShelfRow["is_deleted"] != 1:
                update_shelfs_in_Calibre = True
                calibreBookShelfNamesList.append(shelfName)

            # If collection is in Calibre, it is already processed, so do nothing.


        # If Calibre collections list is changed, we must save the object with changed list and book id for updating 

        if update_shelfs_in_Calibre:
            to_load_shelf = {"status": calibreBookShelfNamesList}
    
    return to_load_shelf



# Function to load read statuses from reader to Calibre
# We don't update Calibre metadata directly from this module, because it would not refresh the GUI. 
# Instead we collect all the data that need to be updated and return it to later update in sync_done function in ui module.

def load_read_status(data, db, calibreAPI, calibre_book_ID, reader_book_ID, profile_ID):
    
    to_load_read = None

    if data["has_read_column"]:
        cursor = db.cursor()

        # Find read status in reader

        statusRow = cursor.execute("SELECT bookid, completed FROM books_settings WHERE bookid = " + reader_book_ID + " AND profileid = " + profile_ID).fetchone()

        # If status exist, check the status in Calibre

        read = calibreAPI.field_for(prefs["read_lookup_name"], calibre_book_ID)
        completed = "0"
        if read: 
            completed = "1"

        if statusRow:
            

            # If status in Calibre is different from reader,  save status to update Calibre
            if completed != str(statusRow["completed"]):
                completed = str(statusRow["completed"])

                if str(completed) == "1":
                    read = True
                else:
                    read = False

                to_load_read = {"status": read}
                
        # If status in reader not exist delete status in Calibre
        elif completed == "1":
            to_load_read = {"status": None}
        
    return to_load_read
       


    
# Function to load favorite statuses from reader to Calibre
# We don't update Calibre metadata directly from this module, because it would not refresh the GUI. 
# Instead we collect all the data that need to be updated and return it to later update in sync_done function in ui module.

def load_favorite_status(data, db, calibreAPI, calibre_book_ID, reader_book_ID, profile_ID):
    
    to_load_fav = None

    if data["has_fav_column"]:
        cursor = db.cursor()

        # Find favorite status in reader
        
        statusRow = cursor.execute("SELECT bookid, favorite FROM books_settings WHERE bookid = " + reader_book_ID + " AND profileid = " + profile_ID).fetchone()

        # If status exist, check the status in Calibre

        fav = calibreAPI.field_for(prefs["fav_lookup_name"], calibre_book_ID)
        favorite = "0"
        if fav: 
            favorite = "1"

        if statusRow:
            
            # If status in Calibre is different from reader,  save status to update Calibre
            if favorite != str(statusRow["favorite"]):
                favorite = str(statusRow["favorite"])

                if str(favorite) == "1":
                    fav = True
                else:
                    fav = False

                to_load_fav = {"status": fav}

        # If status in reader not exist delete status in Calibre
        elif favorite == "1":
            to_load_fav = {"status": None}

    return to_load_fav




# Function to load both statuses from reader to Calibre
# We don't update Calibre metadata directly from this module, because it would not refresh the GUI. 
# Instead we collect all the data that need to be updated and return it to later update in sync_done function in ui module.
    
def load_statuses(data, db, calibreAPI, calibre_book_ID, reader_book_ID, profile_ID):
    
    cursor = db.cursor()
    to_load_statuses = {
        "read": None,
        "fav": None
    }

    if data["has_read_column"] or data["has_fav_column"]:

        # Find statuses in reader

        statusRow = cursor.execute("SELECT bookid, completed, favorite FROM books_settings WHERE bookid = " + reader_book_ID + " AND profileid = " + profile_ID).fetchone()

        # If at least one status exist, check statuses in Calibre

        completed = "0"
        favorite = "0"

        if data["has_read_column"]:
            read = calibreAPI.field_for(prefs["read_lookup_name"], calibre_book_ID)
            if read: 
                completed = "1"

        if data["has_fav_column"]:
            fav = calibreAPI.field_for(prefs["fav_lookup_name"], calibre_book_ID)
            if fav: 
                favorite = "1"

        if statusRow:
            

            

            loadStatusFromDevice = False

            # If statuses in Calibre are different from reader,  save statuses to update Calibre
            
            if data["has_read_column"] and completed != str(statusRow["completed"]):
                completed = str(statusRow["completed"])
                loadStatusFromDevice = True

            if data["has_fav_column"] and favorite != str(statusRow["favorite"]):
                favorite = str(statusRow["favorite"])
                loadStatusFromDevice = True
            
            if loadStatusFromDevice:
                if str(completed) == "1":
                    read = True
                else:
                    read = False

                if str(favorite) == "1":
                    fav = True
                else:
                    fav = False

                to_load_statuses["read"] = {"status": read}
                to_load_statuses["fav"] = {"status": fav}
        
        # If statuses in reader not exist delete status in Calibre
        else:
            if completed == "1":
                to_load_statuses["read"] = {"status": None}
            if favorite == "1":
                to_load_statuses["fav"] = {"status": None}


    return to_load_statuses


def create_shelfs_dicts(data, db, calibreAPI):
    cursor = db.cursor()

    shelf_dicts = {
        "shelfsReaderToCalibre": {},
        "shelfsNameToReader": {}
    }

    if data["has_shelf_column"]:
        calibreShelfIDs = calibreAPI.all_field_ids(prefs["shelf_lookup_name"])
        if calibreShelfIDs == None:
            calibreShelfIDs = []
    else:
        calibreShelfIDs = []

    readerShelfRows = cursor.execute("SELECT id, name FROM bookshelfs").fetchall()

    for calibreShelfID in calibreShelfIDs:
        shelfName = calibreAPI.get_item_name(prefs["shelf_lookup_name"], calibreShelfID)
        readerShelfID = None

        i = 0
        while i < len(readerShelfRows):
            readerShelf = readerShelfRows[i]
            if readerShelf['name'] == shelfName:
                readerShelfID = readerShelf["id"]
                shelf_dicts["shelfsReaderToCalibre"][readerShelfID] = {"calibreShelfID": calibreShelfID, "name": shelfName}
                del readerShelfRows[i]
            else:
                i += 1

        shelf_dicts["shelfsNameToReader"][shelfName] = readerShelfID
            

    for readerShelf in readerShelfRows:
        readerShelfID = readerShelf["id"]
        shelfName = readerShelf["name"]
        shelf_dicts["shelfsNameToReader"][shelfName] = readerShelfID

        if readerShelfID != None:
            shelf_dicts["shelfsReaderToCalibre"][readerShelfID] = {"calibreShelfID": None, "name": shelfName}

    return shelf_dicts



def get_current_profile_ID(device_main_storage, device_DB_path):

    profileLinkPath = os.path.join(device_main_storage, "system", "profiles", ".current.lnk")

    if os.path.exists(profileLinkPath):
        with open(profileLinkPath,'rb') as file:
            link = str(file.read())
            profileName = re.sub(r".*/", "", link).replace("'", "")

        with closing(sqlite.connect(device_DB_path)) as db:
            db.row_factory = lambda cursor, row: {col[0]: row[i] for i, col in enumerate(cursor.description)}
            cursor = db.cursor()
            profile_ID = cursor.execute("SELECT id FROM profiles WHERE name = '" + profileName + "'").fetchone()["id"]
    
    else:
        profile_ID = 1
        
    return profile_ID




        
        
def get_cr3hist_path(device_main_storage, device_card, storage_prefix_main, storage_prefix_card):

    profileLinkPath = os.path.join(device_main_storage, "system", "profiles", ".current.lnk")

    if os.path.exists(profileLinkPath):
        with open(profileLinkPath,'rb') as file:
            link = str(file.read())
            profileName = re.sub(r".*/", "", link).replace("'", "")

            link_split = link.split("/")
            link_storage = "/" + link_split[-5] + "/" + link_split[-4]

            if link_storage == storage_prefix_main:
                cr3hist_path = os.path.join(device_main_storage, 'system', "profiles", profileName,  "state", "cr3", ".cr3hist")

            if link_storage == storage_prefix_card:
                cr3hist_path = os.path.join(device_card, 'system', "profiles", profileName,  "state", "cr3", ".cr3hist")

    else:
        cr3hist_path = os.path.join(device_main_storage, 'system', "state", "cr3", ".cr3hist")

    return cr3hist_path

        
        
def cr3_sync_position(cr3hist_path, bookPath_folder, bookPath_file, book_filesize, calibre_book_ID, calibreAPI):
    
 
    # Create Coolreader history file if not exist
    
    if not os.path.exists(cr3hist_path):

        cr3hist_dir = os.path.split(cr3hist_path)[0]
        make_dir(cr3hist_dir)
        root = ET.Element("FictionBookMarks")
        tree = ET.ElementTree(root)
        tree.write(cr3hist_path)


    


    
    # Get position from Calibre

    calibre_position_string = calibreAPI.field_for(prefs["cr3_position_lookup_name"], calibre_book_ID)

    # Get position string from reader

    bookPath_folder = bookPath_folder.replace("''", "'") + "/"
    bookPath_file = bookPath_file.replace("''", "'")
    book_file = None
    position_bookmark = None
    tree = ET.parse(cr3hist_path)
    root = tree.getroot()

    
    for file in root:
        filename = file.find("file-info/doc-filename").text
        filepath = file.find("file-info/doc-filepath").text
        if filename == bookPath_file and filepath == bookPath_folder:
            book_file = file
            break

    if book_file != None:
        position_bookmark = book_file.find("bookmark-list/bookmark[@type='lastpos']")

    # Compare positions


    if position_bookmark != None:
        start_point = position_bookmark.find("start-point")
        position = start_point.text     
        reader_ts = position_bookmark.get('timestamp')
        reader_position_string = position + "_TIMESTAMP_" + reader_ts

    if position_bookmark != None and calibre_position_string == None:
        # Load position to calibre
        return {"status": reader_position_string}


    elif position_bookmark == None and calibre_position_string != None:
        # Send position to reader (add new file element to xml)

        calibre_ts = int(calibre_position_string.split("_TIMESTAMP_")[1])
        calibre_position_text = calibre_position_string.split("_TIMESTAMP_")[0]
        
        file = ET.SubElement(root, "file")
        fileinfo = ET.SubElement(file, "file-info")
        filename = ET.SubElement(fileinfo, "doc-filename")
        filename.text = bookPath_file
        filepath = ET.SubElement(fileinfo, "doc-filepath")
        filepath.text = bookPath_folder

        filesize = ET.SubElement(fileinfo, "doc-filesize")
        filesize.text = str(book_filesize)
        bookmark_list = ET.SubElement(file, "bookmark-list")
        bookmark = ET.SubElement(bookmark_list, "bookmark")
        bookmark.set("type", "lastpos")
        bookmark.set("timestamp", str(calibre_ts))
        start_point = ET.SubElement(bookmark, "start-point")
        start_point.text = calibre_position_text
        tree.write(cr3hist_path)

        return None

    elif position_bookmark != None and calibre_position_string != None and calibre_position_string != reader_position_string:
        # Compare position timestamps

        reader_ts = int(position_bookmark.get('timestamp'))
        calibre_ts = int(calibre_position_string.split("_TIMESTAMP_")[1])

        if calibre_ts > reader_ts:
            # Send position to reader (edit element in xml)
            calibre_position_text = calibre_position_string.split("_TIMESTAMP_")[0]
            start_point.text = calibre_position_text
            position_bookmark.set('timestamp', str(calibre_ts))
            tree.write(cr3hist_path)

            return None
            

        elif calibre_ts < reader_ts:
            # Load position to calibre
            return {"status": reader_position_string}

    return None












def pb_sync_position(db, calibreAPI, calibre_book_ID, reader_book_ID, profile_ID):
    
    calibre_position_string = calibreAPI.field_for(prefs["pb_position_lookup_name"], calibre_book_ID)

    if calibre_position_string != None:
        calibre_position = calibre_position_string.split("_TIMESTAMP_")[0]
        calibre_ts = calibre_position_string.split("_TIMESTAMP_")[1]
        calibre_ts_int = int(calibre_ts)

    cursor = db.cursor()

    reader_position_row = cursor.execute("SELECT position, position_ts FROM books_settings WHERE bookid = " + reader_book_ID + " AND profileid = " + profile_ID).fetchone()

    reader_position_string = None

    if reader_position_row and reader_position_row["position"]:
        reader_ts_int = reader_position_row["position_ts"]
        reader_ts = str(reader_ts_int)
        reader_position_string = reader_position_row["position"] + "_TIMESTAMP_" + reader_ts
        

    if reader_position_row == None and calibre_position_string:
        cursor.execute("INSERT INTO books_settings(bookid, profileid, position, position_ts) VALUES(" + reader_book_ID + ", " + profile_ID + ", '" + calibre_position + "', " + calibre_ts + ")")
        db.commit()
        return None

    elif reader_position_row and calibre_position_string == None:
        return {"status": reader_position_string}

    elif reader_position_row and calibre_position_string and calibre_position_string != reader_position_string:

        if reader_position_row["position"] == None or calibre_ts_int > reader_ts_int:
            cursor.execute("UPDATE books_settings SET position = '" + calibre_position + "', position_ts = " + calibre_ts + " WHERE bookid = " + reader_book_ID + " AND profileid = " + profile_ID)
            db.commit()
            return None
        
        elif calibre_ts_int < reader_ts_int:
            return {"status": reader_position_string}
        


def make_dir(path):
    try:
        os.makedirs(path, exist_ok=True)  # Python>3.2
    except TypeError:
        try:
            os.makedirs(path)
        except OSError as exc: # Python >2.5
            if exc.errno == errno.EEXIST and os.path.isdir(path):
                pass
            else: raise



