#!/usr/bin/env python

from __future__ import absolute_import, division, print_function, unicode_literals

__license__   = 'GPL v3'
__copyright__ = '2019-2023, Thiago Oliveira <thiago.eec@gmail.com>'
__docformat__ = 'restructuredtext en'

# Load translation files (.mo) on the folder 'translations'
load_translations()

# Standard libraries
import os
import os.path
import zipfile
import base64
import sys
from functools import partial

# PyQt libraries
try:
    from qt.core import Qt, QtCore, QMenu, QIcon, QPixmap, QVBoxLayout, QDialogButtonBox, QApplication
except ImportError:
    from PyQt5.Qt import Qt, QMenu, QIcon, QPixmap, QVBoxLayout, QDialogButtonBox, QApplication
    from PyQt5 import QtCore

# Get PyQt version
Qt_version = int(QtCore.PYQT_VERSION_STR[0])

# Calibre libraries
from calibre.gui2.actions import InterfaceAction
from calibre.gui2.actions import menu_action_unique_name
from calibre.gui2.keyboard import finalize
from calibre.utils.config import config_dir
from calibre_plugins.Skoob_Sync.config import prefs, get_icon
from calibre_plugins.Skoob_Sync.main import SkoobSyncTools
from calibre_plugins.Skoob_Sync.__init__ import PLUGIN_VERSION, PLUGIN_NAME


def is_dark_theme():
    try:
        return QApplication.instance().is_dark_theme
    except:
        return False


class InterfacePlugin(InterfaceAction):

    name = PLUGIN_NAME

    # Declare the main action associated with this plugin
    # The keyboard shortcut can be None if you dont want to use a keyboard
    # shortcut. Remember that currently calibre has no central management for
    # keyboard shortcuts, so try to use an unusual/unused shortcut.

    action_spec = (PLUGIN_NAME, None, _('Syncs calibre library with Skoob'), 'Ctrl+S')

    def genesis(self):
        # This method is called once per plugin, do initial setup here

        # Set the icon for this interface action
        # The get_icons function is a builtin function defined for all your
        # plugin code. It loads icons from the plugin zip file. It returns
        # QIcon objects, if you want the actual data, use the analogous
        # get_resources builtin function.
        #
        # Note that if you are loading more than one icon, for performance, you
        # should pass a list of names to get_icons. In this case, get_icons
        # will return a dictionary mapping names to QIcons. Names that
        # are not found in the zip file will result in null QIcons.

        self.key_themes_idx = self.check_theme()

        self.main_button_icon = get_icon('icon.png')

        # Extract password and user icons based on theme chosen. Those cannot be extracted with get_icon,
        # since we need the image file, not a QIcon. That is because we are using them on a QLabel
        self.extract_resources(cookie=False)

        # Menu creation
        # Only the toolbar menu is created here. All the others are created after the main GUI initialization
        # This allows us to change menus on the fly
        self.tools = SkoobSyncTools(self.gui)
        self.m = QMenu(self.gui)
        self.qaction.setMenu(self.m)
        self.qaction.setIcon(self.main_button_icon)

        # This action is configurable by the user. It is actioned by clicking on the main plugin button
        self.main_button_action = self.qaction
        self.main_button_action.triggered.connect(self.set_main_action)

        # First build of menus
        self.rebuild_menus()

        # Set up a listener to catch a system theme change
        try:
            QApplication.instance().palette_changed.connect(self.rebuild_menus)
        except:
            pass

    def initialization_complete(self):
        # This method is called once per action, after the initialization of the main GUI
        # This is right place to create changing menus
        self.m.aboutToShow.connect(self.about_to_show_menu)

    def check_theme(self):
        # Check for calibre theme
        if is_dark_theme():
            try:  # Legacy support (versions before 0.4.4)
                prefs['themes_idx']
                if prefs['themes_idx'] in [0, 1, 2]:
                    prefs.set('dark_themes_idx', 0)
                    key_themes_idx = 4
                else:
                    prefs.set('dark_themes_idx', 1)
                    key_themes_idx = 5
                prefs.commit()
                del prefs['themes_idx'], prefs['themes']
            except KeyError:
                key_themes_idx = prefs['dark_themes_idx'] + 4
        else:
            try:  # Legacy support (versions before 0.4.4)
                prefs['themes_idx']
                if prefs['themes_idx'] in [0, 1, 2]:
                    prefs.set('light_themes_idx', prefs['themes_idx'])
                    key_themes_idx = prefs['themes_idx'] + 1
                else:
                    prefs.set('light_themes_idx', 2)
                    key_themes_idx = 5
                prefs.commit()
                del prefs['themes_idx'], prefs['themes']
            except KeyError:
                key_themes_idx = prefs['light_themes_idx'] + 1
        return key_themes_idx

    def do_config(self):
        from calibre.gui2.widgets2 import Dialog
        from calibre_plugins.Skoob_Sync.config import ConfigWidget

        tool = self

        class ConfigDialog(Dialog):

            def __init__(self, gui):
                self.gui = gui
                Dialog.__init__(self, _('Options'), 'plugin-skoob-sync-config-dialog', parent=tool.gui)
                self.setWindowIcon(get_icon('icon.png'))

            def setup_ui(self):
                try:
                    # Show 'What's this' symbol. This must not be on the __init__, or
                    # it will cause the config widget to move everytime it is opened
                    Dialog.setWindowFlag(self, Qt.WindowContextHelpButtonHint, True)
                except AttributeError:
                    pass
                self.box = QVBoxLayout(self)
                self.widget = ConfigWidget(self, self.gui)
                self.box.addWidget(self.widget)
                self.button = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
                self.box.addWidget(self.button)
                self.button.accepted.connect(self.accept)
                self.button.rejected.connect(self.reject)

            def accept(self):
                self.widget.save_settings()
                Dialog.accept(self)

        d = ConfigDialog(self)
        d.exec_()

    def show_configuration(self):
        restart_message = _("Calibre must be restarted before the plugin can be configured.")
        # Check if a restart is needed. If the restart is needed, but the user does not
        # trigger it, the result is true and we do not do the configuration.
        if self.check_if_restart_needed(restart_message=restart_message):
            return

        self.do_config()
        # self.interface_action_base_plugin.do_user_config(self.gui)

        finalize(self.gui.keyboard.shortcuts)
        self.rebuild_menus()

        restart_message = _("New custom columns have been created. You will need"
                            "\nto restart calibre for this change to be applied."
                            )
        self.check_if_restart_needed(restart_message=restart_message)

    def check_if_restart_needed(self, restart_message=None, restart_needed=False):
        if self.gui.must_restart_before_config or restart_needed:
            if restart_message is None:
                restart_message = _("Calibre must be restarted before the plugin can be configured.")
            from calibre.gui2 import show_restart_warning
            do_restart = show_restart_warning(restart_message)
            if do_restart:
                self.gui.quit(restart=True)
            else:
                return True
        return False

    def apply_settings(self):
        from calibre_plugins.Skoob_Sync.config import prefs
        # In an actual non trivial plugin, you would probably need to
        # do something based on the settings in prefs
        prefs

    def about_to_show_menu(self):
        finalize(self.gui.keyboard.shortcuts)
        self.rebuild_menus()

        # Get user credentials
        self.key_user = prefs['user']
        if sys.version_info[0] < 3:
            self.key_password = prefs['password'].decode('base64')
        else:
            self.key_password = base64.b64decode(prefs['password'])

        # Check if credentials are filled. If not, disable menu itens.
        if all(hasattr(self, attr) for attr in ['download_from_skoob', 'upload_to_skoob', 'add_to_skoob',
                                                'remove_from_skoob', 'add_to_shelf', 'mark_as', 'add_to_reading_goal',
                                                'current_year_statistics', 'get_read_date', 'import_library']):
            if self.key_user in ('user', '') or self.key_password in ('password', '', b''):
                _enabled = False
            else:
                _enabled = True
            self.download_from_skoob.setEnabled(_enabled)
            self.upload_to_skoob.setEnabled(_enabled)
            self.add_to_skoob.setEnabled(_enabled)
            self.remove_from_skoob.setEnabled(_enabled)
            self.add_to_shelf.setEnabled(_enabled)
            self.mark_as.setEnabled(_enabled)
            self.add_to_reading_goal.setEnabled(_enabled)
            self.current_year_statistics.setEnabled(_enabled)
            self.get_read_date.setEnabled(_enabled)
            self.import_library.setEnabled(_enabled)

    def rebuild_menus(self):
        self.extract_resources()

        # Check for calibre theme
        # Get theme setting again, since this can change during the use of the plugin
        if is_dark_theme():
            self.key_themes_idx = prefs['dark_themes_idx'] + 4
        else:
            self.key_themes_idx = prefs['light_themes_idx'] + 1

        # Set main button icon based on theme chosen
        self.main_button_icon = get_icon('icon.png')
        self.main_button_action.setIcon(self.main_button_icon)

        # Clear all menu actions. This way we can rebuild them, with different icons.
        self.m.clear()

        # Main menu itens - Created here, so icons may change on the fly.
        # Before, they were created at initialization_complete().
        cm = partial(self.create_menu_action_unique)

        self.has_skoob_id_item = cm(self.m, _('&List books with Skoob ID'),
           image='skoob.png',
           tooltip=_('Filter book list to show only books with a Skoob ID'),
           triggered=self.tools.has_skoob_id,
           unique_name='SkoobSync_List')
        self.m.addSeparator()

        # Main actions
        self.download_from_skoob = cm(self.m, _('&Download from Skoob'),
           image='download_from_skoob.png',
           tooltip=_('Download book information from Skoob'),
           triggered=partial(self.tools.update_status, sync_option='from_skoob', shelf=None),
           unique_name='SkoobSync_Download')
        self.upload_to_skoob = cm(self.m, _('&Upload to Skoob'),
           image='upload_to_skoob.png',
           tooltip=_('Upload book information to Skoob'),
           triggered=partial(self.tools.update_status, sync_option='to_skoob', shelf=None),
           unique_name='SkoobSync_Upload')
        self.m.addSeparator()

        # Add/remove books
        self.add_to_skoob = cm(self.m, _('&Add book'),
           image='add_to_skoob.png',
           tooltip=_('Add book to Skoob, on Reading shelf'),
           triggered=partial(self.tools.update_status, sync_option='add_to_skoob', shelf=None),
           unique_name='SkoobSync_Add')
        self.remove_from_skoob = cm(self.m, _('&Remove book'),
           image='remove_from_skoob.png',
           tooltip=_('Remove book from your shelves on Skoob'),
           triggered=partial(self.tools.update_status, sync_option='remove_from_skoob', shelf=None),
           unique_name='SkoobSync_Remove')
        self.m.addSeparator()

        # Add to shelf
        self.sm = QMenu()
        self.shelf_read = cm(self.sm, _('&Read'),
           image='read.png',
           tooltip=_('Add book to Read shelf'),
           triggered=partial(self.tools.update_status, sync_option='add_to_skoob', shelf=1),
           unique_name='SkoobSync_Read')
        self.shelf_reading = cm(self.sm, _('R&eading'),
           image='reading.png',
           tooltip=_('Add book to Reading shelf'),
           triggered=partial(self.tools.update_status, sync_option='add_to_skoob', shelf=2),
           unique_name='SkoobSync_Reading')
        self.shelf_to_read = cm(self.sm, _('&Want to read'),
           image='to_read.png',
           tooltip=_('Add book to Want to read shelf'),
           triggered=partial(self.tools.update_status, sync_option='add_to_skoob', shelf=3),
           unique_name='SkoobSync_Want')
        self.shelf_rereading = cm(self.sm, _('Rerea&ding'),
           image='rereading.png',
           tooltip=_('Add book to Rereading shelf'),
           triggered=partial(self.tools.update_status, sync_option='add_to_skoob', shelf=4),
           unique_name='SkoobSync_Rereading')
        self.shelf_abandoned = cm(self.sm, _('&Abandoned'),
           image='abandoned.png',
           tooltip=_('Add book to Abandoned shelf'),
           triggered=partial(self.tools.update_status, sync_option='add_to_skoob', shelf=5),
           unique_name='SkoobSync_Abandoned')

        self.add_to_shelf = cm(self.m, _('Add to &shelf'),
                               image='shelf.png',
                               shortcut=False,
                               tooltip=_('Add book to a specific shelf'))
        self.add_to_shelf.setMenu(self.sm)
        self.m.addSeparator()

        # Mark book
        self.sm2 = QMenu()
        self.favorite = cm(self.sm2, _('&Favorite'),
                             image='images/favorite.png',
                             tooltip=_('Mark book as favorite'),
                             triggered=partial(self.tools.update_status, sync_option='add_to_skoob', shelf=8),
                             unique_name='SkoobSync_Favorite')

        self.wanted = cm(self.sm2, _('&Wanted'),
                           image='images/wanted.png',
                           tooltip=_('Mark book as wanted'),
                           triggered=partial(self.tools.update_status, sync_option='add_to_skoob', shelf=9),
                           unique_name='SkoobSync_Wanted')

        self.owned = cm(self.sm2, _('&Owned'),
                           image='images/owned.png',
                           tooltip=_('Mark book as owned'),
                           triggered=partial(self.tools.update_status, sync_option='add_to_skoob', shelf=6),
                           unique_name='SkoobSync_Owned')

        self.lent = cm(self.sm2, _('&Lent'),
                           image='images/lent.png',
                           tooltip=_('Mark book as lent'),
                           triggered=partial(self.tools.update_status, sync_option='add_to_skoob', shelf=11),
                           unique_name='SkoobSync_Lent')

        self.exchange = cm(self.sm2, _('&Exchange'),
                           image='images/exchange.png',
                           tooltip=_('Mark book for exchange'),
                           triggered=partial(self.tools.update_status, sync_option='add_to_skoob', shelf=10),
                           unique_name='SkoobSync_Exchange')

        self.mark_as = cm(self.m, _('Mar&k book'),
                               image='mark.png',
                               shortcut=False,
                               tooltip=_('Choose how to mark this book'))
        self.mark_as.setMenu(self.sm2)
        self.m.addSeparator()

        # Add to Reading Goal
        self.add_to_reading_goal = cm(self.m, _('Add to Reading &Goal'),
           image='add_to_goal.png',
           tooltip=_('Add book to your current year reading goal'),
           triggered=partial(self.tools.update_status, sync_option='add_to_goal', shelf=None),
           unique_name='SkoobSync_Goal')

        # Reading Goal statistics
        self.current_year_statistics = cm(self.m, _('Reading Goal sta&tistics'),
                                      image='stats.png',
                                      tooltip=_('Show your Reading Goal statistics'),
                                      triggered=self.tools.reading_goal_main,
                                      unique_name='SkoobSync_Statistics')
        self.m.addSeparator()

        # Get read date
        self.get_read_date = cm(self.m, _('G&et Read date'),
           image='read_date.png',
           tooltip=_('Get the date you finished reading the book'),
           triggered=partial(self.tools.update_status, sync_option='get_read_date', shelf=None),
           unique_name='SkoobSync_Date')

        # Import Skoob Library
        self.import_library = cm(self.m, _('&Import Skoob library'),
           image='import.png',
           tooltip=_('Create new entries on your calibre library based on your Skoob library'),
           triggered=self.tools.skoob_library,
           unique_name='SkoobSync_Library')
        self.m.addSeparator()

        # Config
        self.config_item = cm(self.m, _('&Customize plugin'),
           image='config.png',
           tooltip=_('Open configuration dialog'),
           shortcut=False,
           triggered=self.show_configuration)

        self.gui.keyboard.finalize()

    def set_main_action(self):
        # Define main action based on the user choice
        self.key_main_button = prefs['main_action_idx']
        if self.key_main_button == 0:
            sync_option = 'from_skoob'
        else:
            sync_option = 'to_skoob'
        self.tools.update_status(sync_option=sync_option, shelf=None)

    def extract_resources(self, cookie=False):
        # Extract resource files to use internally

        # Check for calibre theme
        if is_dark_theme():
            self.key_themes_idx = prefs['dark_themes_idx'] + 4
        else:
            self.key_themes_idx = prefs['light_themes_idx'] + 1
        plugin_zip = os.path.join(config_dir, 'plugins', PLUGIN_NAME + '.zip')
        plugin_dir = os.path.join(config_dir, 'plugins', PLUGIN_NAME)
        with zipfile.ZipFile(plugin_zip, 'r') as zf:
            if Qt_version >= 6:
                zf.extract('images/theme' + str(self.key_themes_idx) + '/user.png', plugin_dir)
                zf.extract('images/theme' + str(self.key_themes_idx) + '/password.png', plugin_dir)
            else:
                zf.extract('images/theme' + str(self.key_themes_idx) + '/user2.png', plugin_dir)
                zf.extract('images/theme' + str(self.key_themes_idx) + '/password2.png', plugin_dir)
            zf.extract('images/theme' + str(self.key_themes_idx) + '/add_column.png', plugin_dir)
            zf.extract('images/theme' + str(self.key_themes_idx) + '/help.png', plugin_dir)
            zf.extract('images/loading.gif', plugin_dir)
            if cookie:
                zf.extract('cookie/cookie.txt', plugin_dir)

    def create_menu_action_unique(ia, parent_menu, menu_text, image=None, tooltip=None,
                                  shortcut=None, triggered=None, is_checked=None, shortcut_name=None,
                                  unique_name=None):
        '''
        Create a menu action with the specified criteria and action, using the new
        InterfaceAction.create_menu_action() function which ensures that regardless of
        whether a shortcut is specified it will appear in Preferences->Keyboard
        '''
        orig_shortcut = shortcut
        kb = ia.gui.keyboard
        if unique_name is None:
            unique_name = menu_text
        if not shortcut == False:
            full_unique_name = menu_action_unique_name(ia, unique_name)
            if full_unique_name in kb.shortcuts:
                shortcut = False
            else:
                if shortcut is not None and not shortcut == False:
                    if len(shortcut) == 0:
                        shortcut = None
                    else:
                        shortcut = _(shortcut)

        if shortcut_name is None:
            shortcut_name = menu_text.replace('&', '')

        ac = ia.create_menu_action(parent_menu, unique_name, menu_text, icon=None, shortcut=shortcut,
                                   description=tooltip, triggered=triggered, shortcut_name=shortcut_name)
        if shortcut == False and not orig_shortcut == False:
            if ac.calibre_shortcut_unique_name in ia.gui.keyboard.shortcuts:
                kb.replace_action(ac.calibre_shortcut_unique_name, ac)
        if image:
            ac.setIcon(get_icon(image))
        if is_checked is not None:
            ac.setCheckable(True)
            if is_checked:
                ac.setChecked(True)
        return ac
