#!/usr/bin/env python

from __future__ import absolute_import, division, print_function, unicode_literals
from six.moves import map

__license__   = 'GPL v3'
__copyright__ = '2019-2023, Thiago Oliveira <thiago.eec@gmail.com>'
__docformat__ = 'restructuredtext en'

# Load translation files (.mo) on the folder 'translations'
load_translations()

# Standard libraries
import os
import os.path
import re
import six.moves.http_cookiejar
import six.moves.urllib.request
import six.moves.urllib.parse
import six.moves.urllib.error
import datetime
import time
import threading
import json
import base64
import sys
from timeit import default_timer as timer
from lxml.html import fromstring, tostring

# PyQt libraries
try:
    from qt.webengine import QWebEngineView
except ImportError:
    pass
try:
    from qt.core import (Qt, QApplication, QTreeWidget, QTreeWidgetItem, QVBoxLayout, QHBoxLayout, QProgressBar, QLabel,
                         QtCore, QtGui, QTreeWidgetItemIterator, QAbstractItemView, QDialogButtonBox, QComboBox,
                         QGroupBox, QSize)
except ImportError:
    from PyQt5.Qt import (Qt, QApplication, QTreeWidget, QTreeWidgetItem, QVBoxLayout, QHBoxLayout, QProgressBar, QSize,
                          QTreeWidgetItemIterator, QAbstractItemView, QDialogButtonBox, QComboBox, QGroupBox, QLabel)
    from PyQt5 import QtCore, QtGui

# Get PyQt version
Qt_version = int(QtCore.PYQT_VERSION_STR[0])

# Calibre libraries
from calibre.utils.config import config_dir
from calibre_plugins.Skoob_Sync.config import prefs, get_icon
from calibre.utils.cleantext import clean_ascii_chars
from calibre.gui2 import error_dialog, info_dialog, warning_dialog, question_dialog
from calibre.gui2.widgets2 import Dialog
from calibre.library.comments import sanitize_comments_html
from calibre.ebooks.metadata import MetaInformation
from calibre_plugins.Skoob_Sync.__init__ import PLUGIN_VERSION, PLUGIN_NAME

try:
    from calibre.gui2.actions.show_quickview import get_quickview_action_plugin
    QUICK_VIEW_SUPPORT = True
except:
    QUICK_VIEW_SUPPORT = False


# Get a random user agent from calibre. This is used on Skoob access.
def random_ua():
    try:
        from calibre import random_user_agent
        try:
            hdr = {'User-Agent': random_user_agent(allow_ie=False)}
            return hdr
        except TypeError:
            hdr = {'User-Agent': random_user_agent()}
            return hdr
    except ImportError:
        hdr = {'User-Agent': 'Mozilla/5.0 (Windows NT .1; Win64; x64)'}
        return hdr


# Check for dark theme
def is_dark_theme():
    try:
        return QApplication.instance().is_dark_theme
    except:
        return False


# Globals
goal_is_set = False

# Starting credentials. Used to check if it is changed.
KEY_USER = prefs['user']
if sys.version_info[0] < 3:
    KEY_PASSWORD = prefs['password'].decode('base64')
else:
    KEY_PASSWORD = base64.b64decode(prefs['password'])


def do_config(self, current_tab=0):
    from calibre.gui2.widgets2 import Dialog
    from calibre_plugins.Skoob_Sync.config import ConfigWidget

    class ConfigDialog(Dialog):

        def __init__(self, gui, tab):
            self.tab = tab
            self.gui = gui
            Dialog.__init__(self, _('Options'), 'plugin-skoob-sync-config-dialog')
            self.setWindowIcon(get_icon('icon.png'))

        def setup_ui(self):
            try:
                # Show 'What's this' symbol. This must not be on the __init__, or
                # it will cause the config widget to move everytime it is opened
                Dialog.setWindowFlag(self, Qt.WindowContextHelpButtonHint, True)
            except AttributeError:
                pass
            self.box = QVBoxLayout(self)
            self.widget = ConfigWidget(self, self.gui, self.tab)
            self.box.addWidget(self.widget)
            self.button = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
            self.box.addWidget(self.button)
            self.button.accepted.connect(self.accept)
            self.button.rejected.connect(self.reject)

        def accept(self):
            self.widget.save_settings()
            Dialog.accept(self)

    d = ConfigDialog(self, current_tab)
    d.exec_()


def show_configuration(self, tab=0):
    restart_message = _("Calibre must be restarted before the plugin can be configured.")
    # Check if a restart is needed. If the restart is needed, but the user does not
    # trigger it, the result is true and we do not do the configuration.
    if check_if_restart_needed(self, restart_message=restart_message):
        return

    do_config(self, tab)
    restart_message = _("New custom columns have been created. You will need"
                        "\nto restart calibre for this change to be applied."
                        )
    check_if_restart_needed(self, restart_message=restart_message)


def check_if_restart_needed(self, restart_message=None, restart_needed=False):
    if self.gui.must_restart_before_config or restart_needed:
        if restart_message is None:
            restart_message = _("Calibre must be restarted before the plugin can be configured.")
        from calibre.gui2 import show_restart_warning
        do_restart = show_restart_warning(restart_message)
        if do_restart:
            self.gui.quit(restart=True)
        else:
            return True
    return False


# Open urls, and return the result
def open_url(url, payload=None):
    success = False
    bad_connection = False
    # Use urllib to encode the payload
    if payload:
        data = six.moves.urllib.parse.urlencode(payload).encode()
    else:
        data = None

    # Build our Request object (supplying 'data' makes it a POST)
    req = six.moves.urllib.request.Request(url, data, headers=random_ua())

    try:
        req_open = six.moves.urllib.request.urlopen(req)
        success = True

    except six.moves.urllib.error.URLError:
        bad_connection = True

    return success, bad_connection


class SkoobSyncTools:

    def __init__(self, gui):
        self.gui = gui

    def has_skoob_id(self):
        """ Show books with Skoob Books IDs """

        self.gui.search.setEditText('identifiers:"=skoob:"')
        self.gui.search.do_search()

    def update_status(self, sync_option, shelf):
        """Sync book metatada with Skoob."""

        self.shelf = shelf
        self.sync_option = sync_option

        # Get preferences
        self.prefs = prefs
        if sys.version_info[0] < 3:
            self.key_password = prefs['password'].decode('base64')
        else:
            self.key_password = base64.b64decode(prefs['password'])

        # Check if credentials are set
        if self.prefs['user'] in ('user', '') or self.key_password in ('password', '', b''):
            if question_dialog(self.gui, _('Invalid Credentials'), _('You must first setup your Skoob account') + '. '
                               + _('\nDo you want to configure the plugin now?'), show_copy_button=False):
                return show_configuration(self, tab=0)
            else:
                return

        # Check if no sync option is checked
        sync = [self.prefs['sync_reading_progress'], self.prefs['sync_review'], self.prefs['sync_rating'],
                self.prefs['sync_page_count']]
        if not self.any(sync):
            if question_dialog(self.gui, _('Nothing to sync'), _('No sync actions selected') + '. '
                               + _('\nDo you want to configure the plugin now?'), show_copy_button=False):
                return show_configuration(self, tab=0)
            else:
                return

        # Check if the necessary custom columns are set
        col = None
        if self.prefs['sync_reading_progress'] and self.prefs['sync_read_books']:
            if len(self.prefs['read_books_column']) == 0: col = _('read books custom column')
        if self.prefs['sync_page_count']:
            if len(self.prefs['page_count_column']) == 0: col = _('page count custom column')
        if self.prefs['sync_reading_progress']:
            if len(self.prefs['reading_progress_column']) == 0: col = _('reading progress custom column')
        if col:
            if question_dialog(self.gui, _('Configuration needed'),
                               _('You must choose the {0}').format(col) + '. '
                               + _('\nDo you want to configure the plugin now?'), show_copy_button=False):
                return show_configuration(self, tab=1)
            else:
                return

        # Get currently selected books
        rows = self.gui.library_view.selectionModel().selectedRows()

        # Check if selection is empty
        if not rows or len(rows) == 0:
            return error_dialog(self.gui, _('Empty selection'),
                                _('No books selected'), show_copy_button=False, show=True)

        # Check if selected books have a skoob id
        is_selection_valid = self.are_selected_books_linked()
        if is_selection_valid is False:
            if len(rows) == 1:
                return error_dialog(self.gui, _('Invalid selection'),
                                    _('Selected book does not have a Skoob ID'), show_copy_button=False, show=True)
            else:
                return error_dialog(self.gui, _('Invalid selection'),
                                    _('All selected books must have a Skoob ID'), show_copy_button=False, show=True)

        # Map the rows to book ids
        ids = list(map(self.gui.library_view.model().id, rows))
        db = self.gui.current_db.new_api

        # Create list of altered books
        self.books_not_updated = []
        self.books_partially_updated = []
        self.books_updated = []
        self.books_not_in_shelf = []
        self.bad_skoob_id = []
        self.bad_connection = False

        # Check if too many books are selected at once
        if (self.prefs['sync_tags'] and len(ids) <= 20) or (not self.prefs['sync_tags'] and len(ids) <= 50):
            if len(ids) > 20:
                if question_dialog(self.gui, _('Caution'),
                                   _('You selected %d books. Selecting too many books may cause calibre to crash. '
                                     'Are you sure you want to continue?') % len(ids), show_copy_button=False):
                    pass
                else:
                    return

            start = timer()

            # Login to Skoob
            login_status = self.login()
            if login_status is False or login_status is None:
                return self.gui.status_bar.clearMessage()
            self.gui.status_bar.clearMessage()

            # Choose a message for the user based on the current task.
            # This must not be inside the threads, or we will get an error (parent element not in the same thread)
            message_dict = {
                'from_skoob': _("Downloading from Skoob..."),
                'to_skoob': _("Uploading to Skoob..."),
                'add_to_skoob': [_("Adding to Skoob..."), _("Adding to shelf..."), _("Marking book...")],
                'remove_from_skoob': _("Removing from Skoob..."),
                'add_to_goal': _("Adding to Reading Goal..."),
                'get_read_date': _("Updating Read date...")
            }

            if self.sync_option == 'add_to_skoob':
                if self.shelf is None:
                    message = message_dict[self.sync_option][0]
                elif self.shelf < 6:
                    message = message_dict[self.sync_option][1]
                else:
                    message = message_dict[self.sync_option][2]
            else:
                message = message_dict[self.sync_option]
            self.gui.status_bar.showMessage(message)

            # Check each book on its own thread
            aux_list = []
            for book_id in ids:
                aux_list.append((book_id,))
            self.run_parallel_in_threads(self.update_routine, aux_list)
            self.gui.status_bar.clearMessage()

            end = timer()
            time = end - start
            print(datetime.timedelta(seconds=time))

            # Refresh the GUI, so the Book Info panel shows the new info
            self.gui.library_view.model().refresh_ids(ids)
            self.gui.library_view.model().refresh_ids(ids, current_row=self.gui.library_view.currentIndex().row())

            # Final message to the user, informing how many books were updated
            if self.bad_connection:
                error_dialog(self.gui, _('No internet connection'), _('Seems like you have no internet access'),
                             show_copy_button=False, show=True)

            if len(self.books_not_updated) > 0:
                if len(self.books_updated) > 0:
                    warning_dialog(self.gui, _('Update status'),
                                   _('Updated %d book(s). ') % len(self.books_updated) +
                                   _('No updates for %d book(s).') % len(self.books_not_updated),
                                   show_copy_button=False, show=True)
                else:
                    warning_dialog(self.gui, _('Nothing to update'), _('No updates for %d book(s)')
                                   % len(self.books_not_updated), show_copy_button=False, show=True)
            else:
                if len(self.books_updated) > 0:
                    info_dialog(self.gui, _('Update status'), _('Updated %d book(s)')
                                % len(self.books_updated), show_copy_button=False, show=True)

            if len(self.books_partially_updated) > 0:
                book_list = []
                for book_id in self.books_partially_updated:
                    mi = db.get_metadata(book_id)
                    book_list.append(mi.title)
                book_list_str = ', '.join(book_list)
                error_dialog(self.gui, _('Partially update'), _('%d book(s) were partially updated')
                             % len(self.books_partially_updated) + ':\n' + book_list_str,
                             show_copy_button=False, show=True)

            if len(self.books_not_in_shelf) > 0:
                book_list = []
                for book_id in self.books_not_in_shelf:
                    mi = db.get_metadata(book_id)
                    book_list.append(mi.title)
                book_list_str = ', '.join(book_list)
                error_dialog(self.gui, _('Not in shelf'), _('%d book(s) not in any shelf')
                             % len(self.books_not_in_shelf) + ':\n' + book_list_str, show_copy_button=False, show=True)

            if len(self.bad_skoob_id) > 0:
                book_list = []
                for book_id in self.bad_skoob_id:
                    mi = db.get_metadata(book_id)
                    book_list.append(mi.title)
                book_list_str = ', '.join(book_list)
                if sync_option in ('add_to_skoob', 'add_go_toal'):
                    error_dialog(self.gui, _('Not added'), _('%d book(s) with unknown Skoob IDs')
                                 % len(self.bad_skoob_id) + ':\n' + book_list_str,
                                 show_copy_button=False, show=True)
                else:
                    error_dialog(self.gui, _('Not removed'), _('%d book(s) are not on your shelves '
                                 'or have invalid IDs') % len(self.bad_skoob_id) + ':\n' + book_list_str,
                                 show_copy_button=False, show=True)

        else:
            # To avoid overloading the server, we must update no more than 50 books at a time. This is just a guess.
            # This plugin does not use Skoob API, since it has not been made public yet, according to their FAQ.
            if self.prefs['sync_tags'] and len(ids) > 20:
                msg = _('%d books selected. Since you are syncing user tags, try to select 20 books at most.') % len(ids)
            else:
                msg = _('%d books selected. Try to select 50 books at most.') % len(ids)
            return error_dialog(self.gui, _('Too many books'), msg, show_copy_button=False, show=True)

    def run_parallel_in_threads(self, target, args_list):
        """ Utility - spawn a thread to execute target for each args """
        # https://stackoverflow.com/questions/3490173/how-can-i-speed-up-fetching-pages-with-urllib2-in-python/3491455

        threads = [threading.Thread(target=target, args=args) for args in args_list]

        count = 0

        for t in threads:
            count += 1
            t.start()
            # Don't send all requests at once
            list_size = len(args_list)
            if list_size > 30:
                if count == 20:
                    time.sleep(1)
                elif count == 30:
                    time.sleep(1)
                elif count == 40:
                    time.sleep(2)
            elif list_size > 20:
                if count == 20:
                    time.sleep(1)
            time.sleep(0.1)
            try:
                QApplication.processEvents(QtCore.QEventLoop.ProcessEventsFlag.ExcludeUserInputEvents)
            except AttributeError:
                pass
        for t in threads:
            t.join()

    def update_routine(self, book_id):
        """ This method does all the Skoob syncing activity """
        # Get the current metadata for this book from the db
        db = self.gui.current_db.new_api
        mi = db.get_metadata(book_id)
        # Get skoob_id from calibre
        identifiers = mi.get_identifiers()
        skoob_id = identifiers['skoob']
        is_skoob_id = re.search('(ed)(\\d+)', skoob_id.lower())
        if is_skoob_id is None:
            return self.bad_skoob_id.append(book_id)
        else:
            skoob_id = re.search('(ed)(\\d+)', skoob_id.lower()).groups(0)[1]

        # Download FROM Skoob
        if self.sync_option == 'from_skoob':
            self.download_from_skoob(skoob_id, book_id, db, mi)

        # Upload TO Skoob
        if self.sync_option == 'to_skoob':
            self.upload_to_skoob(skoob_id, book_id, db, mi)

        # Add to Skoob / Add to shelf
        if self.sync_option == 'add_to_skoob':
            self.bad_connection, valid_id, to_update = self.add_to_skoob(skoob_id, book_id, mi)
            if self.bad_connection:
                return self.bad_connection
            # If update_added is checked, it will return True. This will prevent duplicating final message.
            if to_update:
                self.upload_to_skoob(skoob_id, book_id, db, mi)
            else:
                # If the Skoob ID is invalid, then abort
                if valid_id:
                    # Set the read column according to reading progress
                    read_percentage = mi.metadata_for_field(self.prefs['reading_progress_column'])['#value#']
                    if self.prefs['sync_read_books']:
                        if len(self.prefs['read_books_column']) > 0:
                            if read_percentage is None:
                                pass
                                # db.new_api.set_field(self.prefs['read_books_column'], {book_id: ''})
                            else:
                                if int(read_percentage) >= 100:
                                    db.new_api.set_field(self.prefs['read_books_column'], {book_id: True})
                                else:
                                    db.new_api.set_field(self.prefs['read_books_column'], {book_id: ''})
                else:
                    return

        # Remove from Skoob
        if self.sync_option == 'remove_from_skoob':
            self.bad_connection, valid_id = self.remove_from_skoob(skoob_id, book_id, mi)
            if self.bad_connection:
                return self.bad_connection
            # If the Skoob ID is invalid, then abort
            if valid_id:
                # Set the read column according to reading progress
                read_percentage = mi.metadata_for_field(self.prefs['reading_progress_column'])['#value#']
                if self.prefs['sync_read_books']:
                    if len(self.prefs['read_books_column']) > 0:
                        if read_percentage is None:
                            pass
                            # db.new_api.set_field(self.prefs['read_books_column'], {book_id: ''})
                        else:
                            if int(read_percentage) >= 100:
                                db.new_api.set_field(self.prefs['read_books_column'], {book_id: True})
                            else:
                                db.new_api.set_field(self.prefs['read_books_column'], {book_id: ''})
            else:
                return

        # Add to Reading Goal
        if self.sync_option == 'add_to_goal':
            self.add_to_goal(skoob_id, book_id, mi)

        # Get Read Date
        if self.sync_option == 'get_read_date':
            self.get_read_date(skoob_id, book_id, db, mi)

    def download_from_skoob(self, skoob_id, book_id, db, mi):
        # Download FROM Skoob
        try:
            details = self.get_details(skoob_id)
            if details is False:
                return self.books_not_in_shelf.append(book_id)
            elif details is None:
                self.bad_connection = True
                return self.bad_connection
            status = details[0]  # Get reading progress and status date
            review = details[1]  # Get review title and text
            rating = details[2]  # Get rating
            page_count = details[3]  # Get the page count
            shelf = details[4]  # Get the book shelf
            tags = details[5]  # Get user tags
            physical = details[6]  # Get book media type
            result = [status[0], status[1], review[0], review[1], rating, page_count, shelf, tags, physical]

            if self.any(result):

                # -- Set Reading Progress --
                if self.prefs['sync_reading_progress']:
                    status_date = status[0]
                    status_progress = status[1]

                    # Fix wrong values (usually due to updates in the book page on Skoob)
                    if status_progress:
                        if int(status_progress) > 100:
                            status_progress = '100'

                    # Set the read column according to reading progress
                    if self.prefs['sync_read_books']:
                        if len(self.prefs['read_books_column']) > 0:
                            if status_progress is not None:
                                if status_progress == '100' or shelf == '1':
                                    # The book might not be 100% because of changes on the page count on Skoob
                                    db.new_api.set_field(self.prefs['read_books_column'], {book_id: True})
                                    status_progress = '100'
                                else:
                                    db.new_api.set_field(self.prefs['read_books_column'], {book_id: ''})
                            else:
                                # The book might not be 100% because of changes on the page count on Skoob
                                if shelf == '1':
                                    db.new_api.set_field(self.prefs['read_books_column'], {book_id: True})
                                    status_progress = '100'

                    # Set reading percentage
                    if len(self.prefs['status_date_column']) > 0 and status_date is not None:
                        db.new_api.set_field(self.prefs['status_date_column'], {book_id: status_date})
                    if len(self.prefs['reading_progress_column']) > 0:
                        db.new_api.set_field(self.prefs['reading_progress_column'], {book_id: status_progress})

                # -- Set Review --
                if self.prefs['sync_review']:
                    if review == (None, None):
                        pass
                    else:
                        if self.prefs['review_column'] == 'comments':
                            skoob_review = mi.comments
                        else:
                            skoob_review = mi.metadata_for_field(self.prefs['review_column'])['#value#']
                        # Personalized entry to merge with existing comments, so it can be retrieved easier
                        review_open = '## Skoob ##'
                        review_end = '##########'
                        review_title = review[0]
                        if review_title is None:
                            review_title = ''
                        review_text = tostring(review[1], method='html')
                        review_text = re.sub(b'\n', b'<br/>', review_text)
                        review_text = sanitize_comments_html(review_text)
                        review_text = re.sub('<br />', '</p><p>', review_text)

                        if len(review_text) > 0:
                            # Check if a Skoob Review is already set
                            if self.prefs['review_column'] == 'comments':
                                if skoob_review:
                                    existing_comments_review = re.search('## Skoob ##', skoob_review)
                                    if existing_comments_review:
                                        comments_clean = re.search('([\\s\\S]+)(## Skoob ##)',
                                                                   skoob_review).groups(0)[0]
                                        new_comments = comments_clean + review_open + '<h4>' + review_title + '</h4>' \
                                                       + review_text + review_end
                                    else:
                                        new_comments = skoob_review + review_open + '<h4>' + review_title + '</h4>'\
                                                       + review_text + review_end
                                else:
                                    new_comments = review_open + '<h4>' + review_title + '</h4>'\
                                                   + review_text + review_end
                            else:
                                new_comments = '<h4>' + review_title + '</h4>' + review_text
                            db.new_api.set_field(self.prefs['review_column'], {book_id: new_comments})
                        else:
                            pass

                # -- Set Rating --
                if self.prefs['sync_rating'] and rating is not None:
                    rating_number = float(rating) * 2
                    if len(self.prefs['rating_column']) > 0:
                        db.new_api.set_field(self.prefs['rating_column'], {book_id: rating_number})

                # -- Set Page Count --
                if self.prefs['sync_page_count']:
                    if len(self.prefs['page_count_column']) > 0:
                        db.new_api.set_field(self.prefs['page_count_column'], {book_id: page_count})

                # -- Set User Tags --
                if self.prefs['sync_tags']:
                    tags = tags[0]
                    if tags is None:
                        pass
                    else:
                        if self.prefs['tags_column'] == 'tags':
                            existing_tags = mi.tags
                            for tag_item in existing_tags:
                                tags.append(tag_item)
                            db.new_api.set_field(self.prefs['tags_column'], {book_id: tags})
                        else:
                            db.new_api.set_field(self.prefs['tags_column'], {book_id: tags})

                # -- Set Physical Book --
                if len(self.prefs['physical_column']) > 0:
                    db.new_api.set_field(self.prefs['physical_column'], {book_id: physical})

                # Mark the book as updated
                self.books_updated.append(book_id)
            else:
                # Mark the book as NOT updated
                self.books_not_updated.append(book_id)
        except:
            return

    def upload_to_skoob(self, skoob_id, book_id, db, mi):
        success = False
        try:
            # Reading progress
            if self.prefs['sync_reading_progress'] and self.shelf not in [3, 5]:
                reading_progress_url = 'https://api.skoob.com.br/estante/s_historico_leitura/' + str(skoob_id)
                read_percentage = mi.metadata_for_field(self.prefs['reading_progress_column'])['#value#']
                if read_percentage is not None:
                    # Fix inconsistent values
                    if read_percentage > 100:
                        read_percentage = 100
                        db.new_api.set_field(self.prefs['reading_progress_column'], {book_id: read_percentage})
                    if read_percentage < 100 and self.shelf == 1:
                        if len(self.prefs['reading_progress_column']) > 0:
                            db = self.gui.current_db.new_api
                            db.new_api.set_field(self.prefs['reading_progress_column'], {book_id: 100})
                            read_percentage = 100

                # Set the read column according to reading progress
                if self.prefs['sync_read_books']:
                    if len(self.prefs['read_books_column']) > 0:
                        if read_percentage is None:
                            pass
                            # db.new_api.set_field(self.prefs['read_books_column'], {book_id: ''})
                        else:
                            if int(read_percentage) == 100:
                                db.new_api.set_field(self.prefs['read_books_column'], {book_id: True})
                            else:
                                db.new_api.set_field(self.prefs['read_books_column'], {book_id: ''})

                # Check if book exists on shelves
                root = self.get_user_data(skoob_id)
                exists = root[2]
                if exists:
                    pass
                elif exists is None:
                    self.bad_connection = True
                    return self.bad_connection
                else:
                    return self.books_not_in_shelf.append(book_id)

                if read_percentage:

                    payload = {
                        'data[LendoHistorico][paginas]': read_percentage,
                        'data[LendoHistorico][tipo]': 1,
                    }

                    # Open the url and check if it succeeded
                    success, self.bad_connection = open_url(reading_progress_url, payload=payload)
                    if self.bad_connection or not success:
                        self.books_not_updated.append(book_id)
                        return self.bad_connection

                    # Set status date
                    status_date = datetime.datetime.today().astimezone()
                    if len(self.prefs['status_date_column']) > 0:
                        db.new_api.set_field(self.prefs['status_date_column'], {book_id: status_date})

            # Rating
            if self.prefs['sync_rating']:
                if self.prefs['read_books_column'] == 'rating':
                    rating = mi.rating
                else:
                    rating = mi.metadata_for_field(self.prefs['rating_column'])['#value#']

                if rating is not None:
                    rating_url = 'https://api.skoob.com.br/v1/book_rate/' + str(skoob_id) + '/' + str(rating/2) + '/'

                    # Open the url and check if it succeeded
                    success, self.bad_connection = open_url(rating_url)
                    if self.bad_connection or not success:
                        self.books_partially_updated.append(book_id)
                        return self.bad_connection

            # Review
            if self.prefs['sync_review']:
                review_title = review_text = None
                # Check which column holds the review
                if self.prefs['review_column'] == 'comments':
                    calibre_review = mi.comments
                else:
                    calibre_review = mi.metadata_for_field(self.prefs['review_column'])['#value#']

                if calibre_review:
                    if self.prefs['review_column'] == 'comments':
                        # Check if the review is inside 'comments' column. If it is, then get it.
                        if calibre_review:
                            existing_comments_review = re.search('## Skoob ##', calibre_review)
                            if existing_comments_review:
                                review_title = re.search(
                                    '<p>## Skoob ##</p>[\\s\\S]*<h4.*>([\\s\\S]*)</h4>[\\s\\S]*<p.*>([\\s\\S]*)</p>\\s',
                                    calibre_review).groups(0)[0]
                                review_text = re.search(
                                    '<p>## Skoob ##</p>[\\s\\S]*<h4.*>([\\s\\S]*)</h4>([\\s\\S]*)<p>#',
                                    calibre_review).groups(0)[1]
                                review_text = re.sub('<p>', '', review_text)
                                review_text = re.sub('</p>', '\n', review_text)
                    else:
                        # Check if the review is on a custom column. If it is, then get it.
                        is_review_title = re.search('<h4.*>([\\s\\S]*)</h4>([\\s\\S]*)', calibre_review)
                        is_review_text = re.search('<h4.*>([\\s\\S]*)</h4>[\\s\\S]*<p.*>([\\s\\S]*)</p>', calibre_review)
                        if is_review_title:
                            review_title = re.search('<h4.*>([\\s\\S]*)</h4>[\\s\\S]*<p.*>([\\s\\S]*)</p>',
                                                     calibre_review).groups(0)[0]
                        if is_review_text:
                            review_text = re.search('<h4.*>([\\s\\S]*)</h4>([\\s\\S]*)</p>',
                                                    calibre_review).groups(0)[1]
                            review_text = re.sub('<p>', '', review_text)
                            review_text = re.sub('</p>', '\n', review_text)
                        else:
                            review_title = None
                            review_text = sanitize_comments_html(calibre_review)
                            clean_review_text = re.compile('<.*?>')
                            review_text = re.sub(clean_review_text, '', review_text)

                    if not review_title and not review_text:
                        pass
                    else:
                        review_url = 'https://api.skoob.com.br/estante/s_resenha/' + str(skoob_id)

                        if review_title:
                            payload = {
                                'data[Resenha][titulo_resenha]': review_title,
                                'data[Resenha][resenha]': review_text
                            }
                        else:
                            payload = {
                                'data[Resenha][resenha]': review_text
                            }

                        # Open the url and check if it succeeded
                        success, self.bad_connection = open_url(review_url, payload=payload)
                        if self.bad_connection or not success:
                            self.books_partially_updated.append(book_id)
                            return self.bad_connection

            # User tags
            if self.prefs['sync_tags']:
                # Check which column holds the user tags
                if self.prefs['tags_column'] == 'tags':
                    calibre_tags = mi.tags
                else:
                    calibre_tags = mi.metadata_for_field(self.prefs['tags_column'])['#value#']

                if calibre_tags:
                    calibre_tags = ','.join(calibre_tags)
                    tags_url = 'https://api.skoob.com.br/estante/s_editar/' + str(skoob_id)

                    tags_req = six.moves.urllib.request.Request(tags_url, headers=random_ua())
                    tags_info = six.moves.urllib.request.urlopen(tags_req)
                    tags = tags_info.read()
                    tags_decode = tags.decode('iso-8859-1', errors='replace')
                    root_t = fromstring(clean_ascii_chars(tags_decode))
                    tags, date_read = self.parse_tags(root_t)

                    if date_read == [[], [], []]:
                        payload = {
                            'data[MeusLivro][tags_publicas]': calibre_tags.encode("ISO-8859-1"),
                        }
                    else:
                        payload = {
                            'data[MeusLivro][tags_publicas]': calibre_tags.encode("ISO-8859-1"),
                            'data[MeusLivro][dia_leitura]': date_read[0][0],
                            'data[MeusLivro][mes_leitura]': date_read[1][0],
                            'data[MeusLivro][ano_leitura]': date_read[2][0]
                        }

                    # Open the url and check if it succeeded
                    success, self.bad_connection = open_url(tags_url, payload=payload)
                    if self.bad_connection or not success:
                        self.books_partially_updated.append(book_id)
                        return self.bad_connection

            # Set book media type (physical or digital)
            if len(self.prefs['physical_column']) > 0:
                media_type = mi.metadata_for_field(self.prefs['physical_column'])['#value#']
                if media_type:
                    media_type = 0
                else:
                    media_type = 1
                media_type_url = 'https://api.skoob.com.br/estante/formato/' + str(skoob_id) + '/' + str(media_type)

                # Open the url and check if it succeeded
                success, self.bad_connection = open_url(media_type_url)
                if self.bad_connection or not success:
                    self.books_partially_updated.append(book_id)
                    return self.bad_connection

            if success:
                self.books_updated.append(book_id)
            else:
                self.books_not_updated.append(book_id)

        except:
            return

    def add_to_skoob(self, skoob_id, book_id, mi):
        try:
            read_percentage = mi.metadata_for_field(self.prefs['reading_progress_column'])['#value#']

            # Add to the according shelf
            if self.shelf < 6:
                if self.shelf is None:
                    if read_percentage == 100 and not self.prefs['sync_update_added']:
                        # Read shelf
                        add_to_skoob_url = 'https://api.skoob.com.br/v1/shelf_add/' + str(skoob_id) + '/1/'
                    else:
                        # Reading shelf
                        add_to_skoob_url = 'https://api.skoob.com.br/v1/shelf_add/' + str(skoob_id) + '/2/'
                else:
                    if self.shelf == 1 and self.prefs['sync_update_added'] and read_percentage is not None:
                        add_to_skoob_url = 'https://api.skoob.com.br/v1/shelf_add/' + str(skoob_id) + '/2/'
                    else:
                        add_to_skoob_url = 'https://api.skoob.com.br/v1/shelf_add/' + str(skoob_id) +\
                                       '/' + str(self.shelf) + '/'
            else:
                # Mark book
                add_to_skoob_url = 'https://api.skoob.com.br/v1/label_add/' + str(skoob_id) + '/' + str(self.shelf) + '/'

            # Build our Request object
            add_to_skoob_req = six.moves.urllib.request.Request(add_to_skoob_url, headers=random_ua())

            try:
                # For some reason, Skoob only recognizes the second attempt to add the book... # Bug fixed
                # add_to_skoob_open = six.moves.urllib.request.urlopen(add_to_skoob_req)
                add_to_skoob_open = six.moves.urllib.request.urlopen(add_to_skoob_req)

                # Check if the operation succeeded
                add_response = add_to_skoob_open.read().decode('utf-8')
                if '"success":true' in add_response:
                    valid_id = True
                else:
                    valid_id = False
                    self.bad_skoob_id.append(book_id)
                    return False, valid_id, False

                # If checked, updates the book status on Skoob after adding to shelf
                if self.prefs['sync_update_added'] and self.shelf < 6:
                    to_update = True
                    return False, valid_id, to_update
                else:
                    self.books_updated.append(book_id)
                    return False, valid_id, False

            except six.moves.urllib.error.URLError:
                return True, None, None

        except:
            title = mi.title
            return error_dialog(self.gui, _('Not added'), _('%s could not be added to Skoob.\nPlease verify'
                                                            ' its Skoob ID.') % title, show_copy_button=False, show=True)

    def remove_from_skoob(self, skoob_id, book_id, mi):
        try:
            # Remove from Skoob
            remove_from_skoob_url = 'https://api.skoob.com.br/v1/shelf_del/' + str(skoob_id) + '/'

            # Build our Request object
            remove_from_skoob_req = six.moves.urllib.request.Request(remove_from_skoob_url, headers=random_ua())

            try:
                remove_from_skoob_open = six.moves.urllib.request.urlopen(remove_from_skoob_req)

                # Check if the operation succeeded
                remove_response = remove_from_skoob_open.read().decode('utf-8')
                if '"success":true' in remove_response:
                    self.books_updated.append(book_id)
                    return False, True
                else:
                    self.bad_skoob_id.append(book_id)
                    return False, False

            except six.moves.urllib.error.URLError:
                return True, False

        except:
            title = mi.title
            return error_dialog(self.gui, _('Not removed'), _('%s could not be removed from Skoob.\n'
                                                              'Please verify its Skoob ID.') % title,
                                                              show_copy_button=False, show=True)

    def add_to_goal(self, skoob_id, book_id, mi):
        try:
            add_to_goal_url = 'https://api.skoob.com.br/v1/meta/add/' + str(skoob_id) + '/'

            # Build our Request object
            add_to_goal_req = six.moves.urllib.request.Request(add_to_goal_url, headers=random_ua())

            try:
                add_to_goal_open = six.moves.urllib.request.urlopen(add_to_goal_req)

                # Check if the operation succeeded
                add_response = add_to_goal_open.read().decode('utf-8')
                if '"success":true' in add_response:
                    self.books_updated.append(book_id)
                else:
                    self.bad_skoob_id.append(book_id)

            except six.moves.urllib.error.URLError:
                self.bad_connection = True
                return self.bad_connection

        except:
            title = mi.title
            return error_dialog(self.gui, _('Not added'), _('%s could not be added to Reading Goal.\nPlease verify'
                                                            ' its Skoob ID.') % title, show_copy_button=False, show=True)

    def get_read_date(self, skoob_id, book_id, db, mi):
        try:
            read_date_url = 'https://api.skoob.com.br/estante/s_editar/' + str(skoob_id)

            # Build our Request object
            read_date_req = six.moves.urllib.request.Request(read_date_url, headers=random_ua())

            try:
                read_date_open = six.moves.urllib.request.urlopen(read_date_req)
                read_date_read = read_date_open.read()
                read_date_decode = read_date_read.decode('iso-8859-1', errors='replace')

                if 'Não existe o livro' in read_date_decode:
                    return self.books_not_in_shelf.append(book_id)
                else:
                    pass

                root_rd = fromstring(clean_ascii_chars(read_date_decode))

            except six.moves.urllib.error.URLError:
                self.bad_connection = True
                return self.bad_connection

            try:
                tags, read_date = self.parse_tags(root_rd)
                if read_date == [[], [], []]:
                    status_date = None
                else:
                    status_date = read_date[0][0] + '/' + read_date[1][0] + '/' + read_date[2][0]

            except:
                status_date = None

            if len(self.prefs['status_date_column']) > 0 and status_date is not None:
                db.new_api.set_field(self.prefs['status_date_column'], {book_id: status_date})
                self.books_updated.append(book_id)
            else:
                self.books_not_updated.append(book_id)

        except:
            title = mi.title
            return error_dialog(self.gui, _('Not added'), _('%s has no Read Date.\nPlease verify'
                                                            ' its Skoob ID.') % title, show_copy_button=False, show=True)

    def login(self):
        # Get preferences
        self.prefs = prefs
        if sys.version_info[0] < 3:
            self.key_password = prefs['password'].decode('base64')
        else:
            self.key_password = base64.b64decode(prefs['password'])

        # Login to Skoob. Session will keep open while checking multiple books.
        try:
            self.cookie_path = os.path.join(config_dir, 'plugins', PLUGIN_NAME, 'cookie', 'cookie.txt')
            self.cj = six.moves.http_cookiejar.MozillaCookieJar(self.cookie_path)

            # Check if user credentials have changed
            if not (KEY_USER == self.prefs['user'] and KEY_PASSWORD == self.key_password):
                login_status = self.skoob_login()

            else:
                try:
                    # Check if session cookie is still valid
                    self.cj.load()
                    for cookie in self.cj:
                        if cookie.is_expired():
                            login_status = self.skoob_login()
                            return login_status

                    # Limit the cookie life to 14 days. That seems to be the limit, although the
                    # cookie says it is still valid. The above loop only find the CakeCookie.
                    self.last_modified = os.path.getmtime(self.cookie_path)
                    self.last_modified = datetime.datetime.fromtimestamp(self.last_modified)
                    self.now = datetime.datetime.now()
                    self.cookie_life = self.now - self.last_modified
                    self.cookie_life = self.cookie_life.total_seconds()
                    if self.cookie_life > 1200000:
                        login_status = self.skoob_login()
                        return login_status
                except OSError:
                    login_status = self.skoob_login()
                    return login_status

                self.opener = six.moves.urllib.request.build_opener(six.moves.urllib.request.HTTPCookieProcessor(self.cj))
                six.moves.urllib.request.install_opener(self.opener)
                login_status = True
                if prefs['user_id']:
                    self.user_id = prefs['user_id']
                else:
                    login_status = self.skoob_login()
                    return login_status

        except IOError:
            login_status = self.skoob_login()

        if login_status:
            return True
        elif login_status == False:
            open(self.cookie_path, 'w').close()
            error_dialog(self.gui, _('Invalid credentials'),
                         _('Username and/or password invalid. Can\'t connect to Skoob.'),
                         show_copy_button=False, show=True)
            return False
        else:
            return None

    def skoob_login(self):
        """ Login to Skoob """

        # Solution using urllib.
        # Found here: https://stackoverflow.com/questions/13925983/login-to-website-using-urllib2-python-2-7

        # Store the cookies and create an opener that will hold them.
        # Needed to make multiple requests, after logging in.

        try:
            print(_('Loading stored cookies failed, manually logging in...'))
            self.gui.status_bar.showMessage(_("Connecting to Skoob..."))
            try:
                QApplication.processEvents(QtCore.QEventLoop.ProcessEventsFlag.ExcludeUserInputEvents)
            except AttributeError:
                pass

            self.opener = six.moves.urllib.request.build_opener(six.moves.urllib.request.HTTPCookieProcessor(self.cj))

            # Install our opener (note that this changes the global opener to the one
            # we just made, but you can also just call opener.open() if you want)
            six.moves.urllib.request.install_opener(self.opener)

            # Authentication page
            authentication_url = 'https://api.skoob.com.br/login'

            # Credentials
            payload = {
                'data[Usuario][email]': self.prefs['user'],
                'data[Usuario][senha]': self.key_password,
            }

            # Use urllib to encode the payload
            data = six.moves.urllib.parse.urlencode(payload).encode()

            # Build our Request object (supplying 'data' makes it a POST)
            login_req = six.moves.urllib.request.Request(authentication_url, data, headers = random_ua())

            login = six.moves.urllib.request.urlopen(login_req)

            # Save our cookie for future uses
            self.cj.save(self.cookie_path, ignore_discard=True, ignore_expires=True)

            # Test if credentials were accepted
            user_page_url = login.geturl()
            self.user_id = re.search('(https://api.skoob.com.br/usuario/)(.\\d+)', user_page_url)
            if self.user_id:
                self.user_id = re.search('(https://api.skoob.com.br/usuario/)(.\\d+)', user_page_url).groups(0)[1]
                prefs['user_id'] = self.user_id
            if user_page_url == 'https://api.skoob.com.br/login/':
                return False
            else:
                return True

        except six.moves.urllib.error.HTTPError as err:
            print('err', err)
            print('err.code', err.code)
            # Test if Skoob is down
            if '502' in str(err.code):
                error_dialog(self.gui, _('Skoob is down'), _('Seems like Skoob is down right now'),
                             show_copy_button=False, show=True)
            elif '504' in str(err.code):
                error_dialog(self.gui, _('Skoob is down'), _('Seems like Skoob is down right now'),
                             show_copy_button=False, show=True)
            return None

        except six.moves.urllib.error.URLError:
            # Test if Skoob is down
            try:
                skoob_response = login.read().decode('utf-8')
                if 'Aconteceu um problema.' in skoob_response:
                    error_dialog(self.gui, _('Skoob is down'), _('Seems like Skoob is down right now'),
                                 show_copy_button=False, show=True)
            except:
                error_dialog(self.gui, _('No internet connection'), _('Seems like you have no internet access'),
                             show_copy_button=False, show=True)
            return None

    def get_details(self, skoob_id):
        """Main method to fetch data from Skoob"""

        try:
            root = self.get_user_data(skoob_id)

            if len(root) == 0:
                return

            exists = root[2]
            if exists:
                pass
            elif exists is None:
                return None
            else:
                return False

            try:
                if self.prefs['sync_reading_progress']:
                    status = self.parse_status(root[0])
                    shelf = self.parse_shelf(root[0])
                    type = self.parse_physical(root[0])
                    if type == '0':
                        physical = True
                    else:
                        physical = None
                else:
                    status = (None, None)
                    shelf = None
                    physical = None
            except:
                status = (None, None)
                shelf = None
                physical = None

            try:
                if self.prefs['sync_review']:
                    review = self.parse_review(root[1])
                else:
                    review = (None, None)
            except:
                review = (None, None)

            try:
                if self.prefs['sync_rating']:
                    rating = self.parse_rating(root[0])
                else:
                    rating = None
            except:
                rating = None

            try:
                if self.prefs['sync_page_count']:
                    page_count = self.parse_page_count(root[0])
                else:
                    page_count = None
            except:
                page_count = None

            try:
                if self.prefs['sync_tags']:
                    tags = self.parse_tags(root[3])
                else:
                    tags = None
            except:
                tags = None

        except:
            return error_dialog(self.gui, _('Error'), _('Error parsing book details'),
                                show_copy_button=False, show=True)

        return status, review, rating, page_count, shelf, tags, physical

    def get_user_data(self, skoob_id):
        root_rp = None
        root_p = None
        exists = False
        root_t = None
        # Reading progress, review and tags pages
        status_url = 'https://api.skoob.com.br/estante/s_historico_leitura/' + str(skoob_id)
        review_url = 'https://api.skoob.com.br/estante/s_resenha/' + str(skoob_id)
        tags_url = 'https://api.skoob.com.br/estante/s_editar/' + str(skoob_id)

        # Build our Requests objects
        status_req = six.moves.urllib.request.Request(status_url, headers=random_ua())
        review_req = six.moves.urllib.request.Request(review_url, headers=random_ua())
        tags_req = six.moves.urllib.request.Request(tags_url, headers=random_ua())

        try:
            # Request status of the book
            if self.prefs['sync_reading_progress'] or self.prefs['sync_rating'] or self.prefs['sync_page_count']:
                status_info = six.moves.urllib.request.urlopen(status_req)

                status = status_info.read()
                status_decode = status.decode('iso-8859-1', errors='replace')

                try:
                    root_rp = fromstring(clean_ascii_chars(status_decode))
                    exists = self.is_book_in_shelf(root_rp)
                    if exists:
                        pass
                    else:
                        return None, None, exists
                except:
                    error_dialog(self.gui, _('Failed'), _('Failed to parse reading progress'), show=True)

            # Request review of the book
            if self.prefs['sync_review']:
                review_info = six.moves.urllib.request.urlopen(review_req)
                review = review_info.read()
                review_decode = review.decode('iso-8859-1', errors='replace')

                try:
                    root_p = fromstring(clean_ascii_chars(review_decode))
                    exists = self.is_book_in_shelf(root_p)
                except:
                    error_dialog(self.gui, _('Failed'), _('Failed to parse book review'),
                                 show_copy_button=False, show=True)

            # Request user tags for the book
            if self.prefs['sync_tags']:
                tags_info = six.moves.urllib.request.urlopen(tags_req)
                tags = tags_info.read()
                tags_decode = tags.decode('iso-8859-1', errors='replace')

                try:
                    root_t = fromstring(clean_ascii_chars(tags_decode))
                except:
                    error_dialog(self.gui, _('Failed'), _('Failed to parse user tags'),
                                 show_copy_button=False, show=True)

        except six.moves.urllib.error.URLError:
            return None, None, None

        return root_rp, root_p, exists, root_t

    def is_book_in_shelf(self, root):
        in_shelf_node = root.xpath("id('wd_referente')/div[1]/div[1]/h2[1]")
        if in_shelf_node:
            return True
        else:
            return False

    def parse_status(self, root):
        date_node = root.xpath("id('wd_referente')/div[1]/div[2]/div[1]/div[1]")
        if date_node:
            date = date_node[0].text_content().strip()
        else:
            date = None

        progress_node = root.xpath("id('wd_referente')/div[1]/div[2]/div[1]/div[2]/div[2]")
        if progress_node:
            progress_text = progress_node[0].text_content().strip()
            progress = re.search('(.+)(?=%)', progress_text).groups(0)[0]
        else:
            progress = None

        return date, progress

    def parse_review(self, root):
        review_title_node = root.xpath("//input[@name='data[Resenha][titulo_resenha]']/@value")
        if review_title_node:
            review_title = review_title_node[0]
        else:
            review_title = None

        review_text_node = root.xpath("//textarea[@name='data[Resenha][resenha]']")
        if review_text_node:
            review_text = review_text_node[0]
            aux_review_text = review_text.text_content().strip()
            if aux_review_text == '':
                review_text = None
        else:
            review_text = None

        return review_title, review_text

    def parse_rating(self, root):
        rating_node = root.xpath("//star-rating/@rate")
        if rating_node:
            rating = rating_node[0]
        else:
            rating = None

        return rating

    def parse_page_count(self, root):
        page_count_node = root.xpath("id('wd_referente')/div[1]/div[1]/text()")
        if page_count_node:
            page_count_text = page_count_node[1]
            page_count = re.search('(/ )(.\\d+)', page_count_text).groups(0)[1]
        else:
            page_count = None

        return page_count

    def parse_shelf(self, root):
        shelf_node = root.xpath("//button-shelf/@status")
        if shelf_node:
            shelf = shelf_node[0]
        else:
            shelf = None
        return shelf

    def parse_tags(self, root):
        tags_shelf_node = root.xpath("//input[@name='data[MeusLivro][tags]']/@value")
        tags_public_node = root.xpath("//input[@name='data[MeusLivro][tags_publicas]']/@value")

        day_read = root.xpath("//select[@name='data[MeusLivro][dia_leitura]']/option[@selected]/text()")
        month_read = root.xpath("//select[@name='data[MeusLivro][mes_leitura]']/option[@selected]/text()")
        year_read = root.xpath("//select[@name='data[MeusLivro][ano_leitura]']/option[@selected]/text()")
        date_read = [day_read, month_read, year_read]

        tags = []
        if tags_shelf_node:
            for tag_node in tags_shelf_node:
                tag_list = tag_node.split(',')
                for tag_item in tag_list:
                    tags.append(tag_item)

        if tags_public_node:
            for tag_node in tags_public_node:
                tag_list = tag_node.split(',')
                for tag_item in tag_list:
                    tags.append(tag_item)

        return tags, date_read

    def parse_physical(self, root):
        physical_node = root.xpath("//div[@data-ng-controller='prateleiraCtrl']/@data-ng-init")
        if physical_node:
            physical = re.search('(\\d),(\\d)', physical_node[0]).groups(0)[1]
        else:
            physical = None

        return physical

    # def parse_read_date(self, root):
    #     read_date_node = root.xpath("//div[@data-ng-controller='prateleiraCtrl']/@data-ng-init")
    #     if read_date_node:
    #         read_date = re.search('(\\d),(\\d)', read_date_node[0]).groups(0)[1]
    #     else:
    #         read_date = None
    #
    #     return read_date

    def are_selected_books_linked(self):
        all_has_skoob_id = True
        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0:
            return False
        db = self.gui.library_view.model().db
        for row in rows:
            calibre_id = db.id(row.row())
            identifiers = db.get_identifiers(calibre_id, index_is_id=True)
            if 'skoob' not in identifiers:
                all_has_skoob_id = False
                break

        return all_has_skoob_id

    # Method 'any' reimplemented, since it is marked to change, according to FUTURE warning
    def any(self, iterable):
        for element in iterable:
            if element is not None:
                return True
        return False

    def create_empty_books(self, skoob_id):

        self.ids = []

        # Get preferences
        self.prefs = prefs

        # Create mi object and include standard fields
        skoob_id = int(skoob_id)
        book_title = self.books[skoob_id]['title']
        if type(book_title) is int:
            book_title = str(book_title)
        book_authors = self.books[skoob_id]['authors']
        db = self.gui.library_view.model().db
        mi = MetaInformation(book_title, book_authors.split(','))
        skoob_id_complete = str(self.books[skoob_id]['book_id']) + 'ed' + str(skoob_id)
        mi.set_identifier('skoob', skoob_id_complete)
        mi.publisher = self.books[skoob_id]['publisher']
        mi.pubdate = datetime.datetime(int(self.books[skoob_id]['year']), 1, 2)
        mi.comments = self.books[skoob_id]['comments']
        cover_url = self.books[skoob_id]['cover_url']
        cover_path = os.path.join(config_dir, 'plugins', PLUGIN_NAME, 'images', str(skoob_id) + 'cover.jpg')

        req = six.moves.urllib.request.Request(cover_url, headers=random_ua())
        response = six.moves.urllib.request.urlopen(req)
        cover_data = response.read()
        f = open(cover_path, 'wb')
        f.write(cover_data)
        f.close()

        mi.cover = cover_path

        # Create empty book (no format) based on the mi object
        book_id = db.import_book(mi, [])
        self.ids.append(book_id)

        # Update custom columns
        # Reading progress
        if self.prefs['sync_reading_progress']:
            status_progress = self.books[skoob_id]['reading_progress']
            if status_progress:
                if status_progress > 100:
                    status_progress = 100
            db.new_api.set_field(self.prefs['reading_progress_column'], {book_id: status_progress})
            # Set the status date
            if len(self.prefs['status_date_column']) > 0:
                status_date = self.books[skoob_id]['status_date']
                if len(status_date) > 0:
                    # date_aux = [int(re.search('(\\d+)-(\\d+)-(\\d+)', status_date).groups(0)[0]),
                    #             int(re.search('(\\d+)-(\\d+)-(\\d+)', status_date).groups(0)[1]),
                    #             int(re.search('(\\d+)-(\\d+)-(\\d+)', status_date).groups(0)[2])]
                    # status_date = datetime.datetime(date_aux[0], date_aux[1], date_aux[2], tzinfo=utc_tz)
                    db.new_api.set_field(self.prefs['status_date_column'], {book_id: status_date})
            # Set the read column according to reading progress
            if self.prefs['sync_read_books']:
                if len(self.prefs['read_books_column']) > 0:
                    if status_progress == 100 or self.books[skoob_id]['shelf'] == 1:
                        db.new_api.set_field(self.prefs['read_books_column'], {book_id: True})
                    else:
                        db.new_api.set_field(self.prefs['read_books_column'], {book_id: ''})
        # Rating
        if self.prefs['sync_rating']:
            if self.prefs['rating_column'] == 'rating':
                mi.rating = self.books[skoob_id]['rating']*2
            else:
                db.new_api.set_field(self.prefs['rating_column'], {book_id: self.books[skoob_id]['rating']*2})
        # Page count
        if self.prefs['sync_page_count']:
            db.new_api.set_field(self.prefs['page_count_column'], {book_id: self.books[skoob_id]['pages']})

    def skoob_library(self):

        # Get preferences
        self.prefs = prefs

        if sys.version_info[0] < 3:
            self.key_password = prefs['password'].decode('base64')
        else:
            self.key_password = base64.b64decode(prefs['password'])

        # Check if credentials are set
        if self.prefs['user'] in ('user', '') or self.key_password in ('password', '', b''):
            if question_dialog(self.gui, _('Invalid Credentials'), _('You must first setup your Skoob account') + '. ' +
                               _('\nDo you want to configure the plugin now?'), show_copy_button=False):
                return show_configuration(self, tab=0)
            else:
                return

        start = timer()

        # Login to Skoob.
        self.gui.status_bar.showMessage(_("Connecting to Skoob..."))
        try:
            QApplication.processEvents(QtCore.QEventLoop.ProcessEventsFlag.ExcludeUserInputEvents)
        except AttributeError:
            pass
        login_status = self.login()
        if login_status is False or login_status is None:
            return self.gui.status_bar.clearMessage()
        self.gui.status_bar.clearMessage()

        if self.user_id:
            user_id = self.user_id
        else:
            user_id = prefs['user_id']

        self.gui.status_bar.showMessage(_("Reading Skoob Library..."))
        try:
            QApplication.processEvents(QtCore.QEventLoop.ProcessEventsFlag.ExcludeUserInputEvents)
        except AttributeError:
            pass

        # json_path = os.path.join(config_dir, 'plugins', PLUGIN_NAME, 'library.json')
        skoob_library_url = 'https://api.skoob.com.br/v1/bookcase/books/' + user_id + '/shelf_id:0/page:1/limit:10000/'
        skoob_library_req = six.moves.urllib.request.Request(skoob_library_url, headers=random_ua())
        try:
            skoob_library = six.moves.urllib.request.urlopen(skoob_library_req)
        except six.moves.urllib.error.URLError:
            self.gui.status_bar.clearMessage()
            return error_dialog(self.gui, _('No internet connection'), _('Seems like you have no internet access'),
                                show_copy_button=False, show=True)

        skoob_library = skoob_library.read().strip()
        # json_file = open(json_path, "w")
        # json_file.write(json.dumps(json.loads(skoob_library), indent=4, sort_keys=True))
        # json_file.close()

        if len(json.loads(skoob_library)) > 0:
            self.books = {}
            for book in json.loads(skoob_library)['response']:
                # Get book details
                skoob_id = book['edicao']['id']
                book_id = book['livro_id']
                book_title = book['edicao']['titulo']
                book_authors = book['edicao']['autor']
                book_publisher = book['edicao']['editora']
                book_year = book['edicao']['ano']
                book_pages = book['edicao']['paginas']
                book_read_pages = book['paginas_lidas']
                book_comments = book['edicao']['sinopse']
                book_status_date = book['dt_leitura']
                if book_read_pages > 0:
                    book_reading_progress = book_read_pages/book_pages*100
                else:
                    book_reading_progress = None
                book_rating = book['ranking']
                book_shelf = book['tipo']
                book_cover_url = book['edicao']['capa_grande']
                # Create a nested dict with all the book entries
                self.books[skoob_id] = {
                                   'book_id': book_id,
                                   'title': book_title,
                                   'authors': book_authors,
                                   'publisher': book_publisher,
                                   'year': book_year,
                                   'pages': book_pages,
                                   'comments': book_comments,
                                   'reading_progress': book_reading_progress,
                                   'status_date': book_status_date,
                                   'rating': book_rating,
                                   'shelf': book_shelf,
                                   'cover_url': book_cover_url}
            # print(json.dumps(self.books, sort_keys=True, indent=4))

            self.gui.status_bar.clearMessage()

            # Create a widget listing all books on library
            self.tree = QTreeWidget()
            self.tree.setSelectionMode(QAbstractItemView.ExtendedSelection)
            # self.tree.setRootIsDecorated(False)
            layout = QVBoxLayout()
            layout.addWidget(self.tree)
            self.tree.setHeaderLabels([_('Books'), _('Skoob ID'), _('Title'), _('Authors'),
                                       _('Publisher'), _('Year'), _('Pages')])

            self.read = QTreeWidgetItem(self.tree)
            self.read.setText(0, _('1. Read'))
            if Qt_version >= 6:
                self.read.setFlags(self.read.flags() | Qt.ItemIsAutoTristate | Qt.ItemIsUserCheckable)
            else:
                self.read.setFlags(self.read.flags() | Qt.ItemIsTristate | Qt.ItemIsUserCheckable)
            self.read.setExpanded(True)

            self.reading = QTreeWidgetItem(self.tree)
            self.reading.setText(0, _('2. Reading'))
            if Qt_version >= 6:
                self.reading.setFlags(self.reading.flags() | Qt.ItemIsAutoTristate | Qt.ItemIsUserCheckable)
            else:
                self.reading.setFlags(self.reading.flags() | Qt.ItemIsTristate | Qt.ItemIsUserCheckable)
            self.reading.setExpanded(True)

            self.want_read = QTreeWidgetItem(self.tree)
            self.want_read.setText(0, _('3. Want to read'))
            if Qt_version >= 6:
                self.want_read.setFlags(self.want_read.flags() | Qt.ItemIsAutoTristate | Qt.ItemIsUserCheckable)
            else:
                self.want_read.setFlags(self.want_read.flags() | Qt.ItemIsTristate | Qt.ItemIsUserCheckable)
            self.want_read.setExpanded(True)

            self.rereading = QTreeWidgetItem(self.tree)
            self.rereading.setText(0, _('4. Rereading'))
            if Qt_version >= 6:
                self.rereading.setFlags(self.rereading.flags() | Qt.ItemIsAutoTristate | Qt.ItemIsUserCheckable)
            else:
                self.rereading.setFlags(self.rereading.flags() | Qt.ItemIsTristate | Qt.ItemIsUserCheckable)
            self.rereading.setExpanded(True)

            self.abandoned = QTreeWidgetItem(self.tree)
            self.abandoned.setText(0, _('5. Abandoned'))
            if Qt_version >= 6:
                self.abandoned.setFlags(self.abandoned.flags() | Qt.ItemIsAutoTristate | Qt.ItemIsUserCheckable)
            else:
                self.abandoned.setFlags(self.abandoned.flags() | Qt.ItemIsTristate | Qt.ItemIsUserCheckable)
            self.abandoned.setExpanded(True)

            # Add books to list widget
            index = 0
            shelves = [1, 2, 3, 4, 5]
            for shelf in shelves:
                for entry, values in self.books.items():
                    t_skoob_id = entry
                    t_info = values
                    t_title, t_authors, t_publisher, t_year, t_page_count, t_shelf = (
                        t_info['title'], t_info['authors'], t_info['publisher'],
                        t_info['year'], t_info['pages'], t_info['shelf'])

                    if t_shelf is not shelf:
                        pass
                    else:
                        index = int(index) + 1
                        index = "{0:0=3d}".format(index)
                        if t_shelf == 1:
                            parent = self.read
                        elif t_shelf == 2:
                            parent = self.reading
                        elif t_shelf == 3:
                            parent = self.want_read
                        elif t_shelf == 4:
                            parent = self.rereading
                        else:
                            parent = self.abandoned
                        child = QTreeWidgetItem(parent, [str(index), str(t_skoob_id), str(t_title), str(t_authors),
                                                         str(t_publisher), str(t_year), str(t_page_count)])
                        child.setFlags(child.flags() | Qt.ItemIsUserCheckable)
                        child.setCheckState(0, Qt.Unchecked)
                        if is_dark_theme() and (t_shelf in (1, 2, 3, 4)):
                            child.setForeground(1, QtGui.QBrush(QtGui.QColor("black")))
                            child.setForeground(2, QtGui.QBrush(QtGui.QColor("black")))
                            child.setForeground(3, QtGui.QBrush(QtGui.QColor("black")))
                            child.setForeground(4, QtGui.QBrush(QtGui.QColor("black")))
                            child.setForeground(5, QtGui.QBrush(QtGui.QColor("black")))
                            child.setForeground(6, QtGui.QBrush(QtGui.QColor("black")))
                        # Select background color based on book shelf
                        if t_shelf == 1:
                            bg_color = QtGui.QBrush(QtGui.QColor(180, 255, 180))
                        elif t_shelf == 2:
                            bg_color = QtGui.QBrush(QtGui.QColor(255, 255, 145))
                        elif t_shelf == 3:
                            bg_color = QtGui.QBrush(QtGui.QColor(180, 230, 255))
                        elif t_shelf == 4:
                            bg_color = QtGui.QBrush(QtGui.QColor(250, 190, 80))
                        else:
                            bg_color = QtGui.QBrush(QtGui.QColor(120, 120, 120))
                        if not is_dark_theme():
                            child.setBackground(0, QtGui.QColor(bg_color))
                        child.setBackground(1, QtGui.QColor(bg_color))
                        child.setBackground(2, QtGui.QColor(bg_color))
                        child.setBackground(3, QtGui.QColor(bg_color))
                        child.setBackground(4, QtGui.QColor(bg_color))
                        child.setBackground(5, QtGui.QColor(bg_color))
                        child.setBackground(6, QtGui.QColor(bg_color))
                        parent.addChild(child)

            # Auto adjust column sizes
            self.tree.resizeColumnToContents(0)
            self.tree.resizeColumnToContents(5)
            self.tree.resizeColumnToContents(6)

            # Enable sorting
            self.tree.setSortingEnabled(True)
            self.tree.sortItems(0, Qt.AscendingOrder)
            self.tree.setColumnHidden(1, True)

            # Enable selection when clicking anywhere on the line
            # self.tree.itemClicked.connect(self.on_item_clicked)

            end = timer()
            time = end - start
            print(datetime.timedelta(seconds=time))

            self.show_books(self.tree)

            # Create selected empty books
            if self.skoob_ids is None:
                return
            elif len(self.skoob_ids) > 0:
                start = timer()

                if self.prefs['sync_reading_progress']:
                    if len(self.prefs['reading_progress_column']) == 0:
                        if question_dialog(self.gui, _('Configuration needed'),
                                           _('You must choose the reading progress custom column') + '. '
                                           + _('\nDo you want to configure the plugin now?'), show_copy_button=False):
                            return show_configuration(self, tab=1)
                        else:
                            return

                if self.prefs['sync_page_count']:
                    if len(self.prefs['page_count_column']) == 0:
                        if question_dialog(self.gui, _('Configuration needed'),
                                           _('You must choose the page count custom column') + '. '
                                           + _('\nDo you want to configure the plugin now?'), show_copy_button=False):
                            return show_configuration(self, tab=1)
                        else:
                            return

                # Create empty books (each on a different thread)
                self.gui.status_bar.showMessage(_('Creating new entries to calibre library...'))
                try:
                    QApplication.processEvents(QtCore.QEventLoop.ProcessEventsFlag.ExcludeUserInputEvents)
                except AttributeError:
                    pass

                aux_list = []
                for skoob_id in self.skoob_ids:
                    aux_list.append((skoob_id,))
                self.run_parallel_in_threads(self.create_empty_books, aux_list)

                # Refresh the GUI, so the new books are displayed
                # self.gui.iactions['Add Books'].refresh_gui(len(self.ids)) --- Canonical way, recommended by Kovid
                try:
                    self.gui.iactions['Add Books'].refresh_gui(len(list(self.ids)))
                except AttributeError:
                    m = self.gui.library_view.model()
                    m.refresh_ids(list(self.ids))
                    m.refresh_ids(self.ids, current_row=self.gui.library_view.currentIndex().row())

                current = self.gui.library_view.currentIndex()
                self.gui.refresh_cover_browser()
                try:
                    self.gui.tags_view.recount_with_position_based_index()
                except AttributeError:
                    pass
                if QUICK_VIEW_SUPPORT:
                    qv = get_quickview_action_plugin()
                    if qv:
                        qv.refresh_quickview(current)

                self.gui.status_bar.clearMessage()

                end = timer()
                time = end - start
                print(datetime.timedelta(seconds=time))

                info_dialog(self.gui, _('New Books'), _('%d new book(s) added to calibre library')
                            % len(self.skoob_ids), show_copy_button=False, show=True)

                # Cleans up images folder
                covers_path = os.path.join(config_dir, 'plugins', PLUGIN_NAME, 'images')
                for img in os.listdir(covers_path):
                    if not img.endswith(".jpg"):
                        continue
                    os.remove(os.path.join(covers_path, img))

            else:
                return

        else:
            end = timer()
            time = end - start
            print(datetime.timedelta(seconds=time))

            self.gui.status_bar.clearMessage()
            error_dialog(self.gui, _('Error'), _('Failed to read Skoob Library'),
                         show_copy_button=False, show=True)

    # def on_item_clicked(self):
    #     item = self.tree.currentItem()
    #     item.setCheckState(0, QtCore.Qt.Unchecked if item.checkState(0) == QtCore.Qt.Checked else QtCore.Qt.Checked)

    def show_books(self, tree_widget):
        tool = self
        _gui = self.gui

        class SkoobLibraryDialog(Dialog):

            def __init__(self):
                Dialog.__init__(self, _('Skoob Library'), 'plugin-skoob-sync-dialog', parent=tool.gui)
                self.setWindowIcon(get_icon('icon.png'))

            def setup_ui(self):
                # Help information
                if is_dark_theme():
                    self.key_themes_idx = prefs['dark_themes_idx'] + 4
                else:
                    self.key_themes_idx = prefs['light_themes_idx'] + 1
                help_ico_src = os.path.join(config_dir, 'plugins', PLUGIN_NAME, 'images',
                                            'theme' + str(self.key_themes_idx), 'help.png')
                help_img_src = '<img src="' + help_ico_src + '" width="25" height="25"> '
                skoob_library_whatsthis = \
                    '<h4 style="vertical-align:top">' + help_img_src + _('Skoob Library') + '</h4>' + \
                    '<p>' + _('Here you have a list of all the books in your <i>Skoob Library</i>. They are separated '
                              'by shelf.') + '</p>' + \
                    '<p>' + _('To select the books you want to add to calibre, just use the checkboxes on the left, '
                              'or use your keyboard instead. Selected/highlighted books will be created, regardless '
                              'of whether they are checked or not. Then, just press <i>Create books</i>.') + '</p>' \
                    '<p>' + _('The books will be added to calibre with basic metadata and without a format. You can '
                              'add the actual book file later from your PC, by drag-and-droping it on the book cover, '
                              'in the <i>Book details</i> panel.') + '</p>'

                try:
                    # Show 'What's this' symbol. This must not be on the __init__, or
                    # it will cause the config widget to move everytime it is opened
                    Dialog.setWindowFlag(self, Qt.WindowContextHelpButtonHint, True)
                except AttributeError:
                    pass
                self.box = QVBoxLayout(self)
                self.widget = tree_widget
                self.widget.setWhatsThis(skoob_library_whatsthis)
                self.box.addWidget(self.widget)
                self.button_box = QDialogButtonBox()
                self.button_box.addButton(_('  Create books  '), QDialogButtonBox.AcceptRole)
                self.button_box.addButton(_('Cancel'), QDialogButtonBox.RejectRole)
                self.box.addWidget(self.button_box)
                self.button_box.accepted.connect(self.handler)
                self.button_box.rejected.connect(self.reject)

            def handler(self):
                self.count = -1
                # Get selected books
                SkoobSyncTools.skoob_ids = []

                # Look for checked items
                if Qt_version >= 6:
                    iterator = QTreeWidgetItemIterator(d.widget, QTreeWidgetItemIterator.IteratorFlag.Checked)
                else:
                    iterator = QTreeWidgetItemIterator(d.widget, QTreeWidgetItemIterator.Checked)
                while iterator.value():
                    item = iterator.value()
                    if item.parent():
                        SkoobSyncTools.skoob_ids.append(item.text(1))
                        self.count += 1
                    iterator += 1

                # Look for selected items (so one could use shift/ctrl + arrow keys)
                self.new_count = 0
                if Qt_version >= 6:
                    iterator = QTreeWidgetItemIterator(d.widget, QTreeWidgetItemIterator.IteratorFlag.NotChecked |
                                                       QTreeWidgetItemIterator.IteratorFlag.Selected)
                else:
                    iterator = QTreeWidgetItemIterator(d.widget, QTreeWidgetItemIterator.NotChecked |
                                                       QTreeWidgetItemIterator.Selected)
                while iterator.value():
                    item = iterator.value()
                    if item.parent():
                        SkoobSyncTools.skoob_ids.append(item.text(1))
                        self.new_count += 1
                    iterator += 1

                self.count += self.new_count + 1
                if self.count > 50:
                    SkoobSyncTools.skoob_ids = None
                    error_dialog(_gui, _('Too many books'),
                                 _('%d books selected. Try to select 50 books at most.') % self.count,
                                 show_copy_button=False, show=True)
                elif self.count == -1:
                    SkoobSyncTools.skoob_ids = None
                    error_dialog(_gui, _('Empty selection'), _('No books selected'), show_copy_button=False, show=True)
                else:
                    Dialog.accept(self)

            def reject(self):
                SkoobSyncTools.skoob_ids = None
                Dialog.reject(self)

        d = SkoobLibraryDialog()
        d.exec_()

    def reading_goal_statistics(self, year_text=str(datetime.date.today().year)):
        # Get preferences
        self.prefs = prefs

        if sys.version_info[0] < 3:
            self.key_password = prefs['password'].decode('base64')
        else:
            self.key_password = base64.b64decode(prefs['password'])
        self.key_year_text = year_text

        # Check if credentials are set
        if self.prefs['user'] in ('user', '') or self.key_password in ('password', '', b''):
            if question_dialog(self.gui, _('Invalid Credentials'), _('You must first setup your Skoob account') + '. ' +
                                         _('\nDo you want to configure the plugin now?'), show_copy_button=False):
                return show_configuration(self, tab=0)
            else:
                return

        start = timer()

        # Login to Skoob.
        self.gui.status_bar.showMessage(_("Connecting to Skoob..."))
        try:
            QApplication.processEvents(QtCore.QEventLoop.ProcessEventsFlag.ExcludeUserInputEvents)
        except AttributeError:
            pass
        login_status = self.login()
        if login_status is False or login_status is None:
            self.gui.status_bar.clearMessage()
            return None, None
        self.gui.status_bar.clearMessage()

        if self.user_id:
            user_id = self.user_id
        else:
            user_id = prefs['user_id']

        self.gui.status_bar.showMessage(_("Collecting reading statistics..."))
        try:
            QApplication.processEvents(QtCore.QEventLoop.ProcessEventsFlag.ExcludeUserInputEvents)
        except AttributeError:
            pass

        # List of info from Reading Goal statistics
        args = []

        # Main page for Reading Goal
        reading_goal_statistics_url = 'https://www.skoob.com.br/meta/ano/' + str(self.key_year_text) + '/' + user_id
        reading_goal_statistics_req = six.moves.urllib.request.Request(reading_goal_statistics_url, headers=random_ua())

        # Json for month statistics
        month_stats_url = ('https://www.skoob.com.br/api2/stats/user_year/.json?user_id=' + user_id + '&year=' +
                           str(self.key_year_text))
        month_stats_req = six.moves.urllib.request.Request(month_stats_url, headers=random_ua())

        # Json for Ranking
        ranking_url = 'https://www.skoob.com.br/api2/stats/ranking_preview/' + user_id + '/' + str(self.key_year_text)
        ranking_req = six.moves.urllib.request.Request(ranking_url, headers=random_ua())

        try:
            reading_goal_statistics = six.moves.urllib.request.urlopen(reading_goal_statistics_req)
            reading_goal_statistics = reading_goal_statistics.read()
            reading_goal_statistics_decode = reading_goal_statistics.decode('iso-8859-1', errors='replace')
        except six.moves.urllib.error.URLError:
            return self.gui.status_bar.clearMessage()

        try:
            root = fromstring(clean_ascii_chars(reading_goal_statistics_decode))
        except:
            return error_dialog(self.gui, _('Failed'), _('Failed to get Reading Goal statistics'),
                                show_copy_button=False, show=True)

        # Number of read books
        books_read_node = root.xpath("//div[@class='meta-usuario-paginas-container']/text()")
        books_read_node = books_read_node[1].strip()
        books_read_node = books_read_node.replace('.', '')
        books_read = re.findall('(\\d+)', books_read_node)[0]
        books_to_read = re.findall('(\\d+)', books_read_node)[1]
        args.append([books_read, books_to_read])  # args[0]

        # Percentage of Reading Goal
        books_read_percentage_node = root.xpath("//*[@id='meta-usuario-detalhes']/div[3]/div/div[2]/text()")
        books_read_percentage = books_read_percentage_node[0].strip()
        books_read_percentage = re.search('(\\d+)', books_read_percentage).groups(0)[0]
        args.append(books_read_percentage)  # args[1]

        # Read pages
        read_pages_node = root.xpath("//*[@id='meta-usuario-detalhes']/div[4]/div[1]/strong/text()")
        read_pages = read_pages_node[1].strip()
        args.append(read_pages)  # args[2]

        # Pages yet to read
        to_read_pages_node = root.xpath("//*[@id='meta-usuario-detalhes']/div[4]/div[1]/span/text()")
        to_read_pages = to_read_pages_node[0]
        to_read_pages = re.findall('(\\d*\\.*\\d+)', to_read_pages)[0]
        args.append(to_read_pages)  # args[3]

        # Current pace
        current_pace_node = root.xpath("//*[@id='meta-usuario-detalhes']/div[4]/div[1]/span/text()")
        current_pace = current_pace_node[0]
        current_pace = re.findall('(\\d*\\.*\\d+)', current_pace)
        current_pace = int(current_pace[1])
        current_pace = '{0:,}'.format(current_pace).replace(',', '.')
        args.append(current_pace)  # args[4]

        # Calculate the ideal pace
        last_day = datetime.date(int(self.key_year_text), 12, 31)
        remaining_days = (last_day - datetime.date.today()).days
        to_read_pages = to_read_pages.replace('.', '')
        ideal_pace = int(round(int(to_read_pages)/remaining_days))
        ideal_pace = '{0:,}'.format(ideal_pace).replace(',', '.')
        args.append(ideal_pace if int(ideal_pace.replace('.', '')) > 0 else '—')  # args[5]

        # Get month statistics
        try:
            month_stats = six.moves.urllib.request.urlopen(month_stats_req)
            month_stats = month_stats.read()
            month_stats_decode = month_stats.decode('iso-8859-1', errors='replace')
        except six.moves.urllib.error.URLError:
            return self.gui.status_bar.clearMessage()

        stats = month_stats_decode.strip()
        global goal_is_set
        if '"success":false' in stats:
            goal_is_set = False
        else:
            goal_is_set = True
        if goal_is_set:
            try:
                if len(json.loads(stats)) > 0:
                    months = {}
                    for month in json.loads(stats)['response']['months']:
                        month_name = month['month']
                        read_books = month['total']
                        percent = month['percent']
                        # Create a dict with all the months statistics
                        months[month_name] = [read_books, percent]
                    args.append(months)  # args[6]
            except KeyError:
                goal_is_set = False

        # Get ranking details
        if goal_is_set:
            try:
                ranking_stats = six.moves.urllib.request.urlopen(ranking_req)
                ranking_stats = ranking_stats.read()
                ranking_stats_decode = ranking_stats.decode('iso-8859-1', errors='replace')
            except six.moves.urllib.error.URLError:
                return self.gui.status_bar.clearMessage()

            ranking = ranking_stats_decode.strip()
            ranking_json = json.loads(ranking)

            try:
                if len(ranking_json) > 0:
                    for user in ranking_json['response']:
                        if str(user['id']) == user_id:
                            ranking_friends = str(user['posicao'])
                            args.append(ranking_friends)  # args[7]
                    ranking_all = ranking_json['position_geral']
                    args.append(ranking_all)  # args[8]
            except KeyError:
                goal_is_set = False

        self.gui.status_bar.clearMessage()

        # Create a widget to display the statistics
        tree = QTreeWidget()
        if goal_is_set:
            if Qt_version >= 6:
                tree.setFixedHeight(145)
            else:
                tree.setFixedHeight(160)
        else:
            tree.setFixedHeight(120)
        tree.setStyleSheet(
            """
            QTreeWidget::item {
                padding: 0 0 2px 0;
            }
            """
        )
        layout = QVBoxLayout()
        layout.addWidget(tree)
        tree.setHeaderHidden(True)
        tree.setHeaderLabels([_('Col1'), _('Col2')])
        tree.header().setDefaultAlignment(Qt.AlignHCenter)

        # Ranking positions
        if goal_is_set:
            ranking = QTreeWidgetItem(tree)
            ranking.setText(0, _('Ranking'))
            ranking.setExpanded(True)
            if sys.version_info[0] < 3:
                d = 'º'.encode('iso-8859-1')
            else:
                d = chr(176)
            ranking_stats = QTreeWidgetItem(ranking, [_('Friends:  ') + args[7] + d,
                                                      _('    All:  ') + '{:n}'.format(args[8]) + d])

        # Read pages statistics
        pages = QTreeWidgetItem(tree)
        pages.setText(0, _('Pages'))
        pages.setExpanded(True)
        pages_stats = QTreeWidgetItem(pages, [_('Read:  ') + args[2], _('    To read:  ') + args[3]])

        # Reading pace statistics
        pace = QTreeWidgetItem(tree)
        pace.setText(0, _('Pace'))
        pace.setExpanded(True)
        pace_stats = QTreeWidgetItem(pace, [_('Current:  ') + args[4], _('    Ideal:  ') + str(args[5])])

        # Auto adjust column sizes
        tree.resizeColumnToContents(0)
        tree.resizeColumnToContents(1)

        # Avoid selection and focus
        tree.setSelectionMode(QAbstractItemView.NoSelection)
        tree.setFocusPolicy(Qt.NoFocus)

        end = timer()
        time = end - start
        print(datetime.timedelta(seconds=time))

        self.gui.status_bar.clearMessage()

        return tree, args

    def reading_goal_main(self):
        try:
            tree, args = SkoobSyncTools.reading_goal_statistics(self)
        except TypeError:
            return error_dialog(self.gui, _('No internet connection'), _('Seems like you have no internet access'),
                                show_copy_button=False, show=True)
        if tree is not None:
            SkoobSyncTools.show_statistics(self, tree, args)
        else:
            return

    def show_statistics(self, tree_widget, args):
        tool = self

        class ReadingGoalStatisticsDialog(Dialog):

            def __init__(self):
                Dialog.__init__(self, _('Reading Goal statistics'), 'plugin-stats-dialog', parent=tool.gui)
                self.setWindowIcon(get_icon('icon.png'))

            def setup_ui(self):
                layout = QVBoxLayout(self)
                group_box1 = QGroupBox()
                group_box1_layout = QVBoxLayout()
                group_box1.setLayout(group_box1_layout)
                layout.addWidget(group_box1)
                text = QLabel(_('You\'ve already read {0} out of {1} books').format(args[0][0], args[0][1]), self)
                text.setAlignment(QtCore.Qt.AlignCenter)
                group_box1_layout.addWidget(text)
                group_box1_layout.addSpacing(10)
                bar = QProgressBar()
                bar.setStyleSheet(
                    """
                    QProgressBar{
                        height: 2px;
                        border: 1.2px solid grey;
                        border-radius: 3px;
                        text-align: center;
                    }
                    """
                )
                if int(args[1]) > 100:
                    bar.setRange(0, int(args[1]))
                    bar.setFormat("%v%")
                bar.setValue(int(args[1]))
                group_box1_layout.addWidget(bar)
                layout.addSpacing(10)

                if goal_is_set:

                    bar_color = '#3061de' if is_dark_theme() else '#4d77e3'
                    text_color = '#ddd' if is_dark_theme() else '#000000'
                    background_color = '#121212' if is_dark_theme() else '#ffffff'

                    html_new = html.format(
                                        args[6]['jan'][0], args[6]['jan'][1],
                                        args[6]['fev'][0], args[6]['fev'][1],
                                        args[6]['mar'][0], args[6]['mar'][1],
                                        args[6]['abr'][0], args[6]['abr'][1],
                                        args[6]['mai'][0], args[6]['mai'][1],
                                        args[6]['jun'][0], args[6]['jun'][1],
                                        args[6]['jul'][0], args[6]['jul'][1],
                                        args[6]['ago'][0], args[6]['ago'][1],
                                        args[6]['set'][0], args[6]['set'][1],
                                        args[6]['out'][0], args[6]['out'][1],
                                        args[6]['nov'][0], args[6]['nov'][1],
                                        args[6]['dez'][0], args[6]['dez'][1],
                                        text_color, bar_color, background_color
                                        )

                if Qt_version >= 6 and goal_is_set:
                    group_box2 = QGroupBox()
                    group_box2_layout = QVBoxLayout()
                    group_box2.setLayout(group_box2_layout)
                    layout.addWidget(group_box2)
                    browser = QWebEngineView()
                    browser.page().setHtml(html_new)
                    group_box2_layout.addWidget(browser)
                    group_box2_layout.addSpacing(10)

                group_box3 = QGroupBox()
                group_box3_layout = QVBoxLayout()
                group_box3.setLayout(group_box3_layout)
                layout.addWidget(group_box3)
                group_box3_layout.addWidget(tree_widget)
                layout.addSpacing(10)
                group_box3.setFocus()

                h_layout = QHBoxLayout()
                layout.addLayout(h_layout)
                year_menu = QComboBox()
                current_year = datetime.date.today().year
                y = int(current_year)
                years_list = map(str, [y, y - 1, y - 2, y - 3, y - 4, y - 5, y - 6, y - 7, y - 8, y - 9])
                year_menu.addItems(years_list)
                year_menu.setCurrentText(str(current_year))
                year_menu.setFixedWidth(70)
                h_layout.addWidget(year_menu)

                def update_statistics():
                    # Show updating message
                    icon_path = os.path.join(config_dir, 'plugins', PLUGIN_NAME, 'images/loading.gif')
                    text.setText('<img src="' + icon_path + '" width="15" height="15"> ' + _('Updating...'))
                    bar.hide()

                    # Get updated values
                    new_tree_widget, new_args = tool.reading_goal_statistics(year_text=str(year_menu.currentText()))

                    # Update widget with the new values
                    text.setText(_('You\'ve already read {0} out of {1} books').format(new_args[0][0], new_args[0][1]))
                    if int(new_args[1]) > 100:
                        bar.setRange(0, int(new_args[1]))
                        bar.setFormat("%v%")
                    bar.setValue(int(new_args[1]))
                    bar.show()

                    if Qt_version >= 6 and goal_is_set:
                        new_bar_color = '#3061de' if is_dark_theme() else '#4d77e3'
                        new_text_color = '#ddd' if is_dark_theme() else '#000000'
                        new_background_color = '#121212' if is_dark_theme() else '#ffffff'

                        html_updated = html.format(
                            new_args[6]['jan'][0], new_args[6]['jan'][1],
                            new_args[6]['fev'][0], new_args[6]['fev'][1],
                            new_args[6]['mar'][0], new_args[6]['mar'][1],
                            new_args[6]['abr'][0], new_args[6]['abr'][1],
                            new_args[6]['mai'][0], new_args[6]['mai'][1],
                            new_args[6]['jun'][0], new_args[6]['jun'][1],
                            new_args[6]['jul'][0], new_args[6]['jul'][1],
                            new_args[6]['ago'][0], new_args[6]['ago'][1],
                            new_args[6]['set'][0], new_args[6]['set'][1],
                            new_args[6]['out'][0], new_args[6]['out'][1],
                            new_args[6]['nov'][0], new_args[6]['nov'][1],
                            new_args[6]['dez'][0], new_args[6]['dez'][1],
                            new_text_color, new_bar_color, new_background_color
                        )
                        browser.page().setHtml(html_updated)
                        group_box2.show()
                        self.setMaximumHeight(690)
                        self.setMaximumWidth(410)
                    else:
                        try:
                            group_box2.hide()
                            self.setMaximumHeight(310)
                            self.setMaximumWidth(410)
                        except:
                            pass

                    for i in reversed(range(group_box3_layout.count())):
                        group_box3_layout.itemAt(i).widget().setParent(None)

                    group_box3_layout.addWidget(new_tree_widget)

                year_menu.currentIndexChanged.connect(update_statistics)

        d = ReadingGoalStatisticsDialog()
        d.resize(QSize(410, 690))
        if Qt_version < 6:
            d.setMaximumHeight(350)  # 350
        else:
            d.setMaximumHeight(690)  # 665
        d.setMaximumWidth(410)
        d.exec_()


html = """
<!DOCTYPE html>
<html>
    <head>
        <title>Gráfico</title>
        <style type="text/css" title="override_css">
            @keyframes grow {{
                from {{height: 0%;}}
                to {{height: auto%;}}
                }}
            .jan {{
              height: {1}%;
              animation: grow 1s;
            }}
            .fev {{
              height: {3}%;
              animation: grow 1s;
            }}
            .mar {{
              height: {5}%;
              animation: grow 1s;
            }}
            .abr {{
              height: {7}%;
              animation: grow 1s;
            }}
            .mai {{
              height: {9}%;
              animation: grow 1s;
            }}
            .jun {{
              height: {11}%;
              animation: grow 1s;
            }}
            .jul {{
              height: {13}%;
              animation: grow 1s;
            }}
            .ago {{
              height: {15}%;
              animation: grow 1s;
            }}
            .set {{
              height: {17}%;
              animation: grow 1s;
            }}
            .out {{
              height: {19}%;
              animation: grow 1s;
            }}
            .nov {{
              height: {21}%;
              animation: grow 1s;
            }}
            .dez {{
              height: {23}%;
              animation: grow 1s;
            }}
            .box-container {{
                display: flex;
                flex-direction: column;
                max-width: 400px;
                padding: 0px;
                justify-content: center;
                align-content: center;
                align-items: center;
            }}
            .box-title {{
                font-weight: 500;
                text-align: center;
                width: 300px;
                padding: 8px 8px;
                font-size: 13px;
                color: {24};
            }}
            .box-chart-container {{
                margin-top: 40px;
                height: 200px;
                display: flex;
                flex-direction: row;
            }}
            .sk-bar-label-legend-container {{
                display: flex;
                flex-direction: inherit;
                height: 100%;
            }}
            .sk-bar-label-container {{
                display: flex;
                flex-direction: column;
                flex: 1;
                justify-content: flex-end;
            }}
            .sk-label-container {{
                font-size: 13px;
                font-weight: 500;
                display: flex;
                justify-content: center;
                margin: -40px 0 0 0;
                padding: 0px;
                color: {24};
            }}
            .sk-bar-container {{
                display: flex;
                flex-direction: column;
                justify-content: center;
                align-content: center;
                align-self: center;
                font-weight: normal;
                color: {24};
            }}
            .sk-bar-content {{
                background-color: {25};
                width: 10px;
                border-radius: 5px 5px 0px 0px;
                flex: 1;
            }}
            .sk-bar-legend-container {{
                display: flex;
                flex-direction: column;
                padding: 0px 2px;
                padding-top: 5px;
            }}
            .sk-legend-text {{
                font-size: 11px;
                font-weight: 500;
                color: {24};
                text-transform: uppercase;
            }}
        </style>
    </head>
    <body style="background-color:{26}">
        <div class="box-container">
            <span class="box-title">LIDOS POR MÊS</span>
            <div class="box-chart-container">
                <div class="sk-bar-label-legend-container">
                    <div class="sk-bar-label-container">
                        <div class="sk-label-container">
                            <span>{0}</span>
                        </div>
                        <div class="sk-bar-container jan">
                            <div class="sk-bar-content"/>
                        </div>
                    </div>
                    <div class="sk-bar-legend-container">
                        <span class="sk-legend-text">jan</span>
                    </div>
                </div>
                <div class="sk-bar-label-legend-container">
                    <div class="sk-bar-label-container">
                        <div class="sk-label-container">
                            <span>{2}</span>
                        </div>
                        <div class="sk-bar-container fev">
                            <div class="sk-bar-content"/>
                        </div>
                    </div>
                    <div class="sk-bar-legend-container">
                        <span class="sk-legend-text">fev</span>
                    </div>
                </div>
                <div class="sk-bar-label-legend-container">
                    <div class="sk-bar-label-container">
                        <div class="sk-label-container">
                            <span>{4}</span>
                        </div>
                        <div class="sk-bar-container mar">
                            <div class="sk-bar-content"/>
                        </div>
                    </div>
                    <div class="sk-bar-legend-container">
                        <span class="sk-legend-text">mar</span>
                    </div>
                </div>
                <div class="sk-bar-label-legend-container">
                    <div class="sk-bar-label-container">
                        <div class="sk-label-container">
                            <span>{6}</span>
                        </div>
                        <div class="sk-bar-container abr">
                            <div class="sk-bar-content"/>
                        </div>
                    </div>
                    <div class="sk-bar-legend-container">
                        <span class="sk-legend-text">abr</span>
                    </div>
                </div>
                <div class="sk-bar-label-legend-container">
                    <div class="sk-bar-label-container">
                        <div class="sk-label-container">
                            <span>{8}</span>
                        </div>
                        <div class="sk-bar-container mai">
                            <div class="sk-bar-content"/>
                        </div>
                    </div>
                    <div class="sk-bar-legend-container">
                        <span class="sk-legend-text">mai</span>
                    </div>
                </div>
                <div class="sk-bar-label-legend-container">
                    <div class="sk-bar-label-container">
                        <div class="sk-label-container">
                            <span>{10}</span>
                        </div>
                        <div class="sk-bar-container jun">
                            <div class="sk-bar-content"/>
                        </div>
                    </div>
                    <div class="sk-bar-legend-container">
                        <span class="sk-legend-text">jun</span>
                    </div>
                </div>
                <div class="sk-bar-label-legend-container">
                    <div class="sk-bar-label-container">
                        <div class="sk-label-container">
                            <span>{12}</span>
                        </div>
                        <div class="sk-bar-container jul">
                            <div class="sk-bar-content"/>
                        </div>
                    </div>
                    <div class="sk-bar-legend-container">
                        <span class="sk-legend-text">jul</span>
                    </div>
                </div>
                <div class="sk-bar-label-legend-container">
                    <div class="sk-bar-label-container">
                        <div class="sk-label-container">
                            <span>{14}</span>
                        </div>
                        <div class="sk-bar-container ago">
                            <div class="sk-bar-content"/>
                        </div>
                    </div>
                    <div class="sk-bar-legend-container">
                        <span class="sk-legend-text">ago</span>
                    </div>
                </div>
                <div class="sk-bar-label-legend-container">
                    <div class="sk-bar-label-container">
                        <div class="sk-label-container">
                            <span>{16}</span>
                        </div>
                        <div class="sk-bar-container set">
                            <div class="sk-bar-content"/>
                        </div>
                    </div>
                    <div class="sk-bar-legend-container">
                        <span class="sk-legend-text">set</span>
                    </div>
                </div>
                <div class="sk-bar-label-legend-container">
                    <div class="sk-bar-label-container">
                        <div class="sk-label-container">
                            <span>{18}</span>
                        </div>
                        <div class="sk-bar-container out">
                            <div class="sk-bar-content"/>
                        </div>
                    </div>
                    <div class="sk-bar-legend-container">
                        <span class="sk-legend-text">out</span>
                    </div>
                </div>
                <div class="sk-bar-label-legend-container">
                    <div class="sk-bar-label-container">
                        <div class="sk-label-container">
                            <span>{20}</span>
                        </div>
                        <div class="sk-bar-container nov">
                            <div class="sk-bar-content"/>
                        </div>
                    </div>
                    <div class="sk-bar-legend-container">
                        <span class="sk-legend-text">nov</span>
                    </div>
                </div>
                <div class="sk-bar-label-legend-container">
                    <div class="sk-bar-label-container">
                        <div class="sk-label-container">
                            <span>{22}</span>
                        </div>
                        <div class="sk-bar-container dez">
                            <div class="sk-bar-content"/>
                        </div>
                    </div>
                    <div class="sk-bar-legend-container">
                        <span class="sk-legend-text">dez</span>
                    </div>
                </div>
            </div>
        </div>
    </body>
</html>
"""
