#!/usr/bin/env python

from __future__ import absolute_import, division, print_function, unicode_literals
import six

__license__   = 'GPL v3'
__copyright__ = '2019-2023, Thiago Oliveira <thiago.eec@gmail.com>'
__docformat__ = 'restructuredtext en'

# Load translation files (.mo) on the folder 'translations'
load_translations()

# Standard libraries
import os
import base64
import sys
from functools import partial

# PyQt libraries
try:
    from qt.core import (Qt, QtCore, QWidget, QVBoxLayout, QLabel, QLineEdit, QGroupBox, QComboBox, QDialogButtonBox,
                         QGridLayout, QCheckBox, QApplication, QPushButton, QPixmap, QIcon, QDialog, QTabWidget)
except ImportError:
    from PyQt5.Qt import (Qt, QWidget, QVBoxLayout, QLabel, QLineEdit, QGroupBox, QComboBox, QDialogButtonBox,
                          QGridLayout, QCheckBox, QApplication, QPushButton, QPixmap, QIcon, QDialog, QTabWidget)
    from PyQt5 import QtCore

# Get PyQt version
Qt_version = int(QtCore.PYQT_VERSION_STR[0])

# Calibre libraries
from calibre.utils.config import JSONConfig, config_dir
from calibre.gui2 import gprefs
from calibre.gui2.keyboard import ShortcutConfig
from calibre.constants import numeric_version
from calibre_plugins.Skoob_Sync.__init__ import PLUGIN_VERSION, PLUGIN_NAME

# Check for QT_ENABLE_HIGHDPI_SCALING and QT_SCALE_FACTOR_ROUNDING_POLICY
try:
    high_dpi_scaling = os.environ['QT_ENABLE_HIGHDPI_SCALING']
except KeyError:
    try:
        high_dpi_scaling = os.environ['QT_SCALE_FACTOR_ROUNDING_POLICY']
    except KeyError:
        high_dpi_scaling = False

SUPPORTS_CREATE_CUSTOM_COLUMN = False
try:
    from calibre.gui2.preferences.create_custom_column import CreateNewCustomColumn
    SUPPORTS_CREATE_CUSTOM_COLUMN = True
except ImportError as e:
    SUPPORTS_CREATE_CUSTOM_COLUMN = False

def is_dark_theme():
    try:
        return QApplication.instance().is_dark_theme
    except:
        return False

# Custom columns map
DEFAULT_LOOKUP_READING_PROGRESS_COLUMN     = '#percentage'
DEFAULT_LOOKUP_STATUS_DATE_COLUMN          = '#read_date'
DEFAULT_LOOKUP_REVIEW_COLUMN               = '#review'
DEFAULT_LOOKUP_RATING_COLUMN               = '#custom_rating'
DEFAULT_LOOKUP_PAGE_COUNT_COLUMN           = '#pages'
DEFAULT_LOOKUP_READ_BOOKS_COLUMN           = '#read'
DEFAULT_LOOKUP_TAGS_COLUMN                 = '#user_tags'
DEFAULT_LOOKUP_PHYSICAL_COLUMN             = '#physical'
DEFAULTS = {
                DEFAULT_LOOKUP_READING_PROGRESS_COLUMN : {
                    'column_heading': _("% Read"),
                    'datatype': 'float',
                    'description': _("Reading progress"),
                    'number_format': '{:.0f}%',
                },
                DEFAULT_LOOKUP_STATUS_DATE_COLUMN : {
                    'column_heading': _("Status date"),
                    'datatype': 'datetime',
                    'description': _("Reading progress date"),
                },
                DEFAULT_LOOKUP_REVIEW_COLUMN : {
                    'column_heading': _("Review"),
                    'datatype': 'comments',
                    'description': _("Book review"),
                    'heading_position': 'above',
                    'interpret_as': 'html',
                },
                DEFAULT_LOOKUP_RATING_COLUMN : {
                    'column_heading': _("My rating"),
                    'datatype': 'rating',
                    'description': _("Book rating"),
                    'allow_half_stars': True,
                },
                DEFAULT_LOOKUP_PAGE_COUNT_COLUMN : {
                    'column_heading': _("Pages"),
                    'datatype': 'int',
                    'description': _("Book page count"),
                    'number_format': None,
                },
                DEFAULT_LOOKUP_READ_BOOKS_COLUMN : {
                    'column_heading': _("Read"),
                    'datatype': 'bool',
                    'description': _("Read status"),
                },
                DEFAULT_LOOKUP_TAGS_COLUMN: {
                    'column_heading': _("User tags"),
                    'datatype': 'text',
                    'description': _("User defined tags"),
                },
                DEFAULT_LOOKUP_PHYSICAL_COLUMN : {
                    'column_heading': _("Physical"),
                    'datatype': 'bool',
                    'description': _("Physical book"),
                },
            }

# This is where all preferences for this plugin will be stored
prefs = JSONConfig('plugins/Skoob_Sync')

# Set defaults
prefs.defaults['user'] = ''
if sys.version_info[0] < 3:
    prefs.defaults['password'] = ''
else:
    prefs.defaults['password'] = base64.b64encode(bytes('', 'utf-8'))
prefs.defaults['reading_progress_column'] = None
prefs.defaults['status_date_column'] = None
prefs.defaults['review_column'] = None
prefs.defaults['rating_column'] = None
prefs.defaults['page_count_column'] = None
prefs.defaults['read_books_column'] = None
prefs.defaults['tags_column'] = None
prefs.defaults['physical_column'] = None
prefs.defaults['rating_column'] = None
prefs.defaults['main_action_idx'] = 0
prefs.defaults['main_action'] = 'Download from Skoob'
prefs.defaults['light_themes_idx'] = 0
prefs.defaults['dark_themes_idx'] = 1
prefs.defaults['sync_reading_progress'] = True
prefs.defaults['sync_review'] = True
prefs.defaults['sync_rating'] = True
prefs.defaults['sync_page_count'] = True
prefs.defaults['sync_read_books'] = True
prefs.defaults['sync_update_added'] = True
prefs.defaults['sync_tags'] = False
prefs.defaults['user_id'] = ''

# Cleanup old prefs
try:
    del prefs.defaults['themes']
except KeyError:
    pass
try:
    del prefs.defaults['year_text']
except KeyError:
    pass
try:
    del prefs['year_text']
except KeyError:
    pass


def get_icon(icon_name):
    # Check to see whether the icon exists as a Calibre resource
    # This will enable skinning if the user stores icons within a folder like:
    # ...\AppData\Roaming\calibre\resources\images\Plugin Name\

    # Check for calibre theme
    if is_dark_theme():
        key_themes_idx = prefs['dark_themes_idx'] + 4
    else:
        key_themes_idx = prefs['light_themes_idx'] + 1

    # General icons
    general_icons = ['images/favorite.png', 'images/wanted.png', 'images/owned.png', 'images/lent.png',
                     'images/exchange.png']

    if icon_name not in general_icons:
        icon_name = 'images/theme' + str(key_themes_idx) + '/' + icon_name

    if numeric_version < (5, 99, 0):
        # First, look for the icon on the 'config_dir\resources\images\Plugin Name\' folder
        icon_path = os.path.join(config_dir, 'resources', 'images', PLUGIN_NAME, icon_name.replace('images/', ''))
        if os.path.exists(icon_path):
            pixmap = QPixmap()
            pixmap.load(icon_path)
            return QIcon(pixmap)
        else:
            # Then, look for it on the theme folder
            if not icon_name.startswith('images/'):  # Image does not come with the zip file
                icon_path = os.path.join(config_dir, 'resources', 'images', icon_name)
                if os.path.exists(icon_path):
                    pixmap = QPixmap()
                    pixmap.load(icon_path)
                    return QIcon(pixmap)
                # Use the general icon, in case there is no themed version of it
                else:
                    icon = QIcon(I(icon_name))
                    return icon
    else:
        # First, look for the themed icon (Qt resource files)
        tc = 'dark' if is_dark_theme() else 'light'
        sq, ext = os.path.splitext(icon_name)
        sq = '{0}-for-{1}-theme{2}'.format(sq, tc, ext)
        icon = QIcon.ic(PLUGIN_NAME + '/' + sq.replace('images/', ''))
        if icon.isNull():
            # Then, look for the regular icon
            icon = QIcon.ic(PLUGIN_NAME + '/' + icon_name.replace('images/', ''))
            if icon.isNull():
                # Then, look for it on general icons (Qt resource files)
                if not icon_name.startswith('images/'):  # Image does not come with the zip file
                    return QIcon.ic(icon_name)
            else:
                return icon
        else:
            return icon

    # As we did not find an icon elsewhere, look within our zip resources
    return get_icons(icon_name)


class ConfigWidget(QWidget):

    def __init__(self, gui, plugin_action, tab=0):
        QWidget.__init__(self)
        self.gui = gui
        self.plugin_action = plugin_action
        layout = QVBoxLayout(self)
        self.setLayout(layout)

        tab_widget = QTabWidget(self)
        layout.addWidget(tab_widget)

        self.sync_tab = SyncTab(self)
        self.columns_tab = ColumnsTab(self)
        tab_widget.addTab(self.sync_tab, ' ' + _('S&ync') + ' ')
        tab_widget.addTab(self.columns_tab, ' ' + _('&Columns') + ' ')
        tab_widget.setCurrentIndex(tab)

        self.supports_create_custom_column = SUPPORTS_CREATE_CUSTOM_COLUMN
        self.must_restart = False
        self._get_create_new_custom_column_instance = None

        self.resize(self.sizeHint())

    def refresh_widget(self):
        box_enabled = (self.sync_tab.reading_progress_check.isChecked() and
                       (self.columns_tab._read_books_column_combo.get_selected_column() != ''))
        self.sync_tab.read_books_check.setEnabled(box_enabled)
        if not box_enabled:
            self.sync_tab.read_books_check.setChecked(False)
            self.sync_tab.read_books_check.setStyleSheet('color: gray')
        else:
            self.sync_tab.read_books_check.setStyleSheet('')
        self.update()

    def edit_shortcuts(self):
        # Force the menus to be rebuilt immediately, so we have all our actions registered
        # self.plugin_action.rebuild_menus()
        d = KeyboardConfigDialog(self.plugin_action.gui, PLUGIN_NAME)
        if d.exec_() == d.Accepted:
            self.plugin_action.gui.keyboard.finalize()

    def get_custom_columns(self, column_types):
        custom_columns = self.plugin_action.gui.library_view.model().custom_columns
        available_columns = {}
        for key, column in six.iteritems(custom_columns):
            typ = column['datatype']
            if typ in column_types:
                available_columns[key] = column
        return available_columns

    def create_custom_column(self, combo_box, custom_columns, lookup_name=None, is_multiple=False):
        display_params = {
            'description': DEFAULTS[lookup_name]['description'],
        }

        datatype = DEFAULTS[lookup_name]['datatype']

        if datatype in ['int', 'float']:
            display_params['number_format'] = DEFAULTS[lookup_name]['number_format']
        elif datatype == 'comments':
            display_params['heading_position'] = DEFAULTS[lookup_name]['heading_position']
            display_params['interpret_as'] = DEFAULTS[lookup_name]['interpret_as']
        elif datatype == 'rating':
            display_params['allow_half_stars'] = DEFAULTS[lookup_name]['allow_half_stars']

        column_heading = DEFAULTS[lookup_name]['column_heading']
        new_lookup_name = lookup_name
        create_new_custom_column_instance = self.get_create_new_custom_column_instance()
        result = create_new_custom_column_instance.create_column(new_lookup_name, column_heading, datatype,
                                                                 is_multiple,
                                                                 display=display_params,
                                                                 generate_unused_lookup_name=True,
                                                                 freeze_lookup_name=False)
        if result[0] == CreateNewCustomColumn.Result.COLUMN_ADDED:
            combo_box.populate_combo(custom_columns, result[1], column_heading)
            self.gui.must_restart = True

    # @property
    def get_create_new_custom_column_instance(self):
        if self._get_create_new_custom_column_instance is None and self.supports_create_custom_column:
            self._get_create_new_custom_column_instance = CreateNewCustomColumn(self.plugin_action.gui)
        return self._get_create_new_custom_column_instance

    def about(self):
        text = '<hr/>' \
               '<p>' + _('Created by Thiago Oliveira')+'</p>' \
               '<p>' + _('For help using this plugin, click on the \'?\' icon on the upper right corner '
                         'of the configuration dialog. Then, hover the mouse over the options, and click '
                         'when the cursor changes.') + '</p>' \
               '<p>' + _('You can also find instructions in the plugin thread at ') + \
               '<a href="https://www.mobileread.com/forums/showthread.php?t=322945">Mobileread.com</a>.</p>' + \
               '<p>' + _('If any doubt still persists, feel free to ask there. Suggestions are welcomed too.') + '</p>'

        about_text = "{0}{1}".format(PLUGIN_NAME + ' v' + PLUGIN_VERSION, text)
        AboutDialog(self.plugin_action.gui, get_icon('icon.png'), about_text).exec_()

    def save_settings(self):
        prefs['user'] = self.sync_tab.user.text()
        if sys.version_info[0] < 3:
            prefs['password'] = self.sync_tab.password.text().encode('base64')
        else:
            prefs['password'] = base64.b64encode(bytes(self.sync_tab.password.text(), 'utf-8'))
        prefs['reading_progress_column'] = self.columns_tab._reading_progress_column_combo.get_selected_column()
        prefs['status_date_column'] = self.columns_tab._status_date_column_combo.get_selected_column()
        prefs['review_column'] = self.columns_tab._review_column_combo.get_selected_column()
        prefs['rating_column'] = self.columns_tab._rating_column_combo.get_selected_column()
        prefs['page_count_column'] = self.columns_tab._page_count_column_combo.get_selected_column()
        prefs['read_books_column'] = self.columns_tab._read_books_column_combo.get_selected_column()
        prefs['tags_column'] = self.columns_tab._tags_column_combo.get_selected_column()
        prefs['physical_column'] = self.columns_tab._physical_column_combo.get_selected_column()
        prefs['main_action_idx'] = self.sync_tab.main_action_combo.currentIndex()
        prefs['main_action'] = self.sync_tab.main_action_combo.currentText()
        prefs['light_themes_idx'] = self.sync_tab.light_themes_column_combo.currentIndex()
        prefs['light_themes'] = self.sync_tab.light_themes_column_combo.currentText()
        prefs['dark_themes_idx'] = self.sync_tab.dark_themes_column_combo.currentIndex()
        prefs['dark_themes'] = self.sync_tab.dark_themes_column_combo.currentText()
        prefs['sync_reading_progress'] = self.sync_tab.reading_progress_check.isChecked()
        prefs['sync_review'] = self.sync_tab.review_check.isChecked()
        prefs['sync_rating'] = self.sync_tab.rating_check.isChecked()
        prefs['sync_page_count'] = self.sync_tab.page_count_check.isChecked()
        prefs['sync_read_books'] = self.sync_tab.read_books_check.isChecked()
        prefs['sync_update_added'] = self.sync_tab.update_added_check.isChecked()
        prefs['sync_tags'] = self.sync_tab.tags_check.isChecked()


class SyncTab(QWidget):

    def __init__(self, parent_dialog):
        self.parent_dialog = parent_dialog
        QWidget.__init__(self)
        layout = QGridLayout()
        self.setLayout(layout)

        # Check for calibre theme
        if is_dark_theme():
            self.key_themes_idx = prefs['dark_themes_idx'] + 4
        else:
            self.key_themes_idx = prefs['light_themes_idx'] + 1

        # Help information
        help_ico_src = os.path.join(config_dir, 'plugins', PLUGIN_NAME, 'images',
                                    'theme' + str(self.key_themes_idx), 'help.png')
        help_img_src = '<img src="' + help_ico_src + '" width="25" height="25"> '
        sync_options_whatsthis = \
            '<h4 style="vertical-align:top">' + help_img_src + _('Sync options') + '</h4>' + \
            '<p>' + _('These are the possible sync actions between Skoob and calibre. For each action, you must '
                      'specify a corresponding custom column, in the <b>Columns</b> tab. You can sync up to 50 books '
                      'at a time.') + '</p>' + \
            '<p>' + _('All the actions sync both way, except for the <i>Page count</i> action, which only downloads '
                      'from Skoob. See the detailed info below.') + '</p>' \
            '<ul>' \
            '<li>' + _('<b>Reading progress</b>: sync the book reading percentage and status date.') + '</li>'\
            '<li>' + _('<b>Review</b>: you can format your text using calibre HTML editor. To open '
                       'the editor, just click on your <i>Review</i> cell in the main book list. '
                       'To set a title for your review, use a <i>Heading 4</i> tag. Ex.: &lt;h4&gt;'
                       'My title&lt;/h4&gt;.') + '</li>' + \
            '<li>' + _('<b>Rating</b>: 1 to 5 stars. You can use half stars.') + '</li>' + \
            '<li>' + _('<b>Page count</b>: generally, the page count from Skoob is for the '
                       'physical edition of the book.') + '</li>' + \
            '<li>' + _('<b>User tags</b>: sync both your <i>Public</i> and <i>Shelf</i> tags. '
                       'This option makes syncing a bit slower. When checked, '
                       'syncing is limited to 20 books at a time.') + '</li>' \
            '</ul>'

        other_options_whatsthis = \
            '<h4 style="vertical-align:top">' + help_img_src + _('Other options') + '</h4>' + \
            '<p>' + _('These options define extra actions when syncing with Skoob. ') + '</p>' \
            '<ul>' \
            '<li>' + _('<b>Automatically mark books as read</b>: sets your <i>Read books</i> column to <i>Yes</i> when '
                       'your reading progress reaches 100%. When downloading from Skoob, this info comes from your '
                       'reading history. When uploading, it reads from your <i>Reading progress</i> column.') + '</li>'\
            '<li>' + _('<b>Update books added to Skoob</b>: when using the <i>Add book</i> action, in addition to '
                       'adding the book to your <i>Reading</i> shelf on Skoob, your reading status will also be update '
                       '(like the <i>Upload to Skoob</i> action).') + '</li>'\
            '</ul>'

        # --- Credentials ---
        credentials_group_box = QGroupBox(_('Credentials'), self)
        layout.addWidget(credentials_group_box, 0, 0, 1, 1)
        credentials_group_box_layout = QVBoxLayout()
        credentials_group_box.setLayout(credentials_group_box_layout)

        # User box
        self.user_label = QLabel()
        if Qt_version >= 6:
            user_img = 'user'
        else:
            user_img = 'user2'
        ico_src = os.path.join(config_dir, 'plugins', PLUGIN_NAME, 'images', 'theme' + str(self.key_themes_idx),
                               user_img)
        if Qt_version >= 6:
            if high_dpi_scaling in ('1', 'PassThrough', False):
                img_src = '<img src="' + ico_src + '" width="11" height="11"> '
            else:
                img_src = '<img src="' + ico_src + '" width="13" height="13"> '
        else:
            img_src = '<img src="' + ico_src + '"> '
        self.user_label.setText(img_src + _('&User'))
        layout.addWidget(self.user_label)
        credentials_group_box_layout.addWidget(self.user_label)
        self.user = QLineEdit(self)
        self.user.setMinimumWidth(155)  # 260
        self.user.setText(prefs['user'])
        layout.addWidget(self.user)
        self.user_label.setBuddy(self.user)
        credentials_group_box_layout.addWidget(self.user)

        # Password box
        self.password_label = QLabel()
        if Qt_version >= 6:
            password_img = 'password'
        else:
            password_img = 'password2'
        ico_src = os.path.join(config_dir, 'plugins', PLUGIN_NAME, 'images', 'theme' + str(self.key_themes_idx),
                               password_img)
        if Qt_version >= 6:
            if high_dpi_scaling in ('1', 'PassThrough', False):
                img_src = '<img src="' + ico_src + '" width="7" height="10"> '
            else:
                img_src = '<img src="' + ico_src + '" width="9" height="12"> '
        else:
            img_src = '<img src="' + ico_src + '"> '
        self.password_label.setText(img_src + _('Pass&word'))
        layout.addWidget(self.password_label)
        credentials_group_box_layout.addWidget(self.password_label)
        self.password = QLineEdit(self)
        self.password.setMinimumWidth(155)  # 185
        self.password.setEchoMode(QLineEdit.Password)
        if sys.version_info[0] < 3:
            self.password.setText(prefs['password'].decode('base64'))
        else:
            self.password.setText(base64.b64decode(prefs['password']).decode('utf-8'))
        layout.addWidget(self.password)
        self.password_label.setBuddy(self.password)
        credentials_group_box_layout.addWidget(self.password)

        # --- Sync Options ---
        sync_options_group_box = QGroupBox(_('Sync options'), self)
        sync_options_group_box.setWhatsThis(sync_options_whatsthis)
        layout.addWidget(sync_options_group_box, 0, 1, 1, 1)
        sync_options_group_box_layout = QVBoxLayout()
        sync_options_group_box.setLayout(sync_options_group_box_layout)

        # Reading progress checkbox
        self.reading_progress_check = QCheckBox(_('&Reading progress'), self)
        self.reading_progress_check.setToolTip(_('Sync reading progress with Skoob'))
        sync_options_group_box_layout.addWidget(self.reading_progress_check)
        # Load the checkbox with the current preference setting
        self.reading_progress_check.setChecked(prefs['sync_reading_progress'])

        # Review checkbox
        self.review_check = QCheckBox(_('R&eview'), self)
        self.review_check.setToolTip(_('Sync review with Skoob'))
        sync_options_group_box_layout.addWidget(self.review_check)
        # Load the checkbox with the current preference setting
        self.review_check.setChecked(prefs['sync_review'])

        # Rating checkbox
        self.rating_check = QCheckBox(_('Rati&ng'), self)
        self.rating_check.setToolTip(_('Sync book rating with Skoob'))
        sync_options_group_box_layout.addWidget(self.rating_check)
        # Load the checkbox with the current preference setting
        self.rating_check.setChecked(prefs['sync_rating'])

        # Page count checkbox
        self.page_count_check = QCheckBox(_('&Page count'), self)
        self.page_count_check.setToolTip(_('Get page count from Skoob'))
        sync_options_group_box_layout.addWidget(self.page_count_check)
        # Load the checkbox with the current preference setting
        self.page_count_check.setChecked(prefs['sync_page_count'])

        # User tags checkbox
        self.tags_check = QCheckBox(_('User ta&gs'), self)
        self.tags_check.setToolTip(_('Get user defined tags for the book (Shelf and Public)'))
        sync_options_group_box_layout.addWidget(self.tags_check)
        # Load the checkbox with the current preference setting
        self.tags_check.setChecked(prefs['sync_tags'])

        # --- Customize interface ---
        customize_group_box = QGroupBox(_('Customize interface'))
        layout.addWidget(customize_group_box, 1, 0, 1, 2)
        customize_group_box_layout = QGridLayout()
        customize_group_box.setLayout(customize_group_box_layout)

        # Main action combobox
        self.main_action_box_label = QLabel(_('&Main button:'), self)
        tooltip = _('Select which action will be used for the main button on calibre GUI')
        self.main_action_box_label.setToolTip(tooltip)
        self.main_action_combo = QComboBox()
        self.main_action_combo.setToolTip(tooltip)
        self.main_action_combo.setMinimumWidth(120)
        self.main_action_combo.addItems({_('Download from Skoob'), _('Upload to Skoob')})
        self.main_action_combo.model().sort(0)
        self.main_action_box_label.setBuddy(self.main_action_combo)
        customize_group_box_layout.addWidget(self.main_action_box_label, 0, 0)
        customize_group_box_layout.addWidget(self.main_action_combo, 0, 1)
        self.main_action_combo.setCurrentIndex(prefs['main_action_idx'])

        # Light theme combobox
        self.light_themes_box_label = QLabel(_('&Light theme:'), self)
        tooltip = _('Choose your icons for light theme (automatic change)')
        self.light_themes_box_label.setToolTip(tooltip)
        self.light_themes_column_combo = QComboBox()
        self.light_themes_column_combo.setToolTip(tooltip)
        self.light_themes_column_combo.setMinimumWidth(120)
        self.light_themes_column_combo.addItems({_('1. Blue'), _('2. Grey'), _('3. Grey and Yellow')})
        self.light_themes_column_combo.model().sort(0)
        self.light_themes_box_label.setBuddy(self.light_themes_column_combo)
        customize_group_box_layout.addWidget(self.light_themes_box_label, 1, 0)
        customize_group_box_layout.addWidget(self.light_themes_column_combo, 1, 1)
        self.light_themes_column_combo.setCurrentIndex(prefs['light_themes_idx'])

        # Dark theme combobox
        self.dark_themes_box_label = QLabel(_('&Dark Theme:'), self)
        tooltip = _('Choose your icons for dark theme (automatic change)')
        self.dark_themes_box_label.setToolTip(tooltip)
        self.dark_themes_column_combo = QComboBox()
        self.dark_themes_column_combo.setToolTip(tooltip)
        self.dark_themes_column_combo.setMinimumWidth(120)
        self.dark_themes_column_combo.addItems({_('1. Blue'), _('2. Light grey and Yellow')})
        self.dark_themes_column_combo.model().sort(0)
        self.dark_themes_box_label.setBuddy(self.dark_themes_column_combo)
        customize_group_box_layout.addWidget(self.dark_themes_box_label, 2, 0)
        customize_group_box_layout.addWidget(self.dark_themes_column_combo, 2, 1)
        self.dark_themes_column_combo.setCurrentIndex(prefs['dark_themes_idx'])

        # --- Other options ---
        other_options_group_box = QGroupBox(_('Other options'))
        other_options_group_box.setWhatsThis(other_options_whatsthis)
        layout.addWidget(other_options_group_box, 2, 0, 1, 2)
        other_options_group_box_layout = QGridLayout()
        other_options_group_box.setLayout(other_options_group_box_layout)

        # Read books checkbox
        self.read_books_check = QCheckBox(_('&Automatically mark books as read'), self)
        self.read_books_check.setToolTip(_('<p style="white-space:pre">Use the <i>Read books</i> column to mark books'
                                           ' as finished</p>'))
        other_options_group_box_layout.addWidget(self.read_books_check, 0, 0)
        # Load the checkbox with the current preference setting
        self.read_books_check.setChecked(prefs['sync_read_books'])
        # Enables / disables the read books check box
        self.key_sync_reading_progress = prefs['sync_reading_progress']
        self.key_read_books_column = prefs['read_books_column']
        self.reading_progress_check.stateChanged.connect(self.parent_dialog.refresh_widget)
        box_enabled = self.reading_progress_check.isChecked() and (self.key_read_books_column != '')
        self.read_books_check.setEnabled(box_enabled)
        if not box_enabled:
            self.read_books_check.setChecked(False)
            self.read_books_check.setStyleSheet('color: gray')
        else:
            self.read_books_check.setStyleSheet('')

        # Update added checkbox
        self.update_added_check = QCheckBox(_('Update b&ooks added to Skoob'), self)
        self.update_added_check.setToolTip(_('Automatically update status of books added to Skoob'))
        other_options_group_box_layout.addWidget(self.update_added_check, 1, 0)
        # Load the checkbox with the current preference setting
        self.update_added_check.setChecked(prefs['sync_update_added'])


class ColumnsTab(QWidget):

    def __init__(self, parent_dialog):
        self.parent_dialog = parent_dialog
        QWidget.__init__(self)
        layout = QGridLayout()
        self.setLayout(layout)

        self.add_icon = get_icon('add_column.png')
        self.supports_create_custom_column = SUPPORTS_CREATE_CUSTOM_COLUMN

        # Check for calibre theme
        if is_dark_theme():
            self.key_themes_idx = prefs['dark_themes_idx'] + 4
        else:
            self.key_themes_idx = prefs['light_themes_idx'] + 1

        # Help information
        help_ico_src = os.path.join(config_dir, 'plugins', PLUGIN_NAME, 'images',
                                    'theme' + str(self.key_themes_idx), 'help.png')
        help_img_src = '<img src="' + help_ico_src + '" width="25" height="25"> '
        if self.supports_create_custom_column:
            add_column_ico_src = os.path.join(config_dir, 'plugins', PLUGIN_NAME, 'images',
                                              'theme' + str(self.key_themes_idx), 'add_column')
            add_column_img_src = '<img src="' + add_column_ico_src + \
                                 '" style="vertical-align:bottom" width="20" height="20"> '
            whatsthis_add_column = \
                '<p>' + _('To create a new custom column, just click on the {0} icon. '
                          'In the opened dialog, if you want, edit the <i>Column heading</i> and <i>Description</i>, '
                          'and press OK. Before closing this dialog, repeat the process for each field you want to '
                          'create a new custom column for. You will be prompted to restart calibre after you are done '
                          'creating the custom columns.').format(add_column_img_src) + '</p>'
        else:
            whatsthis_add_column = \
                '<p>' + _('To create a new custom column, you must close this dialog and go to '
                          'Preferences > Add your own columns > Add custom column. Before closing, look at the '
                          'tooltip for each field you want to create a custom column for. It will indicate the '
                          '<i>Column type</i> you must choose when creating the columns. You will be prompted to '
                          'restart calibre after you are done creating the custom columns.') + '</p>'
        custom_columns_whatsthis = \
            '<h4 style="vertical-align:top">' + help_img_src + _('Custom columns') + '</h4>' + \
            '<p>' + _('Once you defined your syncing actions in the <b>Sync</b> tab, you need to assign the custom '
                      'columns that will store the corresponding information. This must be done for each field you '
                      'want to sync.') + '</p>' \
            '<p>' + _('Each field has its specific data type, so it only lists the suitable columns available. '
                      'If you don\'t want to use any of the existing columns, you can create a new custom column '
                      'matching the data type for that field.') + '</p>' \
            + whatsthis_add_column + \
            '<p>' + _('Some items can be synced to standard calibre fields, although not recommended. See why:') + '</p>' \
            '<ul>' \
            '<li>' + _('<b>comments</b>: when you use this field to store your <i>Review</i>, the plugin will add it '
                       'at the end of existing comments, with a <i>## Skoob ##</i> separator. You must remember that '
                       'if you download new metadata, your review might get overwritten.') + '</li>' \
            '<li>' + _('<b>rating</b>: when downloading new metadata, your personal book rating might get overwritten '
                       'by a public rating coming from your metadata source.') + '</li>' \
            '<li>' + _('<b>tags</b>: you personal tags might get mixed with newly added metadata.') + '</li>' \
            '</ul>'

        # --- Custom Columns ---
        custom_columns_group_box = QGroupBox(_('Custom columns'), self)
        custom_columns_group_box.setWhatsThis(custom_columns_whatsthis)
        layout.addWidget(custom_columns_group_box, 0, 0, 1, 1)
        custom_columns_group_box_layout = QGridLayout()
        custom_columns_group_box.setLayout(custom_columns_group_box_layout)

        # Reading progress combobox
        self.reading_progress_box_label = QLabel(_('Rea&ding progress:'), self)
        tooltip = _('Select a custom column to store the reading progress. Type: floating point numbers /integers.')
        self.reading_progress_box_label.setToolTip(tooltip)
        self.reading_progress_box = QComboBox()
        reading_progress_custom_columns = self.parent_dialog.get_custom_columns(['float', 'int'])
        self._reading_progress_column_combo = CustomColumnComboBox(self, reading_progress_custom_columns,
                                                                   initial_items=[''])
        self._reading_progress_column_combo.setToolTip(tooltip)
        self._reading_progress_column_combo.setMinimumWidth(230)  # 250
        self.reading_progress_box_label.setBuddy(self._reading_progress_column_combo)
        custom_columns_group_box_layout.addWidget(self.reading_progress_box_label, 0, 0)
        custom_columns_group_box_layout.addWidget(self._reading_progress_column_combo, 0, 1)
        self._reading_progress_column_combo_index = self._reading_progress_column_combo\
            .findText(prefs['reading_progress_column'], QtCore.Qt.MatchStartsWith)
        if self._reading_progress_column_combo_index == -1:
            self._reading_progress_column_combo.setCurrentIndex(0)
        else:
            self._reading_progress_column_combo.setCurrentIndex(self._reading_progress_column_combo_index)

        # Reading progress - add custom column button
        if self.supports_create_custom_column:
            self.reading_progress_add_button = QPushButton(self.add_icon, '', self)
            self.reading_progress_add_button.setToolTip(_('Create new column for Reading progress'))
            self.reading_progress_add_button.setMaximumWidth(50)
            custom_columns_group_box_layout.addWidget(self.reading_progress_add_button, 0, 2)
            self.reading_progress_add_button.clicked.connect(partial(self.parent_dialog.create_custom_column,
                                                                     self._reading_progress_column_combo,
                                                                     reading_progress_custom_columns,
                                                                     DEFAULT_LOOKUP_READING_PROGRESS_COLUMN, False))

        # Status date combobox
        self.status_date_box_label = QLabel(_('&Status date:'), self)
        tooltip = _('Select a custom column to store the reading progress date. Type: date.')
        self.status_date_box_label.setToolTip(tooltip)
        self.status_date_box = QComboBox()
        status_date_custom_columns = self.parent_dialog.get_custom_columns(['datetime'])
        self._status_date_column_combo = CustomColumnComboBox(self, status_date_custom_columns, initial_items=[''])
        self._status_date_column_combo.setToolTip(tooltip)
        self._status_date_column_combo.setMinimumWidth(230)  # 120
        self.status_date_box_label.setBuddy(self._status_date_column_combo)
        custom_columns_group_box_layout.addWidget(self.status_date_box_label, 1, 0)
        custom_columns_group_box_layout.addWidget(self._status_date_column_combo, 1, 1)
        self._status_date_column_combo_index = self._status_date_column_combo.\
            findText(prefs['status_date_column'], QtCore.Qt.MatchStartsWith)
        if self._status_date_column_combo_index == -1:
            self._status_date_column_combo.setCurrentIndex(0)
        else:
            self._status_date_column_combo.setCurrentIndex(self._status_date_column_combo_index)

        # Status date - add custom column button
        if self.supports_create_custom_column:
            self.status_date_add_button = QPushButton(self.add_icon, '', self)
            self.status_date_add_button.setToolTip(_('Create new column for Status date'))
            self.status_date_add_button.setMaximumWidth(50)
            custom_columns_group_box_layout.addWidget(self.status_date_add_button, 1, 2)
            self.status_date_add_button.clicked.connect(partial(self.parent_dialog.create_custom_column,
                                                                     self._status_date_column_combo,
                                                                     status_date_custom_columns,
                                                                     DEFAULT_LOOKUP_STATUS_DATE_COLUMN, False))

        # Review combobox
        self.review_box_label = QLabel(_('Re&view:'), self)
        tooltip = _('Select a custom column to store the book review. Type: long text, like comments.')
        self.review_box_label.setToolTip(tooltip)
        self.review_box = QComboBox()
        review_custom_columns = self.parent_dialog.get_custom_columns(['comments'])
        self._review_column_combo = CustomColumnComboBox(self, review_custom_columns, initial_items=['comments'])
        self._review_column_combo.setToolTip(tooltip)
        self._review_column_combo.setMinimumWidth(230)  # 120
        self.review_box_label.setBuddy(self._review_column_combo)
        custom_columns_group_box_layout.addWidget(self.review_box_label, 2, 0)
        custom_columns_group_box_layout.addWidget(self._review_column_combo, 2, 1)
        self._review_column_combo_index = self._review_column_combo.findText(
            prefs['review_column'], QtCore.Qt.MatchStartsWith)
        if self._review_column_combo_index == -1:
            self._review_column_combo.setCurrentIndex(0)
        else:
            self._review_column_combo.setCurrentIndex(self._review_column_combo_index)

        # Review - add custom column button
        if self.supports_create_custom_column:
            self.review_add_button = QPushButton(self.add_icon, '', self)
            self.review_add_button.setToolTip(_('Create new column for Review'))
            self.review_add_button.setMaximumWidth(50)
            custom_columns_group_box_layout.addWidget(self.review_add_button, 2, 2)
            self.review_add_button.clicked.connect(partial(self.parent_dialog.create_custom_column,
                                                                self._review_column_combo,
                                                                review_custom_columns,
                                                                DEFAULT_LOOKUP_REVIEW_COLUMN, False))

        # Rating combobox
        self.rating_box_label = QLabel(_('&Rating:'), self)
        tooltip = _('Select a custom column to store the book rating. Type: ratings.')
        self.rating_box_label.setToolTip(tooltip)
        self.rating_box = QComboBox()
        rating_custom_columns = self.parent_dialog.get_custom_columns(['rating'])
        self._rating_column_combo = CustomColumnComboBox(self, rating_custom_columns, initial_items=['rating'])
        self._rating_column_combo.setToolTip(tooltip)
        self._rating_column_combo.setMinimumWidth(230)  # 120
        self.rating_box_label.setBuddy(self._rating_column_combo)
        custom_columns_group_box_layout.addWidget(self.rating_box_label, 3, 0)
        custom_columns_group_box_layout.addWidget(self._rating_column_combo, 3, 1)
        self._rating_column_combo_index = self._rating_column_combo.findText(
            prefs['rating_column'], QtCore.Qt.MatchStartsWith)
        if self._rating_column_combo_index == -1:
            self._rating_column_combo.setCurrentIndex(0)
        else:
            self._rating_column_combo.setCurrentIndex(self._rating_column_combo_index)

        # Rating - add custom column button
        if self.supports_create_custom_column:
            self.rating_add_button = QPushButton(self.add_icon, '', self)
            self.rating_add_button.setToolTip(_('Create new column for Rating'))
            self.rating_add_button.setMaximumWidth(50)
            custom_columns_group_box_layout.addWidget(self.rating_add_button, 3, 2)
            self.rating_add_button.clicked.connect(partial(self.parent_dialog.create_custom_column,
                                                           self._rating_column_combo,
                                                           rating_custom_columns,
                                                           DEFAULT_LOOKUP_RATING_COLUMN, False))

        # Page count combobox
        self.page_count_box_label = QLabel(_('Page &count:'), self)
        tooltip = _('Select a custom column to store the page count. Type: integers.')
        self.page_count_box_label.setToolTip(tooltip)
        self.page_count_box = QComboBox()
        page_count_custom_columns = self.parent_dialog.get_custom_columns(['int'])
        self._page_count_column_combo = CustomColumnComboBox(self, page_count_custom_columns, initial_items=[''])
        self._page_count_column_combo.setToolTip(tooltip)
        self._page_count_column_combo.setMinimumWidth(230)  # 120
        self.page_count_box_label.setBuddy(self._page_count_column_combo)
        custom_columns_group_box_layout.addWidget(self.page_count_box_label, 4, 0)
        custom_columns_group_box_layout.addWidget(self._page_count_column_combo, 4, 1)
        self._page_count_column_combo_index = self._page_count_column_combo.findText(
            prefs['page_count_column'], QtCore.Qt.MatchStartsWith)
        if self._page_count_column_combo_index == -1:
            self._page_count_column_combo.setCurrentIndex(0)
        else:
            self._page_count_column_combo.setCurrentIndex(self._page_count_column_combo_index)

        # Page count - add custom column button
        if self.supports_create_custom_column:
            self.page_count_add_button = QPushButton(self.add_icon, '', self)
            self.page_count_add_button.setToolTip(_('Create new column for Page count'))
            self.page_count_add_button.setMaximumWidth(50)
            custom_columns_group_box_layout.addWidget(self.page_count_add_button, 4, 2)
            self.page_count_add_button.clicked.connect(partial(self.parent_dialog.create_custom_column,
                                                           self._page_count_column_combo,
                                                           page_count_custom_columns,
                                                       DEFAULT_LOOKUP_PAGE_COUNT_COLUMN, False))

        # User tags combobox
        self.tags_box_label = QLabel(_('User &tags:'), self)
        tooltip = _('Select a custom column to store the user tags for a book. Type: comma separated text.')
        self.tags_box_label.setToolTip(tooltip)
        self.tags_box = QComboBox()
        tags_custom_columns = self.parent_dialog.get_custom_columns(['text'])
        self._tags_column_combo = CustomColumnComboBox(self, tags_custom_columns, initial_items=['tags'])
        self._tags_column_combo.setToolTip(tooltip)
        self._tags_column_combo.setMinimumWidth(230)  # 120
        self.tags_box_label.setBuddy(self._tags_column_combo)
        custom_columns_group_box_layout.addWidget(self.tags_box_label, 5, 0)
        custom_columns_group_box_layout.addWidget(self._tags_column_combo, 5, 1)
        self._tags_column_combo_index = self._tags_column_combo.findText(
            prefs['tags_column'], QtCore.Qt.MatchStartsWith)
        if self._tags_column_combo_index == -1:
            self._tags_column_combo.setCurrentIndex(0)
        else:
            self._tags_column_combo.setCurrentIndex(self._tags_column_combo_index)

        # User tags - add custom column button
        if self.supports_create_custom_column:
            self.tags_add_button = QPushButton(self.add_icon, '', self)
            self.tags_add_button.setToolTip(_('Create new column for User tags'))
            self.tags_add_button.setMaximumWidth(50)
            custom_columns_group_box_layout.addWidget(self.tags_add_button, 5, 2)
            self.tags_add_button.clicked.connect(partial(self.parent_dialog.create_custom_column,
                                                         self._tags_column_combo,
                                                         tags_custom_columns,
                                                         DEFAULT_LOOKUP_TAGS_COLUMN, True))

        # Read books combobox
        self.read_books_box_label = QLabel(_('Read &books:'), self)
        tooltip = _('Select a custom column to mark the read books. Type: yes/no.')
        self.read_books_box_label.setToolTip(tooltip)
        self.read_books_box = QComboBox()
        read_books_custom_columns = self.parent_dialog.get_custom_columns(['bool'])
        self._read_books_column_combo = CustomColumnComboBox(self, read_books_custom_columns, initial_items=[''])
        self._read_books_column_combo.setToolTip(tooltip)
        self._read_books_column_combo.setMinimumWidth(230)  # 120
        self.read_books_box_label.setBuddy(self._read_books_column_combo)
        custom_columns_group_box_layout.addWidget(self.read_books_box_label, 6, 0)
        custom_columns_group_box_layout.addWidget(self._read_books_column_combo, 6, 1)
        self._read_books_column_combo_index = self._read_books_column_combo.findText(
            prefs['read_books_column'], QtCore.Qt.MatchStartsWith)
        if self._read_books_column_combo_index == -1:
            self._read_books_column_combo.setCurrentIndex(0)
        else:
            self._read_books_column_combo.setCurrentIndex(self._read_books_column_combo_index)
        self._read_books_column_combo.currentIndexChanged.connect(self.parent_dialog.refresh_widget)

        # Read books - add custom column button
        if self.supports_create_custom_column:
            self.read_books_add_button = QPushButton(self.add_icon, '', self)
            self.read_books_add_button.setToolTip(_('Create new column for Read books'))
            self.read_books_add_button.setMaximumWidth(50)
            custom_columns_group_box_layout.addWidget(self.read_books_add_button, 6, 2)
            self.read_books_add_button.clicked.connect(partial(self.parent_dialog.create_custom_column,
                                                               self._read_books_column_combo,
                                                               read_books_custom_columns,
                                                           DEFAULT_LOOKUP_READ_BOOKS_COLUMN, False))

        # Physical books combobox
        self.physical_box_label = QLabel(_('&Physical books:'), self)
        tooltip = _('Select a custom column to mark the physical books. Type: yes/no.')
        self.physical_box_label.setToolTip(tooltip)
        self.physical_box = QComboBox()
        physical_custom_columns = self.parent_dialog.get_custom_columns(['bool'])
        self._physical_column_combo = CustomColumnComboBox(self, physical_custom_columns, initial_items=[''])
        self._physical_column_combo.setToolTip(tooltip)
        self._physical_column_combo.setMinimumWidth(230)  # 120
        self.physical_box_label.setBuddy(self._physical_column_combo)
        custom_columns_group_box_layout.addWidget(self.physical_box_label, 7, 0)
        custom_columns_group_box_layout.addWidget(self._physical_column_combo, 7, 1)
        self._physical_column_combo_index = self._physical_column_combo.findText(
            prefs['physical_column'], QtCore.Qt.MatchStartsWith)
        if self._physical_column_combo_index == -1:
            self._physical_column_combo.setCurrentIndex(0)
        else:
            self._physical_column_combo.setCurrentIndex(self._physical_column_combo_index)

        # Physical books - add custom column button
        if self.supports_create_custom_column:
            self.physical_add_button = QPushButton(self.add_icon, '', self)
            self.physical_add_button.setToolTip(_('Create new column for Physical'))
            self.physical_add_button.setMaximumWidth(50)
            custom_columns_group_box_layout.addWidget(self.physical_add_button, 7, 2)
            self.physical_add_button.clicked.connect(partial(self.parent_dialog.create_custom_column,
                                                         self._physical_column_combo,
                                                         physical_custom_columns,
                                                     DEFAULT_LOOKUP_PHYSICAL_COLUMN, False))

        # Keyboard shortcuts button
        self.keyboard_shortcuts_button = QPushButton(_('&Keyboard shortcuts'), self)
        self.keyboard_shortcuts_button.setToolTip(_('Edit the keyboard shortcuts associated with this plugin'))
        self.keyboard_shortcuts_button.clicked.connect(self.parent_dialog.edit_shortcuts)
        layout.addWidget(self.keyboard_shortcuts_button, 1, 0, 1, 1)

        # About button
        self.about_button = QPushButton(_('&About'), self)
        self.about_button.setToolTip(_('About the plugin'))
        self.about_button.clicked.connect(self.parent_dialog.about)
        layout.addWidget(self.about_button, 2, 0, 1, 1)


class CustomColumnComboBox(QComboBox):

    def __init__(self, parent, custom_columns={}, selected_column='', initial_items=[''], new_column_created=[]):
        QComboBox.__init__(self, parent)
        self.populate_combo(custom_columns, selected_column, new_column_created, initial_items)

    def populate_combo(self, custom_columns, selected_column, new_colum_created=[], initial_items=['']):
        self.clear()
        self.column_names = list(initial_items)
        if len(initial_items) > 0:
            self.addItems(initial_items)
        selected_idx = 0
        for idx, value in enumerate(initial_items):
            if value == selected_column:
                selected_idx = idx

        # Add new column created to combobox
        if new_colum_created:
            custom_columns.update({selected_column: {'name': new_colum_created}})

        for key in sorted(custom_columns.keys()):
            self.column_names.append(key)
            self.addItem('%s (%s)' % (key, custom_columns[key]['name']))
            if key == selected_column:
                selected_idx = len(self.column_names) - 1
        self.setCurrentIndex(selected_idx)

    def get_selected_column(self):
        return self.column_names[self.currentIndex()]


class SizePersistedDialog(QDialog):
    '''
    This dialog is a base class for any dialogs that want their size/position
    restored when they are next opened.
    '''
    def __init__(self, parent, unique_pref_name):
        QDialog.__init__(self, parent)
        self.unique_pref_name = unique_pref_name
        self.geom = gprefs.get(unique_pref_name, None)
        self.finished.connect(self.dialog_closing)

    def resize_dialog(self):
        if self.geom is None:
            self.resize(self.sizeHint())
        else:
            self.restoreGeometry(self.geom)

    def dialog_closing(self, result):
        geom = bytearray(self.saveGeometry())
        gprefs[self.unique_pref_name] = geom
        self.persist_custom_prefs()

    def persist_custom_prefs(self):
        '''
        Invoked when the dialog is closing. Override this function to call
        save_custom_pref() if you have a setting you want persisted that you can
        retrieve in your __init__() using load_custom_pref() when next opened
        '''
        pass

    def load_custom_pref(self, name, default=None):
        return gprefs.get(self.unique_pref_name+':'+name, default)

    def save_custom_pref(self, name, value):
        gprefs[self.unique_pref_name+':'+name] = value


class KeyboardConfigDialog(SizePersistedDialog):
    '''
    This dialog is used to allow editing of keyboard shortcuts.
    '''

    def __init__(self, gui, group_name):
        SizePersistedDialog.__init__(self, gui, 'Keyboard shortcut dialog')
        self.gui = gui
        self.setWindowTitle(_('Keyboard shortcuts'))
        self.setWindowIcon(get_icon('icon.png'))
        layout = QVBoxLayout(self)
        self.setLayout(layout)

        self.keyboard_widget = ShortcutConfig(self)
        layout.addWidget(self.keyboard_widget)
        self.group_name = group_name

        button_box = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        button_box.accepted.connect(self.commit)
        button_box.rejected.connect(self.reject)
        layout.addWidget(button_box)

        # Cause our dialog size to be restored from prefs or created on first usage
        self.resize_dialog()
        self.initialize()

    def initialize(self):
        self.keyboard_widget.initialize(self.gui.keyboard)
        self.keyboard_widget.highlight_group(self.group_name)

    def commit(self):
        self.keyboard_widget.commit()
        self.accept()


class AboutDialog(QDialog):

    def __init__(self, parent, icon, text):
        QDialog.__init__(self, parent)
        self.resize(515, 350)
        self.l = QGridLayout()
        self.setLayout(self.l)
        self.logo = QLabel()
        self.logo.setMaximumWidth(110)
        self.logo.setPixmap(QPixmap(icon.pixmap(100, 100)))
        self.label = QLabel(text)
        self.label.setOpenExternalLinks(True)
        self.label.setWordWrap(True)
        self.setWindowTitle(_('About ') + PLUGIN_NAME)
        self.setWindowIcon(icon)
        self.l.addWidget(self.logo, 0, 0)
        self.l.addWidget(self.label, 0, 1)
        self.bb = QDialogButtonBox(self)
        b = self.bb.addButton(_("OK"), self.bb.AcceptRole)
        b.setDefault(True)
        self.l.addWidget(self.bb, 2, 0, 1, -1)
        self.bb.accepted.connect(self.accept)
