#!/usr/bin/env python

from __future__ import absolute_import, division, print_function, unicode_literals

__license__   = 'GPL v3'
__copyright__ = '2019-2023, Thiago Oliveira <thiago.eec@gmail.com>'
__docformat__ = 'restructuredtext en'

# Load translation files (.mo) on the folder 'translations'
load_translations()

PLUGIN_NAME          = 'Skoob Sync'
PLUGIN_DESCRIPTION   = _('Sync calibre library with Skoob')
PLUGIN_VERSION_TUPLE = (0, 5, 7)
PLUGIN_VERSION       = '.'.join([str(x) for x in PLUGIN_VERSION_TUPLE])
PLUGIN_AUTHOR        = 'Thiago Oliveira'

# The class that all Interface Action plugin wrappers must inherit from
from calibre.customize import InterfaceActionBase


class SkoobSync(InterfaceActionBase):
    '''
    This class is a simple wrapper that provides information about the actual
    plugin class. The actual interface plugin class is called InterfacePlugin
    and is defined in the ui.py file, as specified in the actual_plugin field
    below.

    The reason for having two classes is that it allows the command line
    calibre utilities to run without needing to load the GUI libraries.
    '''
    name                = PLUGIN_NAME
    description         = PLUGIN_DESCRIPTION
    supported_platforms = ['windows', 'osx', 'linux']
    author              = PLUGIN_AUTHOR
    version             = PLUGIN_VERSION_TUPLE
    minimum_calibre_version = (2, 50, 0)

    #: This field defines the GUI plugin class that contains all the code
    #: that actually does something. Its format is module_path:class_name
    #: The specified class must be defined in the specified module.
    actual_plugin       = 'calibre_plugins.Skoob_Sync.ui:InterfacePlugin'

    def is_customizable(self):
        '''
        This method must return True to enable customization via
        Preferences->Plugins
        '''
        return True

    def config_widget(self):
        '''
        Implement this method and :meth:`save_settings` in your plugin to
        use a custom configuration dialog.

        This method, if implemented, must return a QWidget. The widget can have
        an optional method validate() that takes no arguments and is called
        immediately after the user clicks OK. Changes are applied if and only
        if the method returns True.

        If for some reason you cannot perform the configuration at this time,
        return a tuple of two strings (message, details), these will be
        displayed as a warning dialog to the user and the process will be
        aborted.

        The base class implementation of this method raises NotImplementedError
        so by default no user configuration is possible.
        '''
        # It is important to put this import statement here rather than at the
        # top of the module as importing the config class will also cause the
        # GUI libraries to be loaded, which we do not want when using calibre
        # from the command line
        from calibre_plugins.Skoob_Sync.config import ConfigWidget
        return ConfigWidget(self, self.actual_plugin_)

    def save_settings(self, config_widget):
        '''
        Save the settings specified by the user with config_widget.

        :param config_widget: The widget returned by :meth:`config_widget`.
        '''
        config_widget.save_settings()

        # Apply the changes
        ac = self.actual_plugin_
        if ac is not None:
            ac.apply_settings()