"""Check if a host is in the Google Chrome HSTS Preload list"""

import functools
import os
import typing

__version__ = "2023.1.1"
__checksum__ = "f89ebb83f90f7cdb4520b3b4f8510670affcf57ce06411442e8712209ef38639"
__all__ = ["in_hsts_preload"]

# fmt: off
_GTLD_INCLUDE_SUBDOMAINS = {b'android', b'app', b'azure', b'bank', b'bing', b'boo', b'channel', b'chrome', b'dad', b'day', b'dev', b'eat', b'esq', b'fly', b'foo', b'gle', b'gmail', b'google', b'hangout', b'hotmail', b'ing', b'insurance', b'meet', b'meme', b'microsoft', b'mov', b'new', b'nexus', b'office', b'page', b'phd', b'play', b'prof', b'rsvp', b'search', b'skype', b'windows', b'xbox', b'youtube', b'zip'}  # noqa: E501
_JUMPTABLE = [[(0, 11), (11, 10), (21, 9), (30, 61), (91, 26), (117, 12), None, (129, 19), (148, 22), (170, 7), (177, 20), (197, 18), None, (215, 29), (244, 45), (289, 7), (296, 9), (305, 36), (341, 24), (365, 10), (375, 28), None, (403, 62), (465, 8), (473, 28), (501, 19), (520, 13), (533, 14), (547, 14), None, None, (561, 40), (601, 20), (621, 35), (656, 14), (670, 32), (702, 28), (730, 9), (739, 25), (764, 25), (789, 8), (797, 13), (810, 10), None, (820, 17), (837, 6), (843, 26), (869, 10), (879, 5), (884, 19), (903, 14), (917, 11), (928, 12), (940, 27), None, (967, 24), (991, 11), (1002, 12), (1014, 29), (1043, 18), (1061, 27), (1088, 46), (1134, 25), (1159, 16), (1175, 18), (1193, 5), (1198, 22), (1220, 18), (1238, 6), (1244, 36), (1280, 15), (1295, 8), (1303, 11), None, (1314, 5), (1319, 23), (1342, 18), (1360, 18), None, (1378, 20), (1398, 26), (1424, 48), (1472, 19), (1491, 20), (1511, 59), (1570, 14), (1584, 14), (1598, 20), None, (1618, 10), (1628, 13), (1641, 20), (1661, 29), None, (1690, 13), (1703, 19), (1722, 11), (1733, 4), (1737, 22), (1759, 10), (1769, 13), (1782, 14), (1796, 28), (1824, 11), (1835, 21), (1856, 12), (1868, 32), None, (1900, 10), (1910, 14), (1924, 26), (1950, 45), (1995, 15), None, (2010, 11), (2021, 30), (2051, 21), (2072, 26), (2098, 6), (2104, 6), (2110, 7), (2117, 5), (2122, 30), (2152, 23), (2175, 35), (2210, 17), (2227, 15), (2242, 29), (2271, 12), (2283, 70), (2353, 55), (2408, 12), (2420, 23), (2443, 16), (2459, 42), (2501, 6), (2507, 24), (2531, 44), (2575, 6), (2581, 41), (2622, 22), (2644, 23), (2667, 36), (2703, 20), (2723, 8), (2731, 15), (2746, 12), (2758, 19), (2777, 25), (2802, 15), None, (2817, 46), (2863, 21), (2884, 17), (2901, 23), (2924, 26), (2950, 5), (2955, 37), (2992, 39), (3031, 16), (3047, 33), (3080, 17), (3097, 23), (3120, 14), (3134, 17), (3151, 8), (3159, 14), (3173, 7), (3180, 29), (3209, 6), (3215, 18), (3233, 32), (3265, 20), (3285, 17), (3302, 24), (3326, 12), (3338, 40), (3378, 40), (3418, 12), (3430, 48), (3478, 32), (3510, 17), None, (3527, 8), (3535, 25), (3560, 25), (3585, 6), (3591, 23), None, (3614, 36), (3650, 33), (3683, 14), (3697, 16), (3713, 27), None, (3740, 30), (3770, 41), (3811, 50), (3861, 15), (3876, 20), (3896, 31), (3927, 21), (3948, 32), (3980, 24), (4004, 20), (4024, 17), (4041, 60), (4101, 6), (4107, 9), (4116, 12), (4128, 18), (4146, 11), (4157, 10), (4167, 48), (4215, 42), None, (4257, 36), (4293, 28), None, (4321, 8), (4329, 8), (4337, 7), None, (4344, 25), (4369, 17), None, (4386, 21), (4407, 35), (4442, 21), (4463, 10), (4473, 41), (4514, 20), (4534, 31), (4565, 32), (4597, 19), (4616, 12), (4628, 5), (4633, 30), (4663, 29), (4692, 14), (4706, 14), (4720, 47), (4767, 52), None, None, (4819, 51), (4870, 42), None, (4912, 18), None, (4930, 15), (4945, 8), (4953, 21), (4974, 6), (4980, 16), (4996, 22)], [(5018, 10676), (15694, 10937), (26631, 11126), (37757, 10601), (48358, 10458), (58816, 10363), (69179, 11157), (80336, 10165), (90501, 11209), (101710, 10528), (112238, 11431), (123669, 10377), (134046, 11138), (145184, 12219), (157403, 10669), (168072, 11528), (179600, 11810), (191410, 11105), (202515, 11097), (213612, 10325), (223937, 11451), (235388, 10947), (246335, 11091), (257426, 11164), (268590, 11577), (280167, 10972), (291139, 11181), (302320, 11149), (313469, 10552), (324021, 11001), (335022, 11605), (346627, 11143), (357770, 11013), (368783, 10767), (379550, 10276), (389826, 11052), (400878, 10812), (411690, 11842), (423532, 11473), (435005, 11676), (446681, 11823), (458504, 10517), (469021, 10635), (479656, 10902), (490558, 11016), (501574, 11266), (512840, 10767), (523607, 12244), (535851, 11149), (547000, 9996), (556996, 10954), (567950, 11378), (579328, 11223), (590551, 10922), (601473, 11218), (612691, 10813), (623504, 11160), (634664, 10995), (645659, 11237), (656896, 9600), (666496, 10780), (677276, 10856), (688132, 10819), (698951, 11340), (710291, 11353), (721644, 11469), (733113, 10864), (743977, 11726), (755703, 11617), (767320, 11007), (778327, 10853), (789180, 10460), (799640, 9770), (809410, 11322), (820732, 10844), (831576, 11445), (843021, 10392), (853413, 11775), (865188, 10920), (876108, 10680), (886788, 11243), (898031, 9833), (907864, 10720), (918584, 10958), (929542, 10710), (940252, 11173), (951425, 11626), (963051, 10873), (973924, 11078), (985002, 11168), (996170, 12607), (1008777, 10426), (1019203, 10898), (1030101, 10644), (1040745, 11138), (1051883, 11202), (1063085, 10963), (1074048, 10682), (1084730, 11368), (1096098, 10456), (1106554, 10807), (1117361, 10901), (1128262, 10701), (1138963, 10584), (1149547, 10652), (1160199, 11359), (1171558, 11514), (1183072, 11376), (1194448, 11985), (1206433, 11102), (1217535, 11712), (1229247, 11016), (1240263, 10740), (1251003, 10880), (1261883, 11225), (1273108, 11029), (1284137, 11109), (1295246, 10990), (1306236, 10795), (1317031, 11759), (1328790, 11388), (1340178, 11370), (1351548, 11066), (1362614, 11369), (1373983, 12070), (1386053, 10896), (1396949, 10692), (1407641, 12012), (1419653, 10838), (1430491, 12500), (1442991, 11614), (1454605, 10604), (1465209, 10893), (1476102, 10583), (1486685, 10796), (1497481, 11175), (1508656, 10546), (1519202, 11747), (1530949, 10778), (1541727, 10754), (1552481, 11486), (1563967, 11560), (1575527, 10481), (1586008, 10734), (1596742, 11519), (1608261, 10392), (1618653, 11330), (1629983, 10675), (1640658, 10582), (1651240, 11286), (1662526, 11234), (1673760, 11242), (1685002, 11257), (1696259, 10427), (1706686, 10996), (1717682, 10997), (1728679, 10981), (1739660, 11406), (1751066, 10577), (1761643, 10340), (1771983, 10376), (1782359, 11170), (1793529, 11745), (1805274, 10300), (1815574, 11058), (1826632, 12254), (1838886, 11264), (1850150, 10630), (1860780, 11948), (1872728, 10812), (1883540, 10184), (1893724, 10897), (1904621, 12425), (1917046, 10393), (1927439, 10281), (1937720, 11430), (1949150, 10889), (1960039, 11231), (1971270, 10864), (1982134, 10374), (1992508, 13343), (2005851, 11016), (2016867, 10840), (2027707, 10895), (2038602, 11637), (2050239, 11937), (2062176, 10120), (2072296, 11562), (2083858, 11035), (2094893, 10888), (2105781, 12090), (2117871, 10467), (2128338, 11031), (2139369, 10705), (2150074, 11225), (2161299, 11047), (2172346, 10725), (2183071, 10213), (2193284, 10806), (2204090, 10546), (2214636, 11452), (2226088, 10937), (2237025, 11849), (2248874, 10699), (2259573, 11711), (2271284, 11535), (2282819, 10240), (2293059, 11354), (2304413, 11166), (2315579, 11328), (2326907, 11311), (2338218, 11371), (2349589, 11052), (2360641, 10870), (2371511, 11412), (2382923, 10861), (2393784, 10480), (2404264, 11105), (2415369, 10592), (2425961, 11805), (2437766, 10548), (2448314, 10340), (2458654, 11862), (2470516, 10856), (2481372, 11272), (2492644, 10765), (2503409, 11063), (2514472, 10298), (2524770, 11357), (2536127, 11323), (2547450, 12029), (2559479, 10970), (2570449, 10516), (2580965, 11552), (2592517, 11084), (2603601, 12016), (2615617, 10593), (2626210, 10207), (2636417, 10028), (2646445, 11614), (2658059, 11372), (2669431, 11558), (2680989, 11264), (2692253, 11269), (2703522, 10553), (2714075, 11484), (2725559, 11345), (2736904, 10371), (2747275, 11050), (2758325, 10523), (2768848, 11086), (2779934, 11295), (2791229, 11331), (2802560, 10841), (2813401, 10899), (2824300, 10874)], [(2835174, 1229), (2836403, 926), (2837329, 1025), (2838354, 1193), (2839547, 985), (2840532, 1097), (2841629, 962), (2842591, 1316), (2843907, 920), (2844827, 1081), (2845908, 735), (2846643, 855), (2847498, 1154), (2848652, 1285), (2849937, 1228), (2851165, 1288), (2852453, 1613), (2854066, 907), (2854973, 1118), (2856091, 1037), (2857128, 1209), (2858337, 1049), (2859386, 1255), (2860641, 985), (2861626, 1084), (2862710, 988), (2863698, 1427), (2865125, 1569), (2866694, 1072), (2867766, 1113), (2868879, 1262), (2870141, 1146), (2871287, 980), (2872267, 1054), (2873321, 1333), (2874654, 1107), (2875761, 1003), (2876764, 1113), (2877877, 1049), (2878926, 1446), (2880372, 1062), (2881434, 1302), (2882736, 1053), (2883789, 1028), (2884817, 1059), (2885876, 799), (2886675, 1273), (2887948, 1309), (2889257, 1069), (2890326, 700), (2891026, 1131), (2892157, 1085), (2893242, 1121), (2894363, 1327), (2895690, 1731), (2897421, 945), (2898366, 1009), (2899375, 1033), (2900408, 1018), (2901426, 1053), (2902479, 1104), (2903583, 1230), (2904813, 1361), (2906174, 1250), (2907424, 1005), (2908429, 1184), (2909613, 1087), (2910700, 819), (2911519, 1024), (2912543, 1239), (2913782, 1104), (2914886, 1136), (2916022, 820), (2916842, 1084), (2917926, 896), (2918822, 1035), (2919857, 897), (2920754, 1003), (2921757, 1074), (2922831, 837), (2923668, 1213), (2924881, 948), (2925829, 1213), (2927042, 983), (2928025, 1066), (2929091, 957), (2930048, 1054), (2931102, 1115), (2932217, 1220), (2933437, 1167), (2934604, 1474), (2936078, 1392), (2937470, 1361), (2938831, 979), (2939810, 1092), (2940902, 740), (2941642, 1225), (2942867, 1059), (2943926, 894), (2944820, 926), (2945746, 1093), (2946839, 1276), (2948115, 1123), (2949238, 818), (2950056, 1015), (2951071, 1197), (2952268, 827), (2953095, 789), (2953884, 1226), (2955110, 1449), (2956559, 1185), (2957744, 1016), (2958760, 1108), (2959868, 1113), (2960981, 1074), (2962055, 1090), (2963145, 1021), (2964166, 931), (2965097, 997), (2966094, 893), (2966987, 1445), (2968432, 1032), (2969464, 1127), (2970591, 827), (2971418, 1107), (2972525, 1081), (2973606, 1134), (2974740, 1296), (2976036, 1001), (2977037, 1374), (2978411, 1206), (2979617, 1029), (2980646, 1107), (2981753, 942), (2982695, 1202), (2983897, 1191), (2985088, 1036), (2986124, 1007), (2987131, 1029), (2988160, 855), (2989015, 955), (2989970, 927), (2990897, 1046), (2991943, 853), (2992796, 835), (2993631, 755), (2994386, 1041), (2995427, 979), (2996406, 1072), (2997478, 1007), (2998485, 1056), (2999541, 950), (3000491, 793), (3001284, 1196), (3002480, 1112), (3003592, 981), (3004573, 998), (3005571, 1229), (3006800, 1070), (3007870, 884), (3008754, 1346), (3010100, 1034), (3011134, 1038), (3012172, 1236), (3013408, 1358), (3014766, 1032), (3015798, 922), (3016720, 1183), (3017903, 1025), (3018928, 962), (3019890, 1047), (3020937, 961), (3021898, 1305), (3023203, 1124), (3024327, 1219), (3025546, 1256), (3026802, 1127), (3027929, 884), (3028813, 1104), (3029917, 1025), (3030942, 2251), (3033193, 855), (3034048, 1098), (3035146, 1047), (3036193, 1386), (3037579, 1031), (3038610, 1051), (3039661, 957), (3040618, 877), (3041495, 1350), (3042845, 972), (3043817, 812), (3044629, 1055), (3045684, 1353), (3047037, 1265), (3048302, 1155), (3049457, 1023), (3050480, 869), (3051349, 1024), (3052373, 1011), (3053384, 1141), (3054525, 963), (3055488, 1177), (3056665, 926), (3057591, 1100), (3058691, 858), (3059549, 1128), (3060677, 1350), (3062027, 1015), (3063042, 1296), (3064338, 1070), (3065408, 1089), (3066497, 1210), (3067707, 1421), (3069128, 1049), (3070177, 1015), (3071192, 1190), (3072382, 944), (3073326, 964), (3074290, 749), (3075039, 1055), (3076094, 1182), (3077276, 821), (3078097, 1372), (3079469, 869), (3080338, 1089), (3081427, 1012), (3082439, 1221), (3083660, 1205), (3084865, 1087), (3085952, 1196), (3087148, 879), (3088027, 1259), (3089286, 985), (3090271, 907), (3091178, 940), (3092118, 974), (3093092, 599), (3093691, 1194), (3094885, 1286), (3096171, 1111), (3097282, 924), (3098206, 969), (3099175, 871), (3100046, 1210), (3101256, 909), (3102165, 916), (3103081, 1264), (3104345, 771), (3105116, 1225), (3106341, 2755), (3109096, 939), (3110035, 947), (3110982, 1246), (3112228, 1210), (3113438, 772)], [(3114210, 48), None, (3114258, 35), (3114293, 42), None, None, None, None, None, None, None, None, None, None, (3114335, 21), None, None, (3114356, 42), None, (3114398, 25), (3114423, 44), (3114467, 22), (3114489, 82), None, None, None, None, (3114571, 26), None, None, None, None, (3114597, 21), (3114618, 46), None, (3114664, 21), (3114685, 44), None, (3114729, 16), None, None, (3114745, 71), (3114816, 21), (3114837, 23), None, None, None, None, (3114860, 48), None, None, None, None, None, (3114908, 31), None, None, None, None, (3114939, 42), None, (3114981, 22), None, (3115003, 21), None, (3115024, 43), (3115067, 56), None, None, (3115123, 77), (3115200, 27), None, None, None, None, (3115227, 21), (3115248, 52), None, None, (3115300, 54), (3115354, 42), None, None, None, (3115396, 25), None, None, (3115421, 21), None, None, None, None, None, (3115442, 24), (3115466, 21), None, None, (3115487, 48), None, (3115535, 18), None, (3115553, 54), None, None, None, None, None, None, (3115607, 26), None, None, None, (3115633, 20), (3115653, 28), None, (3115681, 64), (3115745, 42), (3115787, 17), (3115804, 17), (3115821, 26), None, (3115847, 45), None, None, None, (3115892, 26), (3115918, 20), (3115938, 26), (3115964, 16), (3115980, 42), (3116022, 63), None, (3116085, 20), None, (3116105, 40), (3116145, 48), None, None, None, (3116193, 47), (3116240, 16), None, (3116256, 31), None, None, None, None, (3116287, 42), None, (3116329, 80), None, (3116409, 9), None, (3116418, 21), (3116439, 42), None, None, (3116481, 65), (3116546, 82), (3116628, 45), None, (3116673, 72), None, None, (3116745, 24), (3116769, 21), None, None, None, None, None, (3116790, 42), (3116832, 21), (3116853, 21), None, (3116874, 42), (3116916, 25), None, (3116941, 38), (3116979, 21), (3117000, 73), None, None, (3117073, 21), (3117094, 19), (3117113, 26), (3117139, 24), (3117163, 16), None, (3117179, 21), None, None, (3117200, 38), None, (3117238, 22), (3117260, 21), (3117281, 21), (3117302, 21), None, (3117323, 63), None, (3117386, 21), (3117407, 42), None, (3117449, 17), None, None, None, None, (3117466, 21), (3117487, 21), None, None, (3117508, 21), None, None, (3117529, 21), None, (3117550, 26), None, (3117576, 50), (3117626, 22), None, None, (3117648, 50), (3117698, 26), (3117724, 21), (3117745, 21), (3117766, 19), None, (3117785, 35), (3117820, 26), (3117846, 23), (3117869, 39), (3117908, 42), None, None, None, None, None, None, (3117950, 21), None, None, None, (3117971, 21), None, None, (3117992, 90), None, (3118082, 239), (3118321, 38), None, None, (3118359, 21), None]]  # noqa: E501
_CRC8_TABLE = [
    0x00, 0x07, 0x0e, 0x09, 0x1c, 0x1b, 0x12, 0x15,
    0x38, 0x3f, 0x36, 0x31, 0x24, 0x23, 0x2a, 0x2d,
    0x70, 0x77, 0x7e, 0x79, 0x6c, 0x6b, 0x62, 0x65,
    0x48, 0x4f, 0x46, 0x41, 0x54, 0x53, 0x5a, 0x5d,
    0xe0, 0xe7, 0xee, 0xe9, 0xfc, 0xfb, 0xf2, 0xf5,
    0xd8, 0xdf, 0xd6, 0xd1, 0xc4, 0xc3, 0xca, 0xcd,
    0x90, 0x97, 0x9e, 0x99, 0x8c, 0x8b, 0x82, 0x85,
    0xa8, 0xaf, 0xa6, 0xa1, 0xb4, 0xb3, 0xba, 0xbd,
    0xc7, 0xc0, 0xc9, 0xce, 0xdb, 0xdc, 0xd5, 0xd2,
    0xff, 0xf8, 0xf1, 0xf6, 0xe3, 0xe4, 0xed, 0xea,
    0xb7, 0xb0, 0xb9, 0xbe, 0xab, 0xac, 0xa5, 0xa2,
    0x8f, 0x88, 0x81, 0x86, 0x93, 0x94, 0x9d, 0x9a,
    0x27, 0x20, 0x29, 0x2e, 0x3b, 0x3c, 0x35, 0x32,
    0x1f, 0x18, 0x11, 0x16, 0x03, 0x04, 0x0d, 0x0a,
    0x57, 0x50, 0x59, 0x5e, 0x4b, 0x4c, 0x45, 0x42,
    0x6f, 0x68, 0x61, 0x66, 0x73, 0x74, 0x7d, 0x7a,
    0x89, 0x8e, 0x87, 0x80, 0x95, 0x92, 0x9b, 0x9c,
    0xb1, 0xb6, 0xbf, 0xb8, 0xad, 0xaa, 0xa3, 0xa4,
    0xf9, 0xfe, 0xf7, 0xf0, 0xe5, 0xe2, 0xeb, 0xec,
    0xc1, 0xc6, 0xcf, 0xc8, 0xdd, 0xda, 0xd3, 0xd4,
    0x69, 0x6e, 0x67, 0x60, 0x75, 0x72, 0x7b, 0x7c,
    0x51, 0x56, 0x5f, 0x58, 0x4d, 0x4a, 0x43, 0x44,
    0x19, 0x1e, 0x17, 0x10, 0x05, 0x02, 0x0b, 0x0c,
    0x21, 0x26, 0x2f, 0x28, 0x3d, 0x3a, 0x33, 0x34,
    0x4e, 0x49, 0x40, 0x47, 0x52, 0x55, 0x5c, 0x5b,
    0x76, 0x71, 0x78, 0x7f, 0x6a, 0x6d, 0x64, 0x63,
    0x3e, 0x39, 0x30, 0x37, 0x22, 0x25, 0x2c, 0x2b,
    0x06, 0x01, 0x08, 0x0f, 0x1a, 0x1d, 0x14, 0x13,
    0xae, 0xa9, 0xa0, 0xa7, 0xb2, 0xb5, 0xbc, 0xbb,
    0x96, 0x91, 0x98, 0x9f, 0x8a, 0x8d, 0x84, 0x83,
    0xde, 0xd9, 0xd0, 0xd7, 0xc2, 0xc5, 0xcc, 0xcb,
    0xe6, 0xe1, 0xe8, 0xef, 0xfa, 0xfd, 0xf4, 0xf3
]
# fmt: on

_IS_LEAF = 0x80
_INCLUDE_SUBDOMAINS = 0x40


try:
    from importlib.resources import open_binary

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open_binary("hstspreload", path)


except ImportError:

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open(
            os.path.join(os.path.dirname(os.path.abspath(__file__)), path),
            "rb",
        )


@functools.lru_cache(maxsize=1024)
def in_hsts_preload(host: typing.AnyStr) -> bool:
    """Determines if an IDNA-encoded host is on the HSTS preload list"""

    if isinstance(host, str):
        host = host.encode("ascii")
    labels = host.lower().split(b".")

    # Fast-branch for gTLDs that are registered to preload all sub-domains.
    if labels[-1] in _GTLD_INCLUDE_SUBDOMAINS:
        return True

    with open_pkg_binary("hstspreload.bin") as f:
        for layer, label in enumerate(labels[::-1]):
            # None of our layers are greater than 4 deep.
            if layer > 3:
                return False

            # Read the jump table for the layer and label
            jump_info = _JUMPTABLE[layer][_crc8(label)]
            if jump_info is None:
                # No entry: host is not preloaded
                return False

            # Read the set of entries for that layer and label
            f.seek(jump_info[0])
            data = bytearray(jump_info[1])
            f.readinto(data)

            for is_leaf, include_subdomains, ent_label in _iter_entries(data):
                # We found a potential leaf
                if is_leaf:
                    if ent_label == host:
                        return True
                    if include_subdomains and host.endswith(b"." + ent_label):
                        return True

                # Continue traversing as we're not at a leaf.
                elif label == ent_label:
                    break
            else:
                return False
    return False


def _iter_entries(data: bytes) -> typing.Iterable[typing.Tuple[int, int, bytes]]:
    while data:
        flags = data[0]
        size = data[1]
        label = bytes(data[2 : 2 + size])
        yield (flags & _IS_LEAF, flags & _INCLUDE_SUBDOMAINS, label)
        data = data[2 + size :]


def _crc8(value: bytes) -> int:
    # CRC8 reference implementation: https://github.com/niccokunzmann/crc8
    checksum = 0x00
    for byte in value:
        checksum = _CRC8_TABLE[checksum ^ byte]
    return checksum
