#!/usr/bin/env python3

import gettext
import time
from queue import Queue, Empty
from threading import Thread

from calibre.ebooks.metadata import check_isbn
from calibre.ebooks.metadata.book.base import Metadata
from calibre.ebooks.metadata.sources.base import Source, Option, fixauthors, fixcase
from calibre_plugins.sfleihbuch.objects import BookList, BookDetail  # , BookCovers

_ = gettext.gettext
load_translations()


# About
# From 1948 to 1979 there where numerous loan book publishers in West Germany whose books were offered in special loan
# libraries.
# The offer consisted of western, adventure, crime and romance novels, but also science fiction novels (but the term
# "science fiction" was not yetwidely used).
# Further reading (in German): https://de.wikipedia.org/wiki/Leihbibliothek
#
# References:
# (Thanks to Michael Peters <mp@mpeters.de>, webmaster of sf-leihbuch.de, for his support!)
# The SF-Leihbuch home page: http://www.sf-leihbuch.org/
# The SF-Leihbuch database scheme:
#           Name                                                      Typ                          Größe
# Tabelle: tblautor
#           autorid                                                   Long Integer                                4
#           nachname                                                  Kurzer Text                                50
#           vorname                                                   Kurzer Text                                50
#           realname                                                  Kurzer Text                               255
#           sid                                                       Long Integer                                4
# Tabelle: tblbuch
#           buchid                                                    Long Integer                                4
#           autorid                                                   Long Integer                                4
#           autorid2                                                  Long Integer                                4
#           titel                                                     Kurzer Text                               255
#           untertitel                                                Kurzer Text                               255
#           coverscan                                                 Kurzer Text                               255
#           coverscan2                                                Kurzer Text                               255
#           grafikerid                                                Long Integer                                4
#           grafikerid2                                               Long Integer                                4
#           verlagid                                                  Long Integer                                4
#           verlagsnr                                                 Integer                                     2
#           serienid                                                  Long Integer                                4
#           serienfolgenr                                             Integer                                     2
#           jahr                                                      Integer                                     2
#           originaltitel                                             Kurzer Text                               255
#           originaljahr                                              Integer                                     2
#           uebersetzer                                               Kurzer Text                                50
#           eingegeben                                                Long Integer                                4
#           eingegebenam                                              Datum mit Uhrzeit                           8
#           geaendert                                                 Long Integer                                4
#           geaendertam                                               Datum mit Uhrzeit                           8
#           abstract                                                  Langer Text                                 -
#           nachdruck                                                 Langer Text                                 -
#           sid1                                                      Long Integer                                4
#           sid2                                                      Long Integer                                4
#           sid3                                                      Long Integer                                4
# Tabelle: tblgrafiker
#           grafikerid                                                Long Integer                                4
#           nachname                                                  Kurzer Text                               255
#           vorname                                                   Kurzer Text                               255
# Tabelle: tblserie                                                                                         Seite: 8
#           serienid                                                  Long Integer                                4
#           serie                                                     Kurzer Text                               255
# Tabelle: tblverlag
#           verlagid                                                  Long Integer                                4
#           verlagkurzname                                            Kurzer Text                                50
#           verlagname                                                Kurzer Text                               255
#           verlagbeschreibung                                        Langer Text                                 -


class SfLeihbuch(Source):
    name = 'SF-Leihbuch'
    description = _(
        'Downloads metadata and covers from the science fiction loan book database (http://www.sf-leihbuch.de/)')
    author = 'feuille'
    version = (1, 0, 1)  # see changelog
    released = ('10-24-2024')
    history = True
    # Version 1.0.0 - 11-01-2021
    # - Initial release
    # Version 1.0.1 - 10-24-2024
    # - Encoding is now utf-8.
    minimum_calibre_version = (5, 0, 0)
    can_get_multiple_covers = True
    has_html_comments = True
    supports_gzip_transfer_encoding = False
    cached_cover_url_is_reliable = True
    prefer_results_with_isbn = False

    capabilities = frozenset(['identify', 'cover'])
    touched_fields = frozenset(['title', 'authors',
                                'series', 'series_index', 'languages',
                                'identifier:sfldb_id',
                                'publisher', 'pubdate', 'comments', 'tags'])
    # ToDo: Combine with other Sources? 'identifier:isbn', 'identifier:dnb', 'identifier:oclc-worldcat',

    # ToDo: Set config values in module config.py
    # import calibre_plugins.sfleihbuch.config as cfg

    # from class 'option' in base.py:
    '''
    :param name: The name of this option. Must be a valid python identifier
    :param type_: The type of this option, one of ('number', 'string', 'bool', 'choices')
    :param default: The default value for this option
    :param label: A short (few words) description of this option
    :param desc: A longer description of this option
    :param choices: A dict of possible values, used only if type='choices'.
    dict is of the form {key:human readable label, ...}
    '''
    options = (
        Option(
            'search_mode',
            'choices',
            'fuzzy_search',
            _('Search mode'),
            _('Choose one of the options for search mode.'),
            {
                'exact_search': _('Exact search'),
                'fuzzy_search': _('Fuzzy search'), }
        ),
        Option(
            'fuzzy_search_target',
            'choices',
            'search_in_title',
            _('Target for fuzzy search'),
            _('Choose one of the options for search target.'),
            {'search_in_title': _('Title'), 'search_in_blurb': _('Blurb')}
        ),
        Option(
            'fuzzy_search_meta_field',
            'choices',
            'search_term_from_title_field',
            _('Meta field for fuzzy search'),
            _('Choose one of the following fields as source for fuzzy search.'),
            {'search_term_from_title_field': _('Title'),
             'search_term_from_author_field': _('Authors (only first)'), }
        ),
        # ToDo: set debug level
        # Option(
        #     'log_level',
        #     'number',
        #     0,
        #     _('log level, from 0 to 7'),
        #     _('Log detail level. 0 = no logging (only global info and error messages. 1 = ...)
        # ),
        Option(
            'max_results',
            'number',
            10,
            _('Maximum number of search results to download:'),
            _('This setting only applies to title / author searches. Book records with a valid SF-Leihbuch ID will '
              'return exactly one result.'),
        ),
        Option(
            'max_covers',
            'number',
            10,
            _('Maximum number of covers to download:'),
            _('The maximum number of covers to download.')
        ),
    )

    def __init__(self, *args, **kwargs):
        super(SfLeihbuch, self).__init__(*args, **kwargs)
        self._publication_id_to_book_id_cache = {}

    # def cache_publication_id_to_book_id(self, sfldb_id, book_id):
    #     with self.cache_lock:
    #         self._publication_id_to_book_id_cache[sfldb_id] = book_id

    # def cached_publication_id_to_book_id(self, sfldb_id):
    #     with self.cache_lock:
    #         return self._publication_id_to_book_id_cache.get(sfldb_id, None)

    def dump_caches(self):
        dump = super(SfLeihbuch, self).dump_caches()
        with self.cache_lock:
            dump.update({
                'publication_id_to_book_id': self._publication_id_to_book_id_cache.copy(),
            })
        return dump

    def load_caches(self, dump):
        super(SfLeihbuch, self).load_caches(dump)
        with self.cache_lock:
            self._publication_id_to_book_id_cache.update(dump['publication_id_to_book_id'])

    def get_book_url(self, identifiers, log):
        # ('*** Enter get_book_url().')
        book_id = identifiers.get('sfldb_id', None)
        if book_id:
            url = BookDetail.url_from_id(book_id)
            # log.info('Getting url with BookDetail.url_from_id():{0}'.format(url))
            return ('sfldb_id', book_id, url)
        # log.info('No url found with BookDetail.url_from_id()!')
        return None

    def get_cached_cover_url(self, identifiers):
        # self.log.info('*** Enter get_cached_cover_url().')
        sfldb_id = identifiers.get('sfldb_id', None)
        if sfldb_id:
            ccu = self.cached_identifier_to_cover_url(sfldb_id)
            # self.log.info('ccu={0}'.format(ccu))
            return ccu  # return self.cached_identifier_to_cover_url(sfldb_id)
        return None

    def get_author_tokens(self, authors, only_first_author=True):
        # We override this because we don't want to strip out middle initials!
        # This *just* attempts to unscramble "surname, first name".
        if only_first_author:
            authors = authors[:1]
        for au in authors:
            if "," in au:
                parts = au.split(",")
                parts = parts[1:] + parts[:1]
                au = " ".join(parts)
            for tok in au.split():
                yield tok

    def clean_downloaded_metadata(self, mi):
        # We override this because we don't want to fixcase if the language is unknown
        '''
        Call this method in your plugin's identify method to normalize metadata
        before putting the Metadata object into result_queue. You can of
        course, use a custom algorithm suited to your metadata source.
        '''
        docase = mi.language == 'eng'  # or mi.is_null('language')
        if docase and mi.title:
            mi.title = fixcase(mi.title)
            mi.authors = fixauthors(mi.authors)
        if mi.tags and docase:
            mi.tags = list(map(fixcase, mi.tags))
        mi.isbn = check_isbn(mi.isbn)

    def identify(self, log, result_queue, abort, title=None, authors=None, identifiers={}, timeout=30):
        '''
        This method will find exactly one result if an SfLeihbuch ID is
        present, otherwise up to the maximum searching first for the
        ISBN and then for title and author.
        '''

        # log.info('*** Enter SfLeihbuch.identify() with following arguments:.')
        # log.info('identifiers={0}'.format(identifiers))
        # log.info('title={0}'.format(title))
        # log.info('authors={0}'.format(authors))

        matches = set()

        ###########################################
        # 1. Search with SF-Leihbuch ID, if given #
        ###########################################

        # The SF-Leibuch DB uses no ISBN (since most this books where published before she was in use) or other official
        # identifiers as clc-worldcat or similar. So, the identifier for SF-Leibuch DB (sflb_id) is taken from the
        # book id of the database (table tblBuch, field buchid)
        # In contrast to the ISFDB (isfdb.org), no distinction is made between title and publication

        # If we have an SF-Leihbuch book ID, we use it to construct the book URL directly
        book_url_tuple = self.get_book_url(identifiers, log)
        # log.info('book_url_tuple={0}'.format(book_url_tuple))

        if book_url_tuple:
            id_type, id_val, url = book_url_tuple
            log.info(_('Book id given: {0}:{1}.').format(id_type, id_val))
            if url is not None:
                matches.add((url, 0))  # most relevant search argument
                # log.info('Add match: id_type={0}, id_val={1}, url={2}.'.format(id_type, id_val, url))
        else:
            if abort.is_set():
                log.info(_('Abort is set.'))
                return

            def stripped(s):
                return "".join(c.lower() for c in s if c.isalpha() or c.isspace())

            ##################################################################
            # 2. Search with content of title field only (in title or blurb) #
            ##################################################################

            # Since the SF loan book database only contains around 1,500 book entries and is practically complete, 
            # a keyword search (in the title or blurb) seems sufficient. 
            # In addition, the author cannot be transmitted in the POST data as a name, but only as an ID.

            log.info(_('No id(s) given. Trying a keyword search with title field content: {0}.').format(title))

            title_tokens = self.get_title_tokens(title, strip_joiners=False, strip_subtitle=True)
            title = ' '.join(title_tokens)
            # log.info('title={0}'.format(title))

            if len(matches) < self.prefs["max_results"] and self.prefs["search_mode"]:
                # Build the quey and put the result(s) in stubs for book details access
                query = BookList.url_from_title(self.prefs, title, log)
                # og.info('query={0} '.format(query))
                stubs = BookList.from_url(self.browser, query, timeout, log)
                # log.info('{0} stubs found with BookList.from_url().'.format(len(stubs)))

                for stub in stubs:
                    # log.info('stub={0}'.format(stub))
                    relevance = 0
                    # If exact title is given in calibre's metada title field
                    if stripped(stub["title"]) == stripped(title):  # Exact title is given, not only keyword(s)
                        relevance = 0
                        log.info(_('Exact title is given.'))
                    if stub["url"] is not None:
                        matches.add((stub["url"], relevance))
                        # log.info('Add match: {0}.'.format(stub["url"]))
                        if len(matches) >= self.prefs["max_results"]:
                            break

        if abort.is_set():
            log.info(_('Abort is set.'))
            return

        # log.info('matches={0}'.format(matches))
        log.info(_('Starting workers...'))

        workers = [Worker(m_url, result_queue, self.browser, log, m_rel, self, timeout) for (m_url, m_rel) in matches]

        for w in workers:
            w.start()
            # Don't send all requests at the same time
            time.sleep(0.1)

        while not abort.is_set():
            a_worker_is_alive = False
            for w in workers:
                w.join(0.2)
                if abort.is_set():
                    break
                if w.is_alive():
                    a_worker_is_alive = True
            if not a_worker_is_alive:
                break

    def download_cover(self, log, result_queue, abort, title=None, authors=None, identifiers={}, timeout=30,
                       get_best_cover=False):
        # log.info("*** Enter SfLeihbuch.download_cover().")
        # log.info("identifiers={0}".format(identifiers))

        cached_url = self.get_cached_cover_url(identifiers)

        if not cached_url:
            log.info(_("No cached book cover url found. Running identify."))
            rq = Queue()
            self.identify(log, rq, abort, title, authors, identifiers, timeout)
            if abort.is_set():
                return
            results = []
            while True:
                try:
                    results.append(rq.get_nowait())
                except Empty:
                    break

        if cached_url:
            log.info("Using cached cover url(s): {0}".format(cached_url))
            self.download_multiple_covers(title, authors, cached_url, get_best_cover, timeout, result_queue, abort, log,
                                          prefs_name='max_covers')
        else:
            log.error(_("We were unable to find any covers."))

        if abort.is_set():
            return


class Worker(Thread):
    '''
    Get book details from SF-Leihbuch DB book page in a seperate thread.
    '''

    def __init__(self, url, result_queue, browser, log, relevance, plugin, timeout=20):
        Thread.__init__(self)
        self.daemon = True
        # log.info('Worker.__init__, url={0}'.format(url))
        self.url = url
        self.result_queue = result_queue
        self.log = log
        self.timeout = timeout
        self.relevance = relevance
        self.plugin = plugin
        self.browser = browser.clone_browser()

    def run(self):

        # self.log.info('*** Enter Worker.run().')

        try:
            self.log.info(_('Worker parsing SF-Leihbuch DB url: %r') % self.url)

            pub = {}

            if BookDetail.is_type_of(self.url, self.log):
                self.log.info(_("This url is a book url. Going to fetch book details."))
                pub = BookDetail.from_url(self.browser, self.url, self.timeout, self.log)
                # self.log.info("BookDetail.from_url() finished. pub={0}".format(pub))
            else:
                self.log.error(_("Out of cheese error! Unrecognised url!"))
                return

            # Put extracted metadata in queue

            # self.log.info('Put extracted metadata in queue.')
            if len(pub["authors"]) == 0:
                pub["authors"] = [_('Unknown')]
            mi = Metadata(pub["title"], pub["authors"])

            # ToDo: use IDENTIFIERS_IDS
            for id_name in ("isbn", "sfldb_id", "dnb", "oclc-worldcat"):
                if id_name in pub:
                    # self.log.info('Set identifier {0}: {1}'.format(id_name, pub[id_name]))
                    mi.set_identifier(id_name, pub[id_name])

            # Fill object mi with data from metadata source

            for attr in ("publisher", "pubdate", "comments", "series", "series_index", "tags", "language"):
                if attr in pub:
                    # self.log.info('Set metadata for attribute {0}: {1}'.format(attr, pub[attr]))
                    setattr(mi, attr, pub[attr])

            # Cache cover url(s) for this book
            if pub.get("cover_url"):
                self.plugin.cache_identifier_to_cover_url(pub["sfldb_id"], pub["cover_url"])
                mi.has_cover = True

            mi.source_relevance = self.relevance

            self.plugin.clean_downloaded_metadata(mi)
            # self.log.info('##### Finally formatted metadata ##### \n {0}'.format(mi))
            # self.log.info(''.join([char * 20 for char in '#']))
            self.result_queue.put(mi)

        except Exception as e:
            self.log.exception(_('Worker failed to fetch and parse url {0} with error {1}' % (self.url, e)))


#     def identify(self, log, result_queue, abort, title=None, authors=None, identifiers={}, timeout=30):
#         log.debug('\nEntering identify(self, log, result_queue, abort, title=None, authors=None,identifiers={}, timeout=30)')
#         log.info('log          : ', log)
#         log.error('result_queue : ', result_queue)
#         log.info('abort        : ', abort)
#         log.info('title        : ', title)
#         log.info('authors      : ', authors, type(authors))
#         log.info('identifiers  : ', identifiers)
#         log.info('\n')
# log.error() will automatically print a traceback if an exception has occurred. Other than that you may or may not see
# a difference depending on the type of logging backend used.
# Entering identify(self, log, result_queue, abort, title=None, authors=None,identifiers={}, timeout=30)
# log          :  <calibre.utils.logging.ThreadSafeLog object at 0x000001D37BA784C0>
# result_queue :  <queue.Queue object at 0x000001D37BA78A90>
# abort        :  <threading.Event object at 0x000001D37BA784F0>
# title        :  Le Printemps d'Helliconia
# authors      :  ['B.W. Aldiss'] <class 'list'>
# identifiers  :  {'isbn': '2-221-10703-9'}

if __name__ == '__main__':  # tests
    # To run these test use:
    # calibre-debug -e __init__.py
    from calibre.ebooks.metadata.sources.test import (test_identify_plugin, title_test, authors_test)

    # Test the plugin.
    # TODO: new test cases
    # by catalog id
    # by title id
    # multiple authors
    # anthology
    # with cover
    # without cover
    test_identify_plugin(SfLeihbuch.name,
                         [
                             (  # By book id
                                 {'identifiers': {'sfldb_id': '286'}},
                                 [title_test('Weltraumkreuzer über Afrika', exact=True),
                                  authors_test(['Berning, Frank'])]  # author id = 10
                             ),
                             (  # By ISBN
                                 {'identifiers': {'isbn': '0330020420'}},
                                 [title_test('All Flesh Is Grass', exact=True), authors_test(['Clifford D. Simak'])]
                             ),
                             (  # By title
                                 {'title': 'weltraum', 'authors': ['']},
                                 [title_test('weltraum', exact=False), authors_test(['Berning, Frank'])]
                             ),
                             (  # By author and title
                                 {'title': 'weltraum', 'authors': ['Berning, Frank']},
                                 [title_test('The End of Eternity', exact=True), authors_test(['Berning, Frank'])]
                             ),

                         ], fail_missing_meta=False)

# if __name__ == '__main__':
#
#     # Run these tests from the directory contatining all files needed for the plugin
#     # that is, __init__.py, plugin-import-name-noosfere.txt and optional worker.py, ui.py
#     # issue in sequence:
#     # calibre-customize -b .
#     # calibre-debug -e __init__.py
#
#     from calibre.ebooks.metadata.sources.test import (test_identify_plugin, title_test, authors_test, series_test)
#     test_identify_plugin(noosfere.name,
#         [
#
#             ( # A book with an ISBN
#                 {'identifiers':{'isbn': '2-221-10703-9'},
#                     'title':"Le Printemps d'Helliconia", 'authors':['B.W. Aldiss']},
#                 [title_test("Le Printemps d'Helliconia",
#                     exact=True), authors_test(['Brian Aldiss']),
#                     series_test('Helliconia', 1.0)]
#
#             ),
#
#         ])
